/** @file
  This file contains functions needed to create PCIE_ROOT_PORT_DEV structure.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Register/PchRegs.h>
#include <Library/DebugLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/PchInfoLib.h>
#include <Library/PchPciBdfLib.h>
#include <Library/PchPcieRpLib.h>
#include <PchPcieRpInfo.h>

#include "PcieRpInitInternal.h"

/**
  Reads a 8-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 8-bit PCI configuration register specified by Address
**/
UINT8
PcieRpCfgRead8 (
  IN PCIE_ROOT_PORT_DEV  *RpDev,
  IN UINT64              Offset
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentRead8 (Private->PciSegmentBase + Offset);
}

/**
  Writes a 8-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT8
PcieRpCfgWrite8 (
  IN PCIE_ROOT_PORT_DEV  *RpDev,
  IN UINT64              Offset,
  IN UINT8               Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentWrite8 (Private->PciSegmentBase + Offset, Value);
}

/**
  Performs a bitwise OR of a 8-bit PCI configuration register with a 8-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT8
PcieRpCfgOr8 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT8                     OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentOr8 (Private->PciSegmentBase + Offset, OrData);
}

/**
  Performs a bitwise AND of a 8-bit PCI configuration register with a 8-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT8
PcieRpCfgAnd8 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT8                     AndData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentAnd8 (Private->PciSegmentBase + Offset, AndData);
}

/**
  Performs a bitwise AND of a 8-bit PCI configuration register with a 8-bit value,
  followed a  bitwise OR with another 8-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT8
PcieRpCfgAndThenOr8 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT8                     AndData,
  IN UINT8                     OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentAndThenOr8 (Private->PciSegmentBase + Offset, AndData, OrData);
}

/**
  Reads a 16-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT16
PcieRpCfgRead16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentRead16 (Private->PciSegmentBase + Offset);
}

/**
  Writes a 16-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT16
PcieRpCfgWrite16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT16                    Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentWrite16 (Private->PciSegmentBase + Offset, Value);
}

/**
  Performs a bitwise OR of a 16-bit PCI configuration register with a 16-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT16
PcieRpCfgOr16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT16                    OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentOr16 (Private->PciSegmentBase + Offset, OrData);
}

/**
  Performs a bitwise AND of a 16-bit PCI configuration register with a 16-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT16
PcieRpCfgAnd16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT16                    AndData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentAnd16 (Private->PciSegmentBase + Offset, AndData);
}

/**
  Performs a bitwise AND of a 16-bit PCI configuration register with a 16-bit value,
  followed a  bitwise OR with another 16-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT16
PcieRpCfgAndThenOr16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT16                    AndData,
  IN UINT16                    OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentAndThenOr16 (Private->PciSegmentBase + Offset, AndData, OrData);
}

/**
  Reads a 32-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT32
PcieRpCfgRead32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentRead32 (Private->PciSegmentBase + Offset);
}

/**
  Writes a 32-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT32
PcieRpCfgWrite32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentWrite32 (Private->PciSegmentBase + Offset, Value);
}

/**
  Performs a bitwise OR of a 32-bit PCI configuration register with a 32-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpCfgOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentOr32 (Private->PciSegmentBase + Offset, OrData);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpCfgAnd32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentAnd32 (Private->PciSegmentBase + Offset, AndData);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value,
  followed a  bitwise OR with another 32-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpCfgAndThenOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData,
  IN UINT32                    OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PciSegmentAndThenOr32 (Private->PciSegmentBase + Offset, AndData, OrData);
}

/**
  Reads a 32-bit PCI configuration register via the SBI message

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT32
PcieRpSbiMsgCfgRead32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset
  )
{
  EFI_STATUS                  Status;
  UINT8                       Response;
  UINT32                      Data32;
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  Status = PchSbiExecutionEx (
             Private->Pid,
             Offset,
             PciConfigRead,
             FALSE,
             0xF,
             0,
             Private->Fid,
             &Data32,
             &Response
             );

  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR,"Sideband Read Failed of Offset 0x%x, Pid = 0x%x, Fid = 0x%x\n", (UINT32)Offset, (UINT32)Private->Pid, (UINT32)Private->Fid));
    Data32 = MAX_UINT32;
  }

  return Data32;
}

/**
  Writes a 32-bit PCI configuration register via the SBI message

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT32
PcieRpSbiMsgCfgWrite32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;
  UINT8                       Response;
  EFI_STATUS                  Status;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  Status = PchSbiExecutionEx (
             Private->Pid,
             Offset,
             PciConfigWrite,
             FALSE,
             0xF,
             0,
             Private->Fid,
             &Value,
             &Response
             );

  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR,"Sideband Write Failed of Offset 0x%x, Pid = 0x%x, Fid = 0x%x\n", (UINT32)Offset, (UINT32)Private->Pid, (UINT32)Private->Fid));
    Value = MAX_UINT32;
  }

  return Value;
}

/**
  Performs a bitwise OR of a 32-bit PCI configuration register with a 32-bit value
  via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpSbiMsgCfgOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    OrData
  )
{
  UINT32                      Data32;

  Data32 = PcieRpSbiMsgCfgRead32 (RpDev, Offset);
  Data32 |= OrData;
  return PcieRpSbiMsgCfgWrite32 (RpDev, Offset, Data32);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value
  via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpSbiMsgCfgAnd32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData
  )
{
  UINT32                      Data32;

  Data32 = PcieRpSbiMsgCfgRead32 (RpDev, Offset);
  Data32 &= AndData;
  return PcieRpSbiMsgCfgWrite32 (RpDev, Offset, Data32);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value,
  followed a  bitwise OR with another 32-bit value via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpSbiMsgCfgAndThenOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData,
  IN UINT32                    OrData
  )
{
  UINT32                      Data32;

  Data32 = PcieRpSbiMsgCfgRead32 (RpDev, Offset);
  Data32 &= AndData;
  Data32 |= OrData;
  return PcieRpSbiMsgCfgWrite32 (RpDev, Offset, Data32);
}

/**
  Reads a 32-bit MEM register via the SBI message

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT32
PcieRpSbiMsgMemRead32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset
  )
{
  EFI_STATUS                  Status;
  UINT8                       Response;
  UINT32                      Data32;
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  Status = PchSbiExecutionEx (
             Private->Pid,
             Offset,
             MemoryRead,
             FALSE,
             0xF,
             0,
             Private->Fid,
             &Data32,
             &Response
             );

  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR,"Sideband Read Failed of Offset 0x%x, Pid = 0x%x, Fid = 0x%x\n", (UINT32)Offset, (UINT32)Private->Pid, (UINT32)Private->Fid));
    Data32 = MAX_UINT32;
  }

  return Data32;
}

/**
  Writes a 32-bit MEM register via the SBI message

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT32
PcieRpSbiMsgMemWrite32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;
  UINT8                       Response;
  EFI_STATUS                  Status;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  Status = PchSbiExecutionEx (
             Private->Pid,
             Offset,
             MemoryWrite,
             FALSE,
             0xF,
             0,
             Private->Fid,
             &Value,
             &Response
             );

  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR,"Sideband Write Failed of Offset 0x%x, Pid = 0x%x, Fid = 0x%x\n", (UINT32)Offset, (UINT32)Private->Pid, (UINT32)Private->Fid));
    Value = MAX_UINT32;
  }

  return Value;
}

/**
  Performs a bitwise OR of a 32-bit MEM register with a 32-bit value
  via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpSbiMsgMemOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    OrData
  )
{
  UINT32                      Data32;

  Data32 = PcieRpSbiMsgMemRead32 (RpDev, Offset);
  Data32 |= OrData;
  return PcieRpSbiMsgMemWrite32 (RpDev, Offset, Data32);
}

/**
  Performs a bitwise AND of a 32-bit MEM register with a 32-bit value
  via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpSbiMsgMemAnd32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData
  )
{
  UINT32                      Data32;

  Data32 = PcieRpSbiMsgMemRead32 (RpDev, Offset);
  Data32 &= AndData;
  return PcieRpSbiMsgMemWrite32 (RpDev, Offset, Data32);
}

/**
  Performs a bitwise AND of a 32-bit MEM register with a 32-bit value,
  followed a  bitwise OR with another 32-bit value via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
PcieRpSbiMsgMemAndThenOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData,
  IN UINT32                    OrData
  )
{
  UINT32                      Data32;

  Data32 = PcieRpSbiMsgMemRead32 (RpDev, Offset);
  Data32 &= AndData;
  Data32 |= OrData;
  return PcieRpSbiMsgMemWrite32 (RpDev, Offset, Data32);
}

/**
  Create PCIE_ROOT_PORT_DEV_PRIVATE object for PCIe root port

  @param[in]  RpIndex         Index of the root port on the PCH
  @param[out] RpDevPrivate    On output pointer to the valid PCIE_ROOT_PORT_DEV_PRIVATE
**/
VOID
PcieGetRpDev (
  IN UINT32                       RpIndex,
  OUT PCIE_ROOT_PORT_DEV_PRIVATE  *RpDevPrivate
  )
{
  PCIE_ROOT_PORT_DEV  *RpDev;

  RpDev = &RpDevPrivate->RpDev;

  RpDev->SipVersion = PcieRpGetSipVer (RpIndex);

  RpDev->Integration = PchPcie;

  RpDev->Sbdf.Segment = DEFAULT_PCI_SEGMENT_NUMBER_PCH;
  RpDev->Sbdf.Bus = DEFAULT_PCI_BUS_NUMBER_PCH;
  RpDev->Sbdf.Device = PchPcieRpDevNumber (RpIndex);
  RpDev->Sbdf.Function = PchPcieRpFuncNumber (RpIndex);

  RpDev->PciCfgAccess.Read8 = PcieRpCfgRead8;
  RpDev->PciCfgAccess.Write8 = PcieRpCfgWrite8;
  RpDev->PciCfgAccess.Or8 = PcieRpCfgOr8;
  RpDev->PciCfgAccess.And8 = PcieRpCfgAnd8;
  RpDev->PciCfgAccess.AndThenOr8 = PcieRpCfgAndThenOr8;

  RpDev->PciCfgAccess.Read16 = PcieRpCfgRead16;
  RpDev->PciCfgAccess.Write16 = PcieRpCfgWrite16;
  RpDev->PciCfgAccess.Or16 = PcieRpCfgOr16;
  RpDev->PciCfgAccess.And16 = PcieRpCfgAnd16;
  RpDev->PciCfgAccess.AndThenOr16 = PcieRpCfgAndThenOr16;

  RpDev->PciCfgAccess.Read32 = PcieRpCfgRead32;
  RpDev->PciCfgAccess.Write32 = PcieRpCfgWrite32;
  RpDev->PciCfgAccess.Or32 = PcieRpCfgOr32;
  RpDev->PciCfgAccess.And32 = PcieRpCfgAnd32;
  RpDev->PciCfgAccess.AndThenOr32 = PcieRpCfgAndThenOr32;

  RpDev->PciSbiMsgCfgAccess.Read32 = PcieRpSbiMsgCfgRead32;
  RpDev->PciSbiMsgCfgAccess.Write32 = PcieRpSbiMsgCfgWrite32;
  RpDev->PciSbiMsgCfgAccess.Or32 = PcieRpSbiMsgCfgOr32;
  RpDev->PciSbiMsgCfgAccess.And32 = PcieRpSbiMsgCfgAnd32;
  RpDev->PciSbiMsgCfgAccess.AndThenOr32 = PcieRpSbiMsgCfgAndThenOr32;

  RpDev->PciSbiMsgMemAccess.Read32 = PcieRpSbiMsgMemRead32;
  RpDev->PciSbiMsgMemAccess.Write32 = PcieRpSbiMsgMemWrite32;
  RpDev->PciSbiMsgMemAccess.Or32 = PcieRpSbiMsgMemOr32;
  RpDev->PciSbiMsgMemAccess.And32 = PcieRpSbiMsgMemAnd32;
  RpDev->PciSbiMsgMemAccess.AndThenOr32 = PcieRpSbiMsgMemAndThenOr32;

  RpDevPrivate->PciSegmentBase = PchPcieRpPciCfgBase (RpIndex);
  RpDevPrivate->Pid = GetRpSbiPid (RpIndex);
  RpDevPrivate->Fid = PcieRpGetSbiFid (RpIndex);
}

/**
  Create PCIE_ROOT_PORT_DEV_PRIVATE object for PCIe controller

  @param[in]  ControllerIndex  Index of the PCIe controller on the PCH
  @param[out] RpDevPrivate     On output pointer to the valid PCIE_ROOT_PORT_DEV_PRIVATE
**/
VOID
PcieGetControllerDev (
  IN  UINT32                      ControllerIndex,
  OUT PCIE_ROOT_PORT_DEV_PRIVATE  *RpDevPrivate
  )
{
  UINT32  RpIndex;

  RpIndex = ControllerIndex * PCH_PCIE_CONTROLLER_PORTS;

  PcieGetRpDev (RpIndex, RpDevPrivate);
}

