/** @file
  This file contains functions that initializes PCI Express Root Ports of PCH in PreMem phase.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/DebugLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/PchInfoLib.h>
#include <Library/PchPcieRpLib.h>
#include <Library/PeiHsioLib.h>
#include <Library/PeiPcieRpLib.h>
#include <PcieRegs.h>
#include <Register/PchPcieRpRegs.h>
#include <Library/PchFiaLib.h>
#include <Library/PsfLib.h>
#include <Library/PchPciBdfLib.h>

#include "PcieRpInitInternal.h"

/**
  This function determines whether root port is configured in non-common clock mode.
  Result is based on the NCC soft-strap setting.

  @param[in] RpBase      Root Port pci segment base address

  @retval TRUE           Port in NCC SSC mode.
  @retval FALSE          Port not in NCC SSC mode.
**/
BOOLEAN
IsPcieNcc (
  IN     UINT64  RpBase
  )
{
  if (PciSegmentRead16 (RpBase + R_PCH_PCIE_CFG_LSTS) & B_PCIE_LSTS_SCC) {
    return FALSE;
  } else {
    return TRUE;
  }
}

/**
  Initialize non-common clock port.
  Ports with NCC configuration need to have their mPHY lanes reconfigured by BIOS before
  endpoint detection can start. Reconfiguration is instant, but detection may take up to
  100ms. In order to save as much time as possible, this reconfiguration should be executed
  in PEI pre-mem, so that detection happens in parallel with memory init
  @param[in] RpIndex    Root Port index
**/
VOID
PcieInitNccPort (
  IN UINT32   RpIndex
  )
{
  UINT64     RpBase;
  UINT32     RpLaneIndex;
  UINT32     MaxLinkWidth;
  HSIO_LANE  HsioLane;
  UINT8      FiaLane;

  DEBUG ((DEBUG_INFO, "PcieInitNccPort(%d)\n", RpIndex+1));

  RpBase = PchPcieRpPciCfgBase (RpIndex);
  MaxLinkWidth = GetMaxLinkWidth (RpBase);
  for (RpLaneIndex = 0; RpLaneIndex < MaxLinkWidth; ++RpLaneIndex) {
    if (PchFiaGetPcieRootPortLaneNum (RpIndex, RpLaneIndex, &FiaLane)) {
      HsioGetLane (FiaLane, &HsioLane);
      HsioPcieNccLaneInit (&HsioLane);
    }
  }
  PciSegmentAnd32 (RpBase + R_PCH_PCIE_CFG_PCIEALC, (UINT32) ~B_PCH_PCIE_CFG_PCIEALC_BLKDQDA);
}

/**
  Hide rootports disabled by policy. This needs to be done in premem,
  because graphics init from SystemAgent code depends on those ports
  being already hidden

  @param[in] PcieRpPreMemConfig   Platform policy
**/
VOID
PcieRpEarlyDisabling (
  IN PCH_PCIE_RP_PREMEM_CONFIG *PcieRpPreMemConfig
  )
{
  UINT32 RpIndex;
  for (RpIndex = 0; RpIndex < GetPchMaxPciePortNum (); RpIndex++) {
    if ((PcieRpPreMemConfig->RpEnabledMask & (UINT32) (1 << RpIndex)) == 0) {
      PsfDisablePcieRootPort (RpIndex);
    }
  }
}

/**
  Initializes ports with NonCommonClock configuration. Such ports can't detect endpoints
  before NCC init ends. To prevent boot delays, NCC handling should happen in pre-mem
  rather than just before endpoint detection in post-mem
**/
VOID
PcieRpEarlyNccHandling (
  VOID
  )
{
  UINT32 RpIndex;

  for (RpIndex = 0; RpIndex < GetPchMaxPciePortNum (); RpIndex++) {
    if (IsPcieNcc (PchPcieRpPciCfgBase (RpIndex))) {
      PcieInitNccPort (RpIndex);
    }
  }
}
