/** @file
  This file contains internal PSF routines for PCH PSF VER4 lib usage

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Uefi/UefiBaseType.h>
#include <Library/IoLib.h>
#include <Library/DebugLib.h>
#include <Library/PchPcrLib.h>
#include <Library/PchInfoLib.h>
#include <Library/SataLib.h>
#include <Library/BaseMemoryLib.h>
#include <Register/PchRegsPsf.h>
#include <Register/PsfRegsVer4.h>
#include "PsfLibInternal.h"
#include <PchPcieRpInfo.h>

GLOBAL_REMOVE_IF_UNREFERENCED PSF_PORT mPchSSerialIoI2cPsfPort[] =
{
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_I2C0_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_I2C1_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_I2C2_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_I2C3_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_I2C4_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_I2C5_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_I2C6_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_I2C7_REG_BASE}
};

/**
  Return PSF_PORT for SerialIO I2C device

  @param[in] I2cNum  Serial IO I2C device (I2C0, I2C1, ....)

  @retval  PsfPort   PSF PORT structure for SerialIO I2C device
**/
PSF_PORT
PsfSerialIoI2cPort (
  IN UINT32  I2cNum
  )
{
  if (I2cNum < ARRAY_SIZE (mPchSSerialIoI2cPsfPort)) {
    return mPchSSerialIoI2cPsfPort[I2cNum];
  }

  ASSERT(FALSE);
  return PSF_PORT_NULL;
}

GLOBAL_REMOVE_IF_UNREFERENCED PSF_PORT mPchSSerialIoSpiPsfPort[] =
{
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_SPI0_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_SPI1_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_SPI2_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_SPI3_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_SPI4_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_SPI5_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_SPI6_REG_BASE}
};

/**
  Return PSF_PORT for SerialIO SPI device

  @param[in] I2cNum  Serial IO SPI device (SPI0, SPI1, ....)

  @retval  PsfPort   PSF PORT structure for SerialIO SPI device
**/
PSF_PORT
PsfSerialIoSpiPort (
  IN UINT32  SpiNum
  )
{
  if (SpiNum < ARRAY_SIZE (mPchSSerialIoSpiPsfPort)) {
    return mPchSSerialIoSpiPsfPort[SpiNum];
  }

  ASSERT(FALSE);
  return PSF_PORT_NULL;
}

GLOBAL_REMOVE_IF_UNREFERENCED PSF_PORT mPchSSerialIoUartPsfPort[] =
{
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_UART0_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_UART1_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_UART2_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_UART3_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_UART4_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_UART5_REG_BASE},
  {PID_PSF3, R_VER4_PCH_S_PSF3_PCR_T0_SHDW_UART6_REG_BASE}
};

/**
  Return PSF_PORT for SerialIO UART device

  @param[in] UartNum  Serial IO UART device (UART0, UART1, ....)

  @retval  PsfPort   PSF PORT structure for SerialIO UART device
**/
PSF_PORT
PsfSerialIoUartPort (
  IN UINT32  UartNum
  )
{
  if (UartNum < ARRAY_SIZE (mPchSSerialIoUartPsfPort)) {
    return mPchSSerialIoUartPsfPort[UartNum];
  }

  ASSERT(FALSE);
  return PSF_PORT_NULL;
}

/**
  Return PSF_PORT for TraceHub device

  @retval    PsfPort         PSF PORT structure for TraceHub device
**/
PSF_PORT
PsfTraceHubPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF1;
  PsfPort.RegBase = R_VER4_PCH_S_PSF1_PCR_T0_SHDW_TRACEHUB_REG_BASE;

  return PsfPort;
}

/**
  Return PSF_PORT for TraceHub ACPI device

  @retval    PsfPort         PSF PORT structure for TraceHub ACPI device
**/
PSF_PORT
PsfTraceHubAcpiDevPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF1;
  PsfPort.RegBase = R_VER4_PCH_S_PSF1_PCR_T0_SHDW_NPK_PHANTOM_REG_BASE;

  return PsfPort;
}

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchSHeciPsfRegBase[] =
{
  R_VER4_PCH_S_PSF1_PCR_T0_SHDW_HECI1_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T0_SHDW_HECI2_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T0_SHDW_HECI3_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T0_SHDW_HECI4_REG_BASE
};

/**
  Return PSF_PORT for HECI device

  @param[in] HeciDevice      HECIx Device (HECI1-4)

  @retval    PsfPort         PSF PORT structure for HECI device
**/
PSF_PORT
PsfHeciPort (
  IN UINT8      HeciDevice
  )
{
  PSF_PORT PsfPort;

  if ((HeciDevice < 1) || (HeciDevice > 4)) {
    ASSERT (FALSE);
    return PSF_PORT_NULL;
  }

  PsfPort.PsfPid = PID_PSF1;
  PsfPort.RegBase = mPchSHeciPsfRegBase[HeciDevice - 1];

  return PsfPort;
}

/**
  Return PSF_PORT for SOL device

  @retval    PsfPort         PSF PORT structure for SOL device
**/
PSF_PORT
PsfSolPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF1;
  PsfPort.RegBase = R_VER4_PCH_S_PSF1_PCR_T0_SHDW_KT_REG_BASE;

  return PsfPort;
}

/**
  Disable IDER device at PSF level
**/
VOID
PsfDisableIderDevice (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF1;
  PsfPort.RegBase = R_VER4_PCH_S_PSF1_PCR_T0_SHDW_IDER_REG_BASE;
  PsfDisableDevice (PsfPort);
}

/**
  Disable HDAudio device at PSF level
**/
VOID
PsfDisableHdaDevice (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF3;
  PsfPort.RegBase = R_VER4_PCH_S_PSF3_PCR_T0_SHDW_AUD_REG_BASE;
  PsfDisableDevice (PsfPort);
}

/**
  Disable DSP Bar (HdAudio BAR4 and BAR5) at PSF level.
**/
VOID
PsfDisableDspBar (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF3;
  PsfPort.RegBase = R_VER4_PCH_S_PSF3_PCR_T0_SHDW_AUD_REG_BASE;

  PsfDisableDeviceBar (PsfPort, (R_PCH_PSFX_PCR_T0_SHDW_BAR5 | R_PCH_PSFX_PCR_T0_SHDW_BAR4));
}

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchSThcPsfRegBase[] =
{
  R_VER4_PCH_S_PSF3_PCR_T0_SHDW_THC0_REG_BASE,
  R_VER4_PCH_S_PSF3_PCR_T0_SHDW_THC1_REG_BASE
};

/**
  Disable THC device at PSF level

  @param[in]  ThcNumber                Touch Host Controller Number THC0 or THC1
**/
VOID
PsfDisableThcDevice (
  IN  UINT32        ThcNumber
  )
{
  PSF_PORT PsfPort;

  if ((IsPchS () && ThcNumber >= ARRAY_SIZE (mPchSThcPsfRegBase))) {
    DEBUG ((DEBUG_ERROR, "Wrong THC number!\n"));
    ASSERT (FALSE);
    return;
  }

  PsfPort.PsfPid = PID_PSF3;
  PsfPort.RegBase = mPchSThcPsfRegBase[ThcNumber];
  PsfDisableDevice (PsfPort);
}

/**
  Disable xDCI device at PSF level
**/
VOID
PsfDisableXdciDevice (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF2;
  PsfPort.RegBase = R_VER4_PCH_S_PSF2_PCR_T0_SHDW_XDCI_REG_BASE;
  PsfDisableDevice (PsfPort);
}

/**
  Disable xHCI device at PSF level
**/
VOID
PsfDisableXhciDevice (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF1;
  PsfPort.RegBase = R_VER4_PCH_S_PSF1_PCR_T0_SHDW_XHCI_REG_BASE;
  PsfDisableDevice (PsfPort);
}

/**
  Disable xHCI VTIO Phantom device at PSF level
**/
VOID
PsfDisableXhciVtioDevice (
  VOID
  )
{
  // Not applicable for ADL platform
}

/**
  Disable SATA device at PSF level

  @param[in]  SataCtrlIndex     SATA controller index
**/
VOID
PsfDisableSataDevice (
  IN UINT32     SataCtrlIndex
  )
{
  PSF_PORT PsfPort;

  ASSERT (SataCtrlIndex < 1);

  PsfPort.PsfPid = PID_PSF1;
  PsfPort.RegBase = R_VER4_PCH_S_PSF1_PCR_T0_SHDW_SATA_REG_BASE;
  PsfDisableDevice (PsfPort);
}


/**
  Program PSF grant counts for SATA
  Call this before SATA ports are accessed for enumeration
**/
VOID
PsfConfigureSataGrantCounts (
  VOID
  )
{
  // No configuration needed for ADL-PCH
  return;
}


/**
  Return PSF_PORT for ISH device

  @retval    PsfPort         PSF PORT structure for ISH device
**/
PSF_PORT
PsfIshPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF3;
  PsfPort.RegBase = R_VER4_PCH_S_PSF3_PCR_T0_SHDW_ISH_REG_BASE;

  return PsfPort;
}

/**
  Disable FPAK device at PSF level
**/
VOID
PsfDisableFpakDevice (
  VOID
  )
{
  if (!IsPchS ()) {
    PsfDisableDevice (PsfFpakPort ());
  }
}

/**
  Return PSF_PORT for FPAK device

  @retval    PsfPort         PSF PORT structure for FPAK device
**/
PSF_PORT
PsfFpakPort (
  VOID
  )
{
  // Not applicable for ADL platform
  ASSERT (FALSE);
  return PSF_PORT_NULL;
}

/**
  Disable GbE device at PSF level
**/
VOID
PsfDisableGbeDevice (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF3;
  PsfPort.RegBase = R_VER4_PCH_S_PSF3_PCR_T0_SHDW_GBE_REG_BASE;
  PsfDisableDevice (PsfPort);
}

/**
  Disable SMBUS device at PSF level
**/
VOID
PsfDisableSmbusDevice (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF3;
  PsfPort.RegBase = R_VER4_PCH_S_PSF3_PCR_T0_SHDW_SMBUS_REG_BASE;
  PsfDisableDevice (PsfPort);
}

/**
  Return PSF_PORT for SCS eMMC device

  @retval    PsfPort      PSF PORT structure for SCS eMMC device
**/
PSF_PORT
PsfScsEmmcPort (
  VOID
  )
{
  // Not applicable for ADL platform
  ASSERT (FALSE);
  return PSF_PORT_NULL;
}

/**
  Return PSF_PORT for SCS SD Card device

  @retval    PsfPort      PSF PORT structure for SCS SD Card device
**/
PSF_PORT
PsfScsSdCardPort (
  VOID
  )
{
// Not applicable for ADL platform
  ASSERT (FALSE);
  return PSF_PORT_NULL;
}

/**
  Return PSF_PORT for SCS UFS device

  @param[in] UfsNum       UFS Device

  @retval    PsfPort      PSF PORT structure for SCS UFS device
**/
PSF_PORT
PsfScsUfsPort (
  IN UINT32  UfsNum
  )
{
// Not applicable for ADL platform
  ASSERT (FALSE);
  return PSF_PORT_NULL;
}

//
// RS0 PSF1_T1_SHDW_PCIExx_REG_BASE registers for PCH-S
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchSPsf1PcieRegBase[] =
{
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE01_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE02_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE03_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE04_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE05_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE06_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE07_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE08_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE09_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE10_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE11_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE12_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE13_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE14_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE15_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE16_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE17_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE18_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE19_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE20_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE21_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE22_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE23_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE24_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE25_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE26_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE27_RS0_REG_BASE,
  R_VER4_PCH_S_PSF1_PCR_T1_SHDW_PCIE28_RS0_REG_BASE
};

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchSPsfPcieSecondLevelPort[] =
{
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE01_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE02_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE03_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE04_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE05_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE06_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE07_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE08_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE09_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE10_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE11_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE12_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE13_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE14_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE15_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE16_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE17_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE18_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE19_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE20_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE21_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE22_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE23_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE24_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE25_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE26_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE27_REG_BASE,
  R_VER4_PCH_S_PSF7_PCR_T1_SHDW_PCIE28_REG_BASE
};

/**
  Return PSF_PORT at root PSF level to which PCIe Root Port device is connected

  @param[in] RpIndex        PCIe Root Port Index (0 based)

  @retval    PsfPort        PSF PORT structure for PCIe

**/
PSF_PORT
PsfRootPciePort (
  IN UINT32  RpIndex
  )
{
  PSF_PORT PsfPort;

  if (RpIndex < ARRAY_SIZE (mPchSPsf1PcieRegBase)) {
    PsfPort.PsfPid = PID_PSF1;
    PsfPort.RegBase = mPchSPsf1PcieRegBase[RpIndex];
    return PsfPort;
  }

  ASSERT (FALSE);
  return PSF_PORT_NULL;
}

/**
  Return second level PSF_PORT to which PCIE Root Port device is connected (directly)

  @param[in] RpIndex        PCIe Root Port Index (0 based)

  @retval    PsfPort        PSF PORT structure for PCIe
**/
PSF_PORT
PsfPcieSecondLevelPort (
  IN UINT32  RpIndex
  )
{
  PSF_PORT PsfPort;

  if (IsPchS ()) {
    if (RpIndex < ARRAY_SIZE (mPchSPsfPcieSecondLevelPort)) {
      PsfPort.PsfPid = PID_PSF7;
      PsfPort.RegBase = mPchSPsfPcieSecondLevelPort[RpIndex];
      return PsfPort;
    }
    ASSERT (FALSE);
  }
  return PSF_PORT_NULL;
}

/**
  Return RS3 PSF_PORT at root PSF level to which PCIe Root Port device is connected

  @param[in] RpIndex        PCIe Root Port Index (0 based)

  @retval    PsfPort        PSF PORT structure for PCIe
**/
PSF_PORT
PsfRootRs3PciePort (
  IN UINT32  RpIndex
  )
{
// Not applicable for ADL platform
  return PSF_PORT_NULL;
}

/**
  Grant count regs data for PSF that is directly connected to PCIe Root Ports

  @param[in]  Controller     PCIe Root Port Controller index (0 based)
  @param[out] GrantCountReg  Structure with PSF Grant Count register data
**/
VOID
PsfPcieGrantCountBaseReg (
  IN  UINT8                Controller,
  OUT PSF_GRANT_COUNT_REG  *GrantCountReg
  )
{
  if (Controller < 7) {
    GrantCountReg->PsfPid = PID_PSF7;
    GrantCountReg->DevGntCnt0Base = R_VER4_PCH_S_PSF7_PCR_DEV_GNTCNT_RELOAD_DGCR0;
    GrantCountReg->TargetGntCntPg1Tgt0Base = R_VER4_PCH_S_PSF7_PCR_TARGET_GNTCNT_RELOAD_PG1_TGT0;
  }
}


GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8 mDgcrH [][PCH_PCIE_CONTROLLER_PORTS] = {
    { 28, 27, 26, 25 }, //SPA
    { 24, 23, 22, 21 }, //SPB
    { 20, 19, 18, 17 }, //SPC
    { 16, 15, 14, 13 }, //SPD
    { 12, 11, 10,  9 }, //SPE
    {  8,  7,  6,  5 }, //SPF
    {  4,  3,  2,  1 }  //SPG
  };

GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8 mPgTgtH [][PCH_PCIE_CONTROLLER_PORTS] = {
    {  0,  1,  2,  3 }, //SPA
    {  4,  5,  6,  7 }, //SPB
    {  8,  9, 10, 11 }, //SPC
    { 12, 13, 14, 15 }, //SPD
    { 16, 17, 18, 19 }, //SPE
    { 20, 21, 22, 23 }, //SPF
    { 24, 25, 26, 27 }  //SPG
  };

/**
  Get Grant Count number (Device Grant Count and Target Grant Count)
  for PSF that is directly connected to PCIe Root Ports

  @param[in]  Controller    PCIe Root Port Controller index
  @param[in]  Channel       PCIe Root Port Channel index
  @param[out] DgcrNo        Device Grant Count number
  @param[out] PgTgtNo       Target Grant Count number
**/
VOID
PsfPcieGrantCountNumber (
  IN  UINT8 Controller,
  IN  UINT8 Channel,
  OUT UINT8 *DgcrNo,
  OUT UINT8 *PgTgtNo
  )
{
  if ((Channel < ARRAY_SIZE (mDgcrH[0])) && (Controller < ARRAY_SIZE (mDgcrH))) {
    *DgcrNo = mDgcrH[Controller][Channel];
    *PgTgtNo = mPgTgtH[Controller][Channel];
    return;
  }

  ASSERT (FALSE);
}

/**
  Grant count regs data for a given PSF-to-PSF port.

  @param[in] PsfTopoPort         PSF-to-PSF port

  @param[out] GrantCountReg      Structure with PSF Grant Count register data
**/
VOID
PsfSegmentGrantCountBaseReg (
  IN PSF_TOPO_PORT         PsfTopoPort,
  OUT PSF_GRANT_COUNT_REG  *GrantCountReg
  )
{
  if (IsPchS () && PsfTopoPort.PsfId == 1) {
    GrantCountReg->PsfPid = PID_PSF1;
    GrantCountReg->DevGntCnt0Base = R_VER4_PCH_S_PSF1_PCR_DEV_GNTCNT_RELOAD_DGCR0;
    GrantCountReg->TargetGntCntPg1Tgt0Base = R_VER4_PCH_S_PSF1_PCR_TARGET_GNTCNT_RELOAD_PG1_TGT0;
  } else {
    ASSERT (FALSE);
  }
}

/**
  Grant Count number (Device Grant Count and Target Grant Count) for a given PSF-to-PSF port.

  @param[in] PsfTopoPort         PSF-to-PSF port
  @param[out] DgcrNo             Device Grant Count number
  @param[out] PgTgtNo            Target Grant Count number
**/
VOID
PsfSegmentGrantCountNumber (
  IN PSF_TOPO_PORT PsfTopoPort,
  OUT UINT8        *DgcrNo,
  OUT UINT8        *PgTgtNo
  )
{
  *DgcrNo = 0;
  *PgTgtNo = 0;

  if (!(IsPchS () && (PsfTopoPort.PsfId == 1))) {
    ASSERT (FALSE);
    return;
  }

  switch (PsfTopoPort.PortId) {
    case 0: //PSF7
      *DgcrNo = 9;
      *PgTgtNo = 0;
      break;
    default:
      ASSERT (FALSE);
      break;
  }
}

//
// PCH-S PSF topology for PCIe
//
GLOBAL_REMOVE_IF_UNREFERENCED CONST PSF_TOPOLOGY mPchSPsf7PcieTreeTopo[] = {
  //{PsfId, Port}, PortType, Child, PcieCtrlIndex
  {{7, 0}, PsfPcieCtrlPort, NULL, {0}}, // SPA
  {{7, 1}, PsfPcieCtrlPort, NULL, {1}}, // SPB
  {{7, 2}, PsfPcieCtrlPort, NULL, {2}}, // SPC
  {{7, 3}, PsfPcieCtrlPort, NULL, {3}}, // SPD
  {{7, 4}, PsfPcieCtrlPort, NULL, {4}}, // SPE
  {{7, 5}, PsfPcieCtrlPort, NULL, {5}}, // SPF
  {{7, 6}, PsfPcieCtrlPort, NULL, {6}}, // SPG
  PSF_TOPOLOGY_END
};

GLOBAL_REMOVE_IF_UNREFERENCED CONST PSF_TOPOLOGY mPchSPsf1PcieTreeTopo[] = {
  //{PsfId, Port}, PortType, Child;
  {{1, 0}, PsfToPsfPort, mPchSPsf7PcieTreeTopo},
  PSF_TOPOLOGY_END
};

GLOBAL_REMOVE_IF_UNREFERENCED CONST PSF_TOPOLOGY mPchSPsfPcieTreeTopo[] = {
  //{PsfId, Port}, PortType, Child;
  {{0, 0}, PsfToPsfPort, mPchSPsf1PcieTreeTopo}, // Abstract root of PSF topology
  PSF_TOPOLOGY_END
};

/**
  Get PSF Pcie Tree topology

  @param[in] PsfTopology          PSF Port from PSF PCIe tree topology

  @retval PsfTopology             PSF PCIe tree topology
**/
CONST PSF_TOPOLOGY*
PsfGetRootPciePsfTopology (
  VOID
  )
{
  if (IsPchS ()) {
    return mPchSPsfPcieTreeTopo;
  }
  //@todo add PCH-LP support
  ASSERT (FALSE);
  return mPchSPsfPcieTreeTopo;
}

/**
  Get EOI register data for given PSF ID

  @param[in]  PsfId           PSF ID (1 - PSF1, 2 - PSF2, ...)
  @param[out] EoiTargetBase   EOI Target register
  @param[out] EoiControlBase  EOI Control register

  @retval MaxTargets          Number of supported targets

**/
UINT8
PsfEoiRegData (
  UINT32        PsfId,
  UINT16        *EoiTargetBase,
  UINT16        *EoiControlBase
  )
{
  UINT8  MaxTargets;

  MaxTargets = 0;
  *EoiTargetBase = 0;
  *EoiControlBase = 0;

  switch (PsfId) {
    case 1:
      *EoiTargetBase = R_VER4_PCH_S_PSF1_PCR_MC_AGENT_MCAST0_TGT0_EOI;
      *EoiControlBase = R_VER4_PCH_S_PSF1_PCR_MC_CONTROL_MCAST0_EOI;
      MaxTargets = 3;
      break;

    case 3:
      *EoiTargetBase = R_VER4_PCH_S_PSF3_PCR_MC_AGENT_MCAST0_TGT0_EOI;
      *EoiControlBase = R_VER4_PCH_S_PSF3_PCR_MC_CONTROL_MCAST0_EOI;
      MaxTargets = 1;
      break;

    case 7:
      *EoiTargetBase = R_VER4_PCH_S_PSF7_PCR_MC_AGENT_MCAST0_TGT0_EOI;
      *EoiControlBase = R_VER4_PCH_S_PSF7_PCR_MC_CONTROL_MCAST0_EOI;
      MaxTargets = 28;
      break;

    default:
      break;

  }
  return MaxTargets;
}

/**
  Get MCTP register data for given PSF ID

  @param[in]  PsfId            PSF ID (1 - PSF1, 2 - PSF2, ...)
  @param[out] MctpTargetBase   MCTP Target register
  @param[out] MctpControlBase  MCTP Control register

  @retval MaxTargets           Number of supported targets

**/
UINT8
PsfMctpRegData (
  UINT32        PsfId,
  UINT16        *MctpTargetBase,
  UINT16        *MctpControlBase
  )
{
  UINT8  MaxTargets;

  MaxTargets = 0;
  *MctpTargetBase = 0;
  *MctpControlBase = 0;

  switch (PsfId) {
    case 1:
      *MctpTargetBase = R_VER4_PCH_S_PSF1_PCR_MC_AGENT_MCAST1_RS0_TGT0_MCTP1;
      *MctpControlBase = R_VER4_PCH_S_PSF1_PCR_MC_CONTROL_MCAST1_RS0_MCTP1;
      MaxTargets = 3;
      break;

    case 7:
      *MctpTargetBase = R_VER4_PCH_S_PSF7_PCR_MC_AGENT_MCAST1_RS0_TGT0_MCTP1;
      *MctpControlBase = R_VER4_PCH_S_PSF7_PCR_MC_CONTROL_MCAST1_RS0_MCTP1;
      MaxTargets = 28;
      break;
  }
  return MaxTargets;
}

/**
  P2SB PSF port Destination ID (psf_id:port_group_id:port_id:channel_id)

  @retval P2SB Destination ID
**/
PSF_PORT_DEST_ID
PsfP2sbDestinationId (
  VOID
  )
{
  PSF_PORT_DEST_ID P2sbTarget;

  P2sbTarget.RegVal = V_VER4_PCH_S_PSFX_PCR_PSF_MC_AGENT_MCAST_TGT_P2SB;

  return P2sbTarget;
}

/**
  DMI PSF port Destination ID (psf_id:port_group_id:port_id:channel_id)

  @retval DMI Destination ID
**/
PSF_PORT_DEST_ID
PsfDmiDestinationId (
  VOID
  )
{
  PSF_PORT_DEST_ID DmiTarget;

  DmiTarget.RegVal = V_VER4_PCH_PSFX_PSF_PCR_MC_AGENT_MCAST_TGT_DMI;

  return DmiTarget;
}

GLOBAL_REMOVE_IF_UNREFERENCED PSF_PORT_DEST_ID PchSRpDestId[] =
{
  {0x78000}, {0x78001}, {0x78002}, {0x78003}, // SPA: PSF7, PortID = 0
  {0x78100}, {0x78101}, {0x78102}, {0x78103}, // SPB: PSF7, PortID = 1
  {0x78200}, {0x78201}, {0x78202}, {0x78203}, // SPC: PSF7, PortID = 2
  {0x78300}, {0x78301}, {0x78302}, {0x78303}, // SPD: PSF7, PortID = 3
  {0x78400}, {0x78401}, {0x78402}, {0x78403}, // SPE: PSF7, PortID = 4
  {0x78500}, {0x78501}, {0x78502}, {0x78503}, // SPF: PSF7, PortID = 5
  {0x78600}, {0x78601}, {0x78602}, {0x78603}  // SPG: PSF7, PortID = 6
};

/**
  PCIe PSF port destination ID (psf_id:port_group_id:port_id:channel_id)

  @param[in] RpIndex        PCIe Root Port Index (0 based)

  @retval Destination ID
**/
PSF_PORT_DEST_ID
PsfPcieDestinationId (
  IN UINT32  RpIndex
  )
{
  if (RpIndex < ARRAY_SIZE (PchSRpDestId)) {
    return PchSRpDestId[RpIndex];
  }

  ASSERT (FALSE);
  return (PSF_PORT_DEST_ID){0};
}

//
// RS0 PSF1_T1_AGENT_FUNCTION_CONFIG_SPx_Dxx_Fx registers for PCH-S
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchSPsf1PcieFuncCfgBase[] =
{
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F0,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F1,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F2,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F3,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F4,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F5,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F6,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F7,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F0,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F1,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F2,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F3,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F4,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F5,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F6,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F7,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS0_D27_F0,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS0_D27_F1,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS0_D27_F2,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS0_D27_F3,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS0_D27_F4,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS0_D27_F5,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS0_D27_F6,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS0_D27_F7,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPG_RS0_D26_F0,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPG_RS0_D26_F1,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPG_RS0_D26_F2,
  R_VER4_PCH_S_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPG_RS0_D26_F3
};

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchSPsf7PcieFuncCfgSecondLevelPort[] =
{
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_D28_F0,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_D28_F1,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_D28_F2,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_D28_F3,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_D28_F4,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_D28_F5,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_D28_F6,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_D28_F7,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_D29_F0,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_D29_F1,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_D29_F2,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_D29_F3,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_D29_F4,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_D29_F5,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_D29_F6,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_D29_F7,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_D27_F0,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_D27_F1,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_D27_F2,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_D27_F3,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_D27_F4,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_D27_F5,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_D27_F6,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_D27_F7,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPG_D26_F0,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPG_D26_F1,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPG_D26_F2,
  R_VER4_PCH_S_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPG_D26_F3
};

/**
  Return the PSF (Root level) Function Config PSF_PORT for PCIe Root Port

  @param[in] RpIndex        PCIe Root Port Index (0 based)

  @retval    PsfPort        PSF PORT structure for PCIe Function Config
**/
PSF_PORT
PsfRootPcieFunctionConfigPort (
  IN UINT32  RpIndex
  )
{
  PSF_PORT  PsfPort;

  PsfPort.PsfPid = PID_PSF1;

  if (RpIndex < ARRAY_SIZE (mPchSPsf1PcieFuncCfgBase)) {
    PsfPort.RegBase = mPchSPsf1PcieFuncCfgBase[RpIndex];
    return PsfPort;
  }
  ASSERT (FALSE);
  return PSF_PORT_NULL;
}

/**
  Return the PSF (Root level) RS3 Function Config PSF_PORT for PCIe Root Port

  @param[in] RpIndex        PCIe Root Port Index (0 based)

  @retval    PsfPort        PSF PORT structure for PCIe Function Config
**/
PSF_PORT
PsfRootRs3PcieFunctionConfigPort (
  IN UINT32  RpIndex
  )
{
// Not applicable for ADL platform
  ASSERT (FALSE);
  return PSF_PORT_NULL;
}

/**
  Return the PSF Function Config Second Level PSF_PORT for PCIe Root Port

  @param[in] RpIndex        PCIe Root Port Index (0 based)

  @retval    PsfPort        PSF PORT structure for PCIe Function Config
**/
PSF_PORT
PsfPcieFunctionConfigSecondLevelPort (
  IN UINT32  RpIndex
  )
{
  PSF_PORT  PsfPort;

  PsfPort.PsfPid = PID_PSF7;

  if (IsPchS ()) {
    if (RpIndex < ARRAY_SIZE (mPchSPsf7PcieFuncCfgSecondLevelPort)) {
      PsfPort.RegBase = mPchSPsf7PcieFuncCfgSecondLevelPort[RpIndex];
      return PsfPort;
    }
    ASSERT (FALSE);
  }
  return PSF_PORT_NULL;
}

/**
  Return PSF_PORT for CNVi device

  @retval    PsfPort         PSF PORT structure for CNVi device
**/
PSF_PORT
PsfCnviPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF3;
  PsfPort.RegBase = R_VER4_PCH_S_PSF3_PCR_T0_SHDW_CNVI_REG_BASE;

  return PsfPort;
}

/**
  Return PSF_PORT for PMC device

  @retval    PsfPort         PSF PORT structure for PMC device
**/
PSF_PORT
PsfPmcPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = PID_PSF3;
  PsfPort.RegBase = R_VER4_PCH_S_PSF3_PCR_T0_SHDW_PMC_REG_BASE;

  return PsfPort;
}

GLOBAL_REMOVE_IF_UNREFERENCED PSF_SEGMENT mPchSPsfTable[] =
{
  {1, PID_PSF1},
  {2, PID_PSF2},
  {3, PID_PSF3},
  {4, PID_PSF4},
  {6, PID_PSF6},
  {7, PID_PSF7}
};

/**
  Get list of supported PSF segments.

  @param[out] PsfTable        Array of supported PSF segments
  @param[out] PsfTableLength  Length of PsfTable
**/
VOID
PsfSegments (
  OUT PSF_SEGMENT  **PsfTable,
  OUT UINT32       *PsfTableLength
  )
{
  *PsfTable = mPchSPsfTable;
  *PsfTableLength = ARRAY_SIZE (mPchSPsfTable);
}

/**
  Disable PSF upstream completion tracking for HDAudio on PCH-LP
**/
VOID
PsfDisableUpstreamCompletionTrackingForHda (
  VOID
  )
{
  // Not applicable to ADL platform
  return;
}

/**
  Disable D3:F0 device at PSF level for PCH-LP
**/
VOID
PsfDisableD3F0 (
  VOID
  )
{
  // Not applicable to ADL platform
  // @todo: revise for ALPHA release
  return;
}

/**
  Check if MCTP is supported

  @retval TRUE              MCTP is supported
          FALSE             MCTP is not supported
**/
BOOLEAN
PsfIsMctpSupported (
  VOID
  )
{
  return TRUE;
}
/**
  This function returns Psf Port Relaxed Ordering Configs

  @param[out] PsfPortRelaxedOrderingConfigRegs                          PCH Series specific table
  @param[out] PsfPortRelaxedOrderingConfigRegsTableSize                 PCH Series specific table size
  @param[out] PsfPortRelaxedOrderingConfigRegsPchTypeSpecific           PCH type specific table
  @param[out] PsfPortRelaxedOrderingConfigRegsPchTypeSpecificTableSize  PCH type specific table size
**/
VOID
GetPsfPortRelaxedOrderingTables (
  PSF_PORT_RELAXED_ORDERING_CONFIG_REG** PsfPortRelaxedOrderingConfigRegs,
  UINT32*                               PsfPortRelaxedOrderingConfigRegsTableSize,
  PSF_PORT_RELAXED_ORDERING_CONFIG_REG** PsfPortRelaxedOrderingConfigRegsPchTypeSpecific,
  UINT32*                               PsfPortRelaxedOrderingConfigRegsPchTypeSpecificTableSize
  )
{
  *PsfPortRelaxedOrderingConfigRegs                         = NULL;
  *PsfPortRelaxedOrderingConfigRegsTableSize                = 0;
  *PsfPortRelaxedOrderingConfigRegsPchTypeSpecific          = NULL;
  *PsfPortRelaxedOrderingConfigRegsPchTypeSpecificTableSize = 0;
}
