/** @file
  The PEI DPIN Init Library Implements After Memory PEIM

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/PeiServicesLib.h>
#include <Library/DebugLib.h>
#include <Library/PcdLib.h>
#include <Uefi/UefiBaseType.h>
#include <Register/IomRegs.h>
#include <Library/ConfigBlockLib.h>
#include <Ppi/SiPolicy.h>
#include <Library/CpuRegbarAccessLib.h>
#include <CpuSbInfo.h>
#include <DpInPreMemConfig.h>
#include <Library/CpuPlatformLib.h>
#include <Library/PeiTcssInitFruLib.h>
#include <Library/PeiDpInInitLib.h>

/**
  This function return CPU specific Private Data to Dp-In PreMem Pei Config Structure.

  @retval 0                       The Dp-In is not supported by CPU.
  @retval Others                  Return the Maximum number of ports supported by Silicon.
**/
UINT8
EFIAPI
GetDpInPrivateCpuData (
  VOID
  )
{
  UINT8                   MaxDpInExtPortCpuSupport;
  DEBUG ((DEBUG_INFO, "Initializing CPU specific PreMem Pei Config in Pre-Mem\n"));

  MaxDpInExtPortCpuSupport = 0; // Assign default value.

  //
  // Initialize Dp-In PreMem Pei Config Structure on CPU Family.
  //
  MaxDpInExtPortCpuSupport = GetDpInCpuPortCount ();

  if ((MaxDpInExtPortCpuSupport > PcdGet8 (PcdMaxDpInExtPortSupported)) \
      || (MaxDpInExtPortCpuSupport == 0)) {
    // If Invalid MaxDpInExtPortCpuSupport is passed, or CPU doesn't Support DP-In.
    DEBUG ((DEBUG_INFO, "GetDpInPrivateCpuData - Dp-In not Supported by Silicon. Max DP-In Port Support = 0x%X\n", MaxDpInExtPortCpuSupport));
    MaxDpInExtPortCpuSupport = 0; // Maximun External Gfx DpIn Port supported by CPU
  } else {
    // If Valid MaxDpInExtPortCpuSupport is passed.
    DEBUG ((DEBUG_INFO, "GetDpInPrivateCpuData - Dp-In Supported by Silicon. Max DP-In Port Support = 0x%X\n", MaxDpInExtPortCpuSupport));
  }

  return MaxDpInExtPortCpuSupport;
}

/**
  This function performs set/update for Dp-In Policy in IOM SW registers.

  @param[in]  DpInPreMemConfig             Pointer to Dp-In PreMem Pei Config Structure
  @param[in]  DpInPortConnectMap           External DpIn Port Connect Map
**/
VOID
EFIAPI
DpInPolicyInitPreMem (
  IN DPIN_PREMEM_CONFIG          *DpInPreMemConfig,
  IN UINT8                       DpInPortConnectMap
  )
{
  UINT32                          IomTypecConfigValue;
  UINT16                          IomTypecSwConfig1Reg;
  UINT32                          DpInGfxSrcPolicyValue;
  UINT16                          DpInGfxSrcPolicyMgmtReg;

  DEBUG ((DEBUG_INFO, "[DPIN] DpInPolicyInitPreMem - Start"));

  IomTypecSwConfig1Reg    = (UINT16)R_IOM_TYPEC_SW_CONFIGURATION_1;
  DpInGfxSrcPolicyMgmtReg = (UINT16)R_DPIN_GFX_SRC_POLICY_MGMT;

  //
  // Read IOM_TYPEC_SW_CONFIGURATION_1
  //
  IomTypecConfigValue = CpuRegbarRead32 (CPU_SB_PID_IOM,IomTypecSwConfig1Reg);

  //
  // Read R_SA_DPIN_GFX_SRC_POLICY_MGMT
  //
  DpInGfxSrcPolicyValue = CpuRegbarRead32 (CPU_SB_PID_IOM,DpInGfxSrcPolicyMgmtReg);

  //
  // CLEAR BITS 7:0 of IOM_TYPEC_SW_CONFIGURATION_1
  //
  IomTypecConfigValue &= ~((UINT32)(DP_IN_STATUS_PORT_DETECT_MAP_DATA_MASK));

  //
  // SET BITS 7:0 of IOM_TYPEC_SW_CONFIGURATION_1
  //
  if ((DpInPreMemConfig->DpInExternalEn == TRUE)
      && ((DpInPortConnectMap & DP_IN_STATUS_PORT_DETECT_MAP_DATA_MASK) != 0)) {
    IomTypecConfigValue |= (DpInPortConnectMap & DP_IN_STATUS_PORT_DETECT_MAP_DATA_MASK);
  }

  //
  // CLEAR BITS 7:0 of R_SA_DPIN_GFX_SRC_POLICY_MGMT
  //
  DpInGfxSrcPolicyValue &= ~((UINT32) (DP_IN_STATUS_PORT_DETECT_MAP_DATA_MASK));
  if ((DpInPreMemConfig->DpInExternalEn == TRUE)
      && ((DpInPortConnectMap & DP_IN_STATUS_PORT_DETECT_MAP_DATA_MASK) != 0)) {
    //
    // SET BITS 7:0 of R_SA_DPIN_GFX_SRC_POLICY_MGMT
    //
    DpInGfxSrcPolicyValue |= 1; // 0x0 = Only Internal, 0x1 = 1st external then internal
    DEBUG ((DEBUG_INFO, "[DPIN] DpInPolicyInitPreMem - DPin Policy is set to External first.\n"));
  } else {
    DEBUG ((DEBUG_INFO, "[DPIN] DpInPolicyInitPreMem - DPin Policy is set to Only Internal.\n"));
  }

  //
  // Write to IOM_TYPEC_SW_CONFIGURATION_1
  //
  CpuRegbarWrite32 (CPU_SB_PID_IOM, IomTypecSwConfig1Reg, IomTypecConfigValue);

  //
  // Write to R_SA_DPIN_GFX_SRC_POLICY_MGMT
  //
  CpuRegbarWrite32 (CPU_SB_PID_IOM, DpInGfxSrcPolicyMgmtReg, DpInGfxSrcPolicyValue);

  DEBUG ((DEBUG_INFO, "[DPIN] DpInPolicyInitPreMem - End"));
}

/**
  This function check if any Platform Data Error Status found for Dp-In Policy.

  @param[in]   DpInPreMemConfig          Pointer to Dp-In PreMem Pei Config Structure
  @param[in]   MaxDpInExtPortCpuSupport  Maximum DP-In External Port supported by CPU.

  @retval     TRUE          Error found in Platform Data. Because of which External DP-In will not work.
  @retval     FALSE         No Error found in Platform Data.
**/
BOOLEAN
EFIAPI
DpInPlatformDataErrorLogging (
  IN  DPIN_PREMEM_CONFIG              *DpInPreMemConfig,
  IN  UINT8                           MaxDpInExtPortCpuSupport
  )
{
  BOOLEAN         PlatformErrorFound;

  PlatformErrorFound = TRUE;
  if ((DpInPreMemConfig->NumberOfDpInPort != 0) &&
      (DpInPreMemConfig->NumberOfDpInPort <= PcdGet8 (PcdMaxDpInExtPortSupported)) &&
      (DpInPreMemConfig->NumberOfDpInPort <= MaxDpInExtPortCpuSupport)) {
    // Now check if atleast one DP-In External Port is connected
    if ((DpInPreMemConfig->DpInPortConnectMap & DP_IN_STATUS_PORT_DETECT_MAP_DATA_MASK) != 0) {
      PlatformErrorFound = FALSE; // No Platform Error Found
    }
  }
  return PlatformErrorFound;
}

/**
  This function performs basic initialization for Dp-In Policy in PEI phase after PEG Graphics Detected at Pre-Mem phase.

  @param[in]  DpInPreMemConfig                 Pointer to Dp-In PreMem Pei Config Structure
  @param[in]  MaxDpInExtPortCpuSupport         Maximum DP-In External Port Sopported by CPU

  @retval EFI_SUCCESS           The DP-In Status HOB was successfully Updated.
  @retval EFI_NOT_FOUND         The Dp-In Hob Not found.
  @retval EFI_ABORTED           DP-In policy is set to Internal only forcefully.
  @retval EFI_ACCESS_DENIED     DP-In External Support not available by CPU.
  @retval EFI_UNSUPPORTED       The Dp-In External Policy is not Supported.
  **/
EFI_STATUS
EFIAPI
DpInPostGfxInit (
  IN DPIN_PREMEM_CONFIG               *DpInPreMemConfig,
  IN UINT8                            MaxDpInExtPortCpuSupport
  )
{
  EFI_STATUS                      Status;

  DEBUG ((DEBUG_INFO, "[DPIN] DpInPostGfxInit() - Start\n"));

  Status = EFI_UNSUPPORTED;

  //
  // Check if CPU support Dp-In feature or not.
  //
  if (MaxDpInExtPortCpuSupport != 0) {
    //
    // Check if Dp-In connection Policy is External First or Not.
    //
    if (DpInPreMemConfig->DpInExternalEn == 1) {
      //
      // Check if Valid Platform Data is passed.
      //
      if (!DpInPlatformDataErrorLogging (DpInPreMemConfig, MaxDpInExtPortCpuSupport)) {
        Status = EFI_SUCCESS;
      } else {
        Status = EFI_ABORTED;
        DpInPreMemConfig->DpInPortConnectMap = 0; // Clearing Port Map as Dp-In policy is forcefully set to Internal Only.
      }
    } else {
      Status = EFI_SUCCESS;
      DpInPreMemConfig->DpInPortConnectMap = 0; // Clearing Port Map as Dp-In policy is Internal Only.
    }
  } else {
    Status = EFI_ACCESS_DENIED;
  }

  DEBUG ((DEBUG_INFO, "[DPIN] DpInPostGfxInit() - End\n"));
  return Status;
}

/**
  This function performs basic initialization for DPIN in PEI phase after Policy produced at Pre-Mem phase.

  @param[in]  SiPreMemPolicyPpi              Pointer to Si PreMem Policy Ppi
**/
VOID
EFIAPI
DpInPreMemInit (
  IN  SI_PREMEM_POLICY_PPI            *SiPreMemPolicyPpi
  )
{
  EFI_STATUS                      Status;
///TODO:  EFI_BOOT_MODE                   BootMode;
  DPIN_PREMEM_CONFIG          *DpInPreMemConfig;
  UINT8                       MaxDpInExtPortCpuSupport;
  UINT8                       DpInPortConnectMap;

  DEBUG ((DEBUG_INFO, "[DpIn] DpInPreMemInit() - Start\n"));
  DpInPortConnectMap = 0;
  MaxDpInExtPortCpuSupport = 0;
  //
  // Performing DpIn PreMem initialization from here.
  //
///TODO:  PeiServicesGetBootMode (&BootMode);
  DpInPreMemConfig = NULL;
  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gDpInPreMemConfigGuid, (VOID *) &DpInPreMemConfig);
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR(Status) || (DpInPreMemConfig == NULL)) {
    return;
  }

  ///
  /// Initialize Dp-In Pei PreMem Config Data.
  ///
  DEBUG ((DEBUG_INFO, "Dp-In External Gfx Card Support = %d From DpIn PreMemConfigBlock\n", DpInPreMemConfig->DpInExternalEn));
  MaxDpInExtPortCpuSupport = GetDpInPrivateCpuData ();

  ///
  /// Initialize Dp-In, Post PEG Graphics Detection
  ///
  DEBUG ((DEBUG_INFO, "Initializing Dp-In Config Policy\n"));
  Status = DpInPostGfxInit (DpInPreMemConfig, MaxDpInExtPortCpuSupport);
  if (EFI_ERROR(Status) && (Status != EFI_ABORTED)) {
    // Return if Status ERROR found, unless Operation was forcefully aborted (EFI_ABORTED)
    return;
  }

  // Assign Dp-In Port Connect Map Value DpIn External Policy supported.
  if (Status == EFI_SUCCESS) {
    DpInPortConnectMap = DpInPreMemConfig->DpInPortConnectMap;
  }

  ///
  /// Configure Dp-In Policy registers. And then Initialize DPIN PreMem
  ///
  if (MaxDpInExtPortCpuSupport != 0) {
    ///
    /// Configure IOM register IOM_TYPEC_SW_CONFIGURATION_1 and DPIN_GFX_SRC_POLICY_MANAGEMENT
    ///
    DEBUG ((DEBUG_INFO, "Initializing Dp-In Policy in IOM\n"));
    DpInPolicyInitPreMem ( DpInPreMemConfig, DpInPortConnectMap);
  }

  DEBUG ((DEBUG_INFO, "[DpIn] DpInPreMemInit() - End\n"));
}
