/** @file
  Initilize TME in PEI

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/PeiServicesLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/DebugLib.h>
#include <Library/TmeLib.h>
#include <Library/MsrFruLib.h>
#include <Library/PmcLib.h>
#include <CpuAccess.h>
#include <Register/Cpuid.h>
#include <Register/ArchMsr.h>
#include <Register/PmcRegs.h>

typedef struct {
  UINT64 TmeEnable          :1;
  UINT64 MkTmeMaxKeyIdBits  :4;
  UINT64 MkTmeMaxKey        :15;
  UINT64 TmePolicy          :4;
  UINT64 KeySelect          :1;
  UINT64 SaveKeyForStandby  :1;
  UINT64 MkTmeKeyIdBits     :4;
  UINT64 MkTmeCryptoAlgs    :16;
} TME_CONFIG;

/**
  Perform Multi-Key Total Memory Encryption initialization.

  @param[in] TmeEnable      - TME policy enable
  @param[in] TmeExcludeBase - Base physical address to be excluded for TME encryption
  @param[in] TmeExcludeSize - Size of range to be excluded from TME encryption

  @retval VOID - No value to return
**/
VOID
TmeInit (
  IN UINT32 TmePolicyEnable,
  IN UINT64 TmeExcludeBase,
  IN UINT64 TmeExcludeSize
  )
{
  EFI_CPUID_REGISTER                 CpuidRegs;
  MSR_TME_EXCLUDE_MASK_REGISTER      ExcludeMask;
  MSR_TME_EXCLUDE_BASE_REGISTER      ExcludeBase;
  EFI_BOOT_MODE                      BootMode;
  EFI_STATUS                         Status;
  UINT8                              Index;
  TME_CONFIG                         TmeConfig;
  BOOLEAN                            WarmReset;

  BootMode  = 0;
  Status    = 0;
  Index     = 0;
  WarmReset = FALSE;
  ZeroMem (&TmeConfig, sizeof (TME_CONFIG));

  DEBUG ((DEBUG_INFO, "Multi-Key Total Memory Encryption (MK-TME) Initialization\n"));

  ///
  /// Check if TME Policy is enabled
  ///
  if (TmePolicyEnable) {
    ///
    /// Verify TME supported through CPUID.7.0.ECX.13
    ///
    AsmCpuidEx (CPUID_STRUCTURED_EXTENDED_FEATURE_FLAGS,
                0,
                &CpuidRegs.RegEax,
                &CpuidRegs.RegEbx,
                &CpuidRegs.RegEcx,
                &CpuidRegs.RegEdx
                );

    if ((CpuidRegs.RegEcx & BIT13) != BIT13) {
      DEBUG ((DEBUG_INFO, "MK-TME: Processor does not support MK-TME (CPUID.7.0.ECX = 0x%X).\n", CpuidRegs.RegEcx));
      DEBUG ((DEBUG_INFO, "MK-TME: Skip MK-TME initialization.\n"));
      return;
    }

    ///
    /// Check if AES-XTS-256 encryption is supported
    ///
    if (MsrIsAesXts256Supported()) {
      DEBUG ((DEBUG_INFO, "MK-TME: AES-XTS-256 encryption is supported.\n"));
      ///
      /// Set TME policy
      ///
      TmeConfig.TmePolicy = V_TME_ACTIVATE_TME_POLICY_AES_XTS_256;
    } else {
      DEBUG ((DEBUG_INFO, "MK-TME: AES-XTS-256 encryption is not supported.\n"));
      DEBUG ((DEBUG_INFO, "MK-TME: Skip MK-TME initialization.\n"));

      ///
      /// Lock MSR before returning
      ///
      MsrLockTmeActivate(0);
      return;
    }

    ///
    /// Read Max supported Keys
    ///
    TmeConfig.MkTmeMaxKeyIdBits = (UINT8) MsrGetMkTmeMaxKeyidBits();
    TmeConfig.MkTmeMaxKey       = (UINT16) MsrGetMkTmeMaxKey();
    if ((TmeConfig.MkTmeMaxKeyIdBits != 0) && (TmeConfig.MkTmeMaxKey != 0)) {
      DEBUG ((EFI_D_INFO, "MK-TME: Multi-Key is supported.\n"));
    } else {
      DEBUG ((EFI_D_INFO, "MK-TME: Multi-Key is not supported.\n"));
      DEBUG ((DEBUG_INFO, "MK-TME: Skip MK-TME initialization.\n"));

      ///
      /// Lock MSR before returning
      ///
      MsrLockTmeActivate(0);
    }

    ///
    /// Configure MSR_TME_EXCLUDE_MASK (983H) with TME Mask and Enable bit
    ///
    if (TmeExcludeSize != 0) {
      ExcludeMask.Uint64 = TmeExcludeSize;
      ExcludeMask.Bits.Enable = 1;
      AsmWriteMsr64 (MSR_TME_EXCLUDE_MASK, ExcludeMask.Uint64);
    }

    ///
    /// Configure MSR_TME_EXCLUDE_BASE (984H) with TME Base
    ///
    if (TmeExcludeBase != 0) {
      ExcludeBase.Uint64 = TmeExcludeBase;
      AsmWriteMsr64 (MSR_TME_EXCLUDE_BASE, ExcludeBase.Uint64);
    }

    ExcludeMask.Uint64 = AsmReadMsr64 (MSR_TME_EXCLUDE_BASE);
    DEBUG ((DEBUG_INFO, "TME_EXCLUDE_BASE (0x984) = %016llx\n"));
    DEBUG ((DEBUG_INFO, " [BIT 11]    Enable:       0x%X\n", ExcludeMask.Bits.Enable));
    DEBUG ((DEBUG_INFO, " [BIT 31:12] ExcludeMask0: 0x%X\n", ExcludeMask.Bits.Tmemask0));
    DEBUG ((DEBUG_INFO, " [BIT 63:32] ExcludeMask1: 0x%X\n", ExcludeMask.Bits.Tmemask1));

    ExcludeBase.Uint64 = AsmReadMsr64 (MSR_TME_EXCLUDE_MASK);
    DEBUG ((DEBUG_INFO, "TME_EXCLUDE_MASK (0x983) = %016llx \n"));
    DEBUG ((DEBUG_INFO, " [BIT 31:12] ExcludeBase0: 0x%X\n", ExcludeBase.Bits.Tmebase0));
    DEBUG ((DEBUG_INFO, " [BIT 63:32] ExcludeBase1: 0x%X\n", ExcludeBase.Bits.Tmebase1));

    ///
    /// Set TME Key Select and Save Key for Warm Reset, Standby and Flash Update Mode
    /// Key Select
    ///   - Set for Warm Reset, S3 resume flow and flash update flow to restore
    ///     TME keys from PCH
    ///   - Clear for cold boot to create new TME keys
    /// Save Key
    ///   - Clear for S3 resume flow and flash update flow
    ///   - Set for cold boot to save key into storage
    ///
    Status = PeiServicesGetBootMode (&BootMode);
    ASSERT_EFI_ERROR (Status);

    ///
    /// Detect Warm Reset boot
    ///
    RegisterVal32 = MmioRead32 ((UINTN) (PmcGetPwrmBase () + R_PMC_PWRM_GEN_PMCON_A));
    if ((((RegisterVal32 & B_PMC_PWRM_GEN_PMCON_A_MEM_SR) != 0) &&
        ((RegisterVal32 & B_PMC_PWRM_GEN_PMCON_A_DISB) != 0))) {
      WarmReset = TRUE;
    }

    if ((BootMode == BOOT_ON_S3_RESUME) ||
        (BootMode == BOOT_ON_FLASH_UPDATE) ||
        (WarmReset == TRUE)) {
      TmeConfig.KeySelect = 1;
      TmeConfig.SaveKeyForStandby = 0;
    } else {
      TmeConfig.KeySelect = 0;
      TmeConfig.SaveKeyForStandby = 1;
    }

    ///
    /// Configure KeyId to max supported
    ///
    TmeConfig.MkTmeKeyIdBits  = TmeConfig.MkTmeMaxKeyIdBits;
    TmeConfig.MkTmeCryptoAlgs = 0x4;  // value 0x4 equates to MSR_TME_ACTIVATE (982H) [BIT50] set

    ///
    /// Set TME enable
    /// TME capability and encryption policy is supported
    ///
    TmeConfig.TmeEnable = 1;

    ///
    /// Configure MSR_TME_ACTIVATE (982H) with TME Enable, Key Select, Save Key, and TME Policy
    /// Lock bit will be set upon successful WRMSR for MSR_TME_ACTIVATE.
    ///   - First SMI will also set the Lock
    ///   - This will also lock MSR_TME_EXCLUDE_MASK and MSR_TME_EXCLUDE_BASE
    ///
    DEBUG ((DEBUG_INFO, "MK-TME: Initialize MSR_TME_ACTIVATE\n"));

    // TME activate retry limit is 10
    for (Index = 0; Index < 10; Index++) {
      MsrConfigureTmeActivate((UINT8) TmeConfig.TmeEnable,
                              (UINT8) TmeConfig.KeySelect,
                              (UINT8) TmeConfig.SaveKeyForStandby,
                              (UINT8) TmeConfig.TmePolicy,
                              (UINT8) TmeConfig.MkTmeKeyIdBits,
                              (UINT16) TmeConfig.MkTmeCryptoAlgs
                              );

      if (MsrIsTmeActivateLockSet()) {
        DEBUG ((DEBUG_INFO, "MK-TME: MSR_TME_ACTIVATE was initialized and locked\n"));
        return;
      }
    }
  } else {
    DEBUG ((DEBUG_INFO, "MK-TME: Policy is disabled in BIOS Setup.\n"));
    DEBUG ((DEBUG_INFO, "MK-TME: Skip MK-TME initialization.\n"));

    ///
    /// Ensure Lock bit of MSR_TME_ACTIVATE (982H) is set
    /// Lock bit will not be set if creation of ephemeral number using DRNG action failed.
    ///
    DEBUG ((DEBUG_INFO, "MK-TME: TME was not enabled. Locking MSR_TME_ACTIVATE.\n"));
    MsrLockTmeActivate(0);
  }

  return;
}

/**
  Reports if TME is enabled or not

  @retval TRUE             If TME is enabled
  @retval FALSE            If TME is not enabled
**/
BOOLEAN
IsTmeEnabled (
  VOID
  )
{
  return MsrIsTmeEnabled();
}

/**
  Reports if Lock bit in TME Activate MSR locked or not

  @retval TRUE             If TME Activate MSR is locked
  @retval FALSE            If TME Activate MSR is not locked
**/
BOOLEAN
IsTmeActivateLockSet (
  VOID
  )
{
  return MsrIsTmeActivateLockSet();
}
