/** @file
  Source file for BZM (Boot Time Zero Memory) feature initialization.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/TwoLmInitLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/PeiServicesTablePointerLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/IoLib.h>
#include <Register/Cpuid.h>
#include <Register/Msr.h>
#include <Register/TwoLmRegs.h>


/**
  Program the BZM range registers of Astro controller.

  @param[in] BzmIndex    index for BZM range registers.
  @param[in] Edrambar    value of Edrambar.
  @param[in] Base        start address of memory range.
  @param[in] Length      length of memory range.

**/
VOID
ProgramBzmRanges (
  IN UINTN  BzmIndex,
  IN UINTN  Edrambar,
  IN UINT64 Base,
  IN UINT64 Length
) {

  if (BzmIndex >= 8) {
    ASSERT(FALSE);
    return;
  }
  MmioWrite64 (Edrambar + ASTRO_OFFSET + R_ASTRO_BZM_START_ADDR0 + (BzmIndex * 16), Base);
  // Adding correction for end addr
  MmioWrite64 (Edrambar + ASTRO_OFFSET + R_ASTRO_BZM_END_ADDR0   + (BzmIndex * 16), (Base+Length));
}


/**
  Sort memory resource entries based upon PhysicalStart, from low to high.

  @param[in, out] MemoryResource    A pointer to the memory resource entry buffer.

**/
VOID
SortMemoryResourceDescriptor (
  IN OUT MEMORY_RESOURCE_DESCRIPTOR *MemoryResource
  )
{
  MEMORY_RESOURCE_DESCRIPTOR        *MemoryResourceEntry;
  MEMORY_RESOURCE_DESCRIPTOR        *NextMemoryResourceEntry;
  MEMORY_RESOURCE_DESCRIPTOR        TempMemoryResource;

  MemoryResourceEntry = MemoryResource;
  NextMemoryResourceEntry = MemoryResource + 1;
  while (MemoryResourceEntry->ResourceLength != 0) {
    while (NextMemoryResourceEntry->ResourceLength != 0) {
      if (MemoryResourceEntry->PhysicalStart > NextMemoryResourceEntry->PhysicalStart) {
        CopyMem (&TempMemoryResource, MemoryResourceEntry, sizeof (MEMORY_RESOURCE_DESCRIPTOR));
        CopyMem (MemoryResourceEntry, NextMemoryResourceEntry, sizeof (MEMORY_RESOURCE_DESCRIPTOR));
        CopyMem (NextMemoryResourceEntry, &TempMemoryResource, sizeof (MEMORY_RESOURCE_DESCRIPTOR));
      }

      NextMemoryResourceEntry = NextMemoryResourceEntry + 1;
    }

    MemoryResourceEntry     = MemoryResourceEntry + 1;
    NextMemoryResourceEntry = MemoryResourceEntry + 1;
  }
}

/**
  Merge continous memory resource entries.

  @param[in, out] MemoryResource    A pointer to the memory resource entry buffer.

**/
VOID
MergeMemoryResourceDescriptor (
  IN OUT MEMORY_RESOURCE_DESCRIPTOR *MemoryResource
  )
{
  MEMORY_RESOURCE_DESCRIPTOR        *MemoryResourceEntry;
  MEMORY_RESOURCE_DESCRIPTOR        *NewMemoryResourceEntry;
  MEMORY_RESOURCE_DESCRIPTOR        *NextMemoryResourceEntry;
  MEMORY_RESOURCE_DESCRIPTOR        *MemoryResourceEnd;

  MemoryResourceEntry = MemoryResource;
  NewMemoryResourceEntry = MemoryResource;
  while (MemoryResourceEntry->ResourceLength != 0) {
    CopyMem (NewMemoryResourceEntry, MemoryResourceEntry, sizeof (MEMORY_RESOURCE_DESCRIPTOR));
    NextMemoryResourceEntry = MemoryResourceEntry + 1;

    while ((NextMemoryResourceEntry->ResourceLength != 0) &&
           (NextMemoryResourceEntry->PhysicalStart == (MemoryResourceEntry->PhysicalStart + MemoryResourceEntry->ResourceLength))) {
      MemoryResourceEntry->ResourceLength += NextMemoryResourceEntry->ResourceLength;
      if (NewMemoryResourceEntry != MemoryResourceEntry) {
        NewMemoryResourceEntry->ResourceLength += NextMemoryResourceEntry->ResourceLength;
      }

      NextMemoryResourceEntry = NextMemoryResourceEntry + 1;
    }

    MemoryResourceEntry = NextMemoryResourceEntry;
    NewMemoryResourceEntry = NewMemoryResourceEntry + 1;
  }

  //
  // Set NULL terminate memory resource descriptor after merging.
  //
  MemoryResourceEnd = NewMemoryResourceEntry;
  ZeroMem (MemoryResourceEnd, sizeof (MEMORY_RESOURCE_DESCRIPTOR));
}

/**
  Build memory resource descriptor from resource descriptor in HOB list.
  1. This function finds the Memory resource descriptors in HOB list.
  2. Looks for the memory regions above 4GB.
  3. If the regions above 4GB are more than one then the region with
     max length is further partitioned into available ranges.
     Other regions above 4GB are reported as one range.
  4. Programs the BZM ranges in Astro registers.

  @return Pointer to the buffer of memory resource descriptor.
          NULL if no memory resource descriptor reported in HOB list
          before capsule Coalesce.
**/
VOID
SplitMemoryRanges (
  VOID
  )
{
  EFI_PEI_HOB_POINTERS          Hob;
  UINTN                         Index;
  UINTN                         SubIndex;
  UINTN                         BzmIndex;
  UINTN                         Above4gbRegions;
  UINTN                         RemainingBzmRanges;
  UINTN                         Remainder;
  EFI_HOB_RESOURCE_DESCRIPTOR   *ResourceDescriptor;
  MEMORY_RESOURCE_DESCRIPTOR    *MemoryResource;
  EFI_STATUS                    Status;
  UINT64                        Dividend;
  UINT64                        MaxLength;
  UINT64                        Base;
  UINT64                        Length;
  UINTN                         Edrambar;
  UINTN                         MchBar;


  MchBar = PegPciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, 0, 0, 0) + R_SA_MCHBAR) & ~BIT0;
   Edrambar = MmioRead64 (MchBar +R_SA_MCHBAR_EDRAMBAR_OFFSET) & ~BIT0;
   if(!Edrambar) {
     ASSERT (EFI_UNSUPPORTED);
   }

  //
  // Get the count of memory resource descriptor above 4GB and without persistent
  // and reserved attributes
  //
  Index = 0;
  BzmIndex = 0;
  Hob.Raw = GetFirstHob (EFI_HOB_TYPE_RESOURCE_DESCRIPTOR);
  while (Hob.Raw != NULL) {
    ResourceDescriptor = (EFI_HOB_RESOURCE_DESCRIPTOR *) Hob.Raw;
    if ((ResourceDescriptor->ResourceType == EFI_RESOURCE_SYSTEM_MEMORY) &&
        (ResourceDescriptor->PhysicalStart >= 0x100000000) &&
        ((ResourceDescriptor->ResourceAttribute & EFI_RESOURCE_ATTRIBUTE_PERSISTENT) != EFI_RESOURCE_ATTRIBUTE_PERSISTENT)) {
      Index++;
    }
    Hob.Raw = GET_NEXT_HOB (Hob);
    Hob.Raw = GetNextHob (EFI_HOB_TYPE_RESOURCE_DESCRIPTOR, Hob.Raw);
  }

  //
  // Allocate memory to hold memory resource descriptor,
  // include extra one NULL terminate memory resource descriptor.
  //
  Status = PeiServicesAllocatePool ((Index + 1) * sizeof (MEMORY_RESOURCE_DESCRIPTOR), (VOID **) &MemoryResource);
  ASSERT_EFI_ERROR (Status);
  ZeroMem (MemoryResource, (Index + 1) * sizeof (MEMORY_RESOURCE_DESCRIPTOR));

  //
  // Get the content of memory resource descriptor.
  //
  Index = 0;
  Hob.Raw = GetFirstHob (EFI_HOB_TYPE_RESOURCE_DESCRIPTOR);
  while (Hob.Raw != NULL) {
    ResourceDescriptor = (EFI_HOB_RESOURCE_DESCRIPTOR *) Hob.Raw;
    if ((ResourceDescriptor->ResourceType == EFI_RESOURCE_SYSTEM_MEMORY) &&
        (ResourceDescriptor->PhysicalStart >= 0x100000000) &&
        ((ResourceDescriptor->ResourceAttribute & EFI_RESOURCE_ATTRIBUTE_PERSISTENT) != EFI_RESOURCE_ATTRIBUTE_PERSISTENT)) {
      MemoryResource[Index].PhysicalStart = ResourceDescriptor->PhysicalStart;
      MemoryResource[Index].ResourceLength = ResourceDescriptor->ResourceLength;
      Index++;
    }
    Hob.Raw = GET_NEXT_HOB (Hob);
    Hob.Raw = GetNextHob (EFI_HOB_TYPE_RESOURCE_DESCRIPTOR, Hob.Raw);
  }

  SortMemoryResourceDescriptor (MemoryResource); // to arrange mem regions in incresing order
  MergeMemoryResourceDescriptor (MemoryResource); // create one region for contiguous ranges

  for (Index = 0; MemoryResource[Index].ResourceLength != 0; Index++) {
    DEBUG (( DEBUG_VERBOSE, "  MemoryResource[0x%x] - Start(0x%0lx) Length(0x%0lx)\n",Index,
        MemoryResource[Index].PhysicalStart,  MemoryResource[Index].ResourceLength));
  }

  // Find the total number of regions above 4GB and a region with MaxLength
  Above4gbRegions = 0;
  MaxLength = 0;
  for (Index = 0; MemoryResource[Index].ResourceLength != 0; Index++) {
    if (MemoryResource[Index].ResourceLength > MaxLength) {
      MaxLength = MemoryResource[Index].ResourceLength;
    }
    Above4gbRegions++;
  }
  RemainingBzmRanges = 8 - Above4gbRegions + 1; // as we are further partitioning one of the largest region in this

  // To further partition region with max length, finding range divider
  Dividend = DivU64x32Remainder (MaxLength, RemainingBzmRanges, &Remainder);
  // Make it MB aligned
  Dividend &= ~ (0xFFFFF);
  // Program ranges
  for (Index = 0; MemoryResource[Index].ResourceLength != 0; Index++) {
    if (MemoryResource[Index].ResourceLength == MaxLength) {
      // further partition and program this range
      Base = MemoryResource[Index].PhysicalStart;
      Length = Dividend;
      for (SubIndex = 1; SubIndex <= RemainingBzmRanges; SubIndex++ ) {
        if (SubIndex == RemainingBzmRanges) {
          // Program remaining length
          Length = MemoryResource[Index].PhysicalStart + MemoryResource[Index].ResourceLength - Base;
        }
        ProgramBzmRanges(BzmIndex, Edrambar, Base, Length);
        BzmIndex++;
        Base += Length;
      }
    } else { // Program complete range
      ProgramBzmRanges(BzmIndex, Edrambar, MemoryResource[Index].PhysicalStart, MemoryResource[Index].ResourceLength);
      BzmIndex++;
    } // end of else
  } // end of for
  return;
}

/**
  BZM Initialization is done after the MrcMemoryInitDonePpi is produced.

  @param[in] PeiServices          General purpose services available to every PEIM.
  @param[in] NotifyDescriptor     The notification structure this PEIM registered on install.
  @param[in] Ppi                  The memory discovered PPI.  Not used.

  @retval EFI_SUCCESS             Succeeds.
          EFI_UNSUPPORTED         if the feature is not supported.
**/
EFI_STATUS
EFIAPI
BzmMemoryInitDoneCallback (
  IN  EFI_PEI_SERVICES             **PeiServices,
  IN  EFI_PEI_NOTIFY_DESCRIPTOR    *NotifyDescriptor,
  IN  VOID                         *Ppi
)
{
  UINT32                            RegEcx;
  MSR_IA32_FEATURE_CONTROL_REGISTER Msr;
  TWOLM_INFO_HOB                    *TwoLmInfoHob;
  SI_PREMEM_POLICY_PPI              *SiPreMemPolicyPpi;
  TWOLM_PREMEM_CONFIG               *TwoLmPreMemConfig;
  EFI_STATUS                        Status;
  EFI_BOOT_MODE                     BootMode;


  TwoLmInfoHob = (TWOLM_INFO_HOB *) GetFirstGuidHob (&gTwoLmInfoHobGuid);
  if (TwoLmInfoHob == NULL) {
    return EFI_UNSUPPORTED;
  }

  if (TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode != MEM_BOOT_MODE_2LM) {
    DEBUG ((DEBUG_INFO, "2LM: TwoLm boot mode is not Enabled\n"));
    return EFI_UNSUPPORTED;
  }

  ///
  /// Get policy settings through the SaInitConfigBlock PPI
  ///
  Status = PeiServicesLocatePpi(
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **)&SiPreMemPolicyPpi
  );
  ASSERT_EFI_ERROR(Status);
  if ((Status != EFI_SUCCESS) || (SiPreMemPolicyPpi == NULL)) {
    return EFI_NOT_FOUND;
  }

  Status = GetConfigBlock((VOID *)SiPreMemPolicyPpi, &gTwoLmPreMemConfigGuid, (VOID *)&TwoLmPreMemConfig);
  ASSERT_EFI_ERROR(Status);

  TwoLmPreMemConfig = NULL;
  Status = GetConfigBlock ((VOID *)SiPreMemPolicyPpi, &gTwoLmPreMemConfigGuid, (VOID *) &TwoLmPreMemConfig);
  ASSERT_EFI_ERROR(Status);

  if (TwoLmPreMemConfig != NULL) {
    if (!TwoLmPreMemConfig->BzmSupport) {
      return EFI_UNSUPPORTED;
    }
  }

  Status = PeiServicesGetBootMode (&BootMode);
  ASSERT_EFI_ERROR (Status);

  if ((BootMode == BOOT_ON_S3_RESUME) || (BootMode == BOOT_ON_FLASH_UPDATE)) {
    return EFI_SUCCESS;
  }

  // Check BZM support through CPUID.7.0.ECX[15]
  // Read the CPUID information
  AsmCpuidEx ( 7, 0, NULL, NULL, &RegEcx, NULL );
  if (!(RegEcx & CPUID_BZM_SUPPORTED)) {
    return EFI_UNSUPPORTED;
  }

  // Split OS visible memory range in to 8 regions. Program each region Start/End address in to BZM_START_ADDR and BZM_END_ADDR Astro registers
  SplitMemoryRanges();

  // Set Bit21 of FEATURE_CONTROL MSR to indicate that BZM is enabled.
  Msr.Uint64  = AsmReadMsr64 (MSR_IA32_FEATURE_CONTROL);
  // Check if BZM is already enabled
  /// Need to reset only if locked and BZM state has to be changed.
  ///
  if (Msr.Uint64 & B_IA32_FEATURE_CONTROL_BZM) {
    return EFI_SUCCESS;
  }
  if (Msr.Bits.Lock == 1) {
    (*GetPeiServicesTablePointer ())->ResetSystem2 (EfiResetCold, EFI_SUCCESS, 0, NULL);
  } else {
    AsmWriteMsr64 (MSR_IA32_FEATURE_CONTROL, (Msr.Uint64 | B_IA32_FEATURE_CONTROL_BZM));
  }
  return EFI_SUCCESS;
}
