/** @file
  Nvdimm Label Protocol Driver

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <NvdimmLabelDxe.h>

#define LABEL_PAYLOAD_SIZE   512

UINT8                 RegionIdPMem;

EFI_NVDIMM_LABEL_PROTOCOL mLabelStorageInterface = {
  NvdimmLabelStorageInformation,
  NvdimmLabelStorageRead,
  NvdimmLabelStorageWrite
};


/**
  Retrieves the Label Storage Area size and the maximum transfer size for the LabelStorageRead and
  LabelStorageWrite methods.

  @param  This                   A pointer to the EFI_NVDIMM_LABEL_PROTOCOL instance.
  @param  SizeOfLabelStorageArea The size of the Label Storage Area for the NVDIMM in bytes.
  @param  MaxTransferLength      The maximum number of bytes that can be transferred in a single call to
                                 LabelStorageRead or LabelStorageWrite.

  @retval EFI_SUCCESS            The size of theLabel Storage Area and maximum transfer size returned are valid.
  @retval EFI_ACCESS_DENIED      The Label Storage Area for the NVDIMM device is not currently accessible.
  @retval EFI_DEVICE_ERROR       A physical device error occurred and the data transfer failed to complete.
**/
EFI_STATUS
EFIAPI
NvdimmLabelStorageInformation (
  IN  EFI_NVDIMM_LABEL_PROTOCOL *This,
  OUT UINT32                    *SizeOfLabelStorageArea,
  OUT UINT32                    *MaxTransferLength
  )
{
  UINT8       DeviceReturnStatus;
  UINT32      OutputData[3];
  UINT8       OutputPayloadSizeDw;
  EFI_STATUS Status;

  OutputPayloadSizeDw = 3;
  Status = GetLabelInfoCommand ( &DeviceReturnStatus, (UINT32 *)OutputData, OutputPayloadSizeDw, RegionIdPMem);

  if(Status == EFI_ACCESS_DENIED) {
    return Status;
  }
  else if (Status != EFI_SUCCESS ) {
    return EFI_DEVICE_ERROR;
  }

  *SizeOfLabelStorageArea = OutputData[1];
  *MaxTransferLength = OutputData[2];

  return Status;
}

/**
  Retrieves the label data for the requested offset and length from within the Label Storage Area for
  the NVDIMM.

  @param  This                   A pointer to the EFI_NVDIMM_LABEL_PROTOCOL instance.
  @param  Offset                 The byte offset within the Label Storage Area to read from.
  @param  TransferLength         Number of bytes to read from the Label Storage Area beginning at the byte
                                 Offset specified. A TransferLength of 0 reads no data.
  @param  LabelData              The return label data read at the requested offset and length from within
                                 the Label Storage Area.

  @retval EFI_SUCCESS            The label data from the Label Storage Area for the NVDIMM was read successfully
                                 at the specified Offset and TransferLength and LabelData contains valid data.
  @retval EFI_INVALID_PARAMETER  Any of the following are true:
                                 - Offset > SizeOfLabelStorageArea reported in the LabelStorageInformation return data.
                                 - Offset + TransferLength is > SizeOfLabelStorageArea reported in the
                                   LabelStorageInformation return data.
                                 - TransferLength is > MaxTransferLength reported in the LabelStorageInformation return
                                   data.
  @retval EFI_ACCESS_DENIED      The Label Storage Area for the NVDIMM device is not currently accessible and labels
                                 cannot be read at this time.
  @retval EFI_DEVICE_ERROR       A physical device error occurred and the data transfer failed to complete.
**/
EFI_STATUS
EFIAPI
NvdimmLabelStorageRead (
  IN CONST EFI_NVDIMM_LABEL_PROTOCOL *This,
  IN UINT32                          Offset,
  IN UINT32                          TransferLength,
  OUT UINT8                          *LabelData
  )
{
  UINT8       OutputPayloadSizeDw;
  UINT32      SizeOfLabelStorageArea;
  UINT32      MaxTransferLength;
  UINT32      *InputBuffer;
  UINT8       DeviceReturnStatus;
  UINT32      i;
  EFI_STATUS Status;

  NvdimmLabelStorageInformation( (EFI_NVDIMM_LABEL_PROTOCOL *)This, &SizeOfLabelStorageArea, &MaxTransferLength);

  if ( Offset > SizeOfLabelStorageArea ||
       Offset + TransferLength > SizeOfLabelStorageArea ||
       TransferLength > MaxTransferLength ||
       LabelData == NULL ) {
    return EFI_INVALID_PARAMETER;
  }
  if (TransferLength % sizeof(UINT32) !=0) {
    DEBUG ((DEBUG_ERROR, "2LM: LabelDriverRead - Transfer Length not DWORD Aligned\n"));
    ASSERT_EFI_ERROR (EFI_INVALID_PARAMETER);
  }

  InputBuffer = (UINT32 *)LabelData;
  InputBuffer[0] = Offset;
  InputBuffer[1] = TransferLength;
  DEBUG ((DEBUG_INFO, "Label Read. Offset = %x, Size in bytes = %x\n", InputBuffer[0], InputBuffer[1]));
  OutputPayloadSizeDw = (UINT8)(TransferLength/4);

  if (0 == OutputPayloadSizeDw) {
    DEBUG ((DEBUG_WARN, "2LM: LabelDriverRead with TransferLength zero. No call necessary\n"));
    return EFI_SUCCESS;
  }

  DEBUG ((DEBUG_INFO, "TransferLength: %x  And OutputPayloadSizeDw: %x \n", TransferLength, OutputPayloadSizeDw));
  DEBUG ((DEBUG_INFO, "2LM: Issuing GetLabelDataCommand\n"));
  Status = GetLabelDataCommand (&DeviceReturnStatus, (UINT32 *)LabelData, OutputPayloadSizeDw, RegionIdPMem);

  if(Status == EFI_SUCCESS && DeviceReturnStatus == MB_SUCCESS) {
    for (i = 0; i < OutputPayloadSizeDw; ++i)
    {
      DEBUG ((DEBUG_INFO, "Index %x value %x\n", i, ((UINT32 *)LabelData)[i] ));
    }
  }
  else if(Status != EFI_ACCESS_DENIED) {
    return EFI_DEVICE_ERROR;
  }

  return Status; // For EFI_SUCCESS and EFI_ACCESS_DENIED leave it unchanged and return Status
}


/**
  Writes the label data for the requested offset and length in to the Label Storage Area for the NVDIMM.

  @param  This                   A pointer to the EFI_NVDIMM_LABEL_PROTOCOL instance.
  @param  Offset                 The byte offset within the Label Storage Area to write to.
  @param  TransferLength         Number of bytes to write to the Label Storage Area beginning at the byte
                                 Offset specified. A TransferLength of 0 writes no data.
  @param  LabelData              The return label data write at the requested offset and length from within
                                 the Label Storage Area.

  @retval EFI_SUCCESS            The label data from the Label Storage Area for the NVDIMM written read successfully
                                 at the specified Offset and TransferLength.
  @retval EFI_INVALID_PARAMETER  Any of the following are true:
                                 - Offset > SizeOfLabelStorageArea reported in the LabelStorageInformation return data.
                                 - Offset + TransferLength is > SizeOfLabelStorageArea reported in the
                                   LabelStorageInformation return data.
                                 - TransferLength is > MaxTransferLength reported in the LabelStorageInformation return
                                   data.
  @retval EFI_ACCESS_DENIED      The Label Storage Area for the NVDIMM device is not currently accessible and labels
                                 cannot be written at this time.
  @retval EFI_DEVICE_ERROR       A physical device error occurred and the data transfer failed to complete.
**/
EFI_STATUS
EFIAPI
NvdimmLabelStorageWrite (
  IN CONST EFI_NVDIMM_LABEL_PROTOCOL *This,
  IN UINT32                          Offset,
  IN UINT32                          TransferLength,
  IN UINT8                           *LabelData
  )
{
  UINT8       InputPayloadSizeDw;
  UINT32      SizeOfLabelStorageArea;
  UINT32      MaxTransferLength;
  UINT32      InputBuffer[LABEL_PAYLOAD_SIZE];
  UINT8       DeviceReturnStatus;
  EFI_STATUS Status;

  NvdimmLabelStorageInformation((EFI_NVDIMM_LABEL_PROTOCOL *)This, &SizeOfLabelStorageArea, &MaxTransferLength);

  if ( Offset > SizeOfLabelStorageArea ||
       Offset + TransferLength > SizeOfLabelStorageArea ||
       TransferLength > MaxTransferLength ||
       LabelData == NULL ) {
    return EFI_INVALID_PARAMETER;
  }
  if (TransferLength % sizeof(UINT32) !=0) {
    DEBUG ((DEBUG_ERROR, "2LM: LabelDriverWrite - Transfer Length not DWORD Aligned\n"));
    ASSERT_EFI_ERROR (EFI_INVALID_PARAMETER);
  }

  InputBuffer[0] = Offset;
  InputBuffer[1] = TransferLength;
  DEBUG ((DEBUG_INFO, "Label Read. Offset = %x, Size in bytes = %x\n", InputBuffer[0], InputBuffer[1]));

  CopyMem ((InputBuffer+256), LabelData, TransferLength);

  InputPayloadSizeDw = (UINT8)((256 + TransferLength)/4);

  if (0 == TransferLength/sizeof(UINT32)) {
    DEBUG ((DEBUG_WARN, "2LM: LabelDriverWrite with TransferLength zero.\n"));
  }

  DEBUG ((DEBUG_INFO, "TransferLength: %x  And OutputPayloadSizeDw: %x \n", TransferLength, InputPayloadSizeDw));
  DEBUG ((DEBUG_INFO, "2LM: Issuing SetLabelDataCommand\n"));

  Status = SetLabelDataCommand (&DeviceReturnStatus, (UINT32 *)InputBuffer, InputPayloadSizeDw, RegionIdPMem);

  if(Status != EFI_ACCESS_DENIED && Status != EFI_SUCCESS) {
    return EFI_DEVICE_ERROR;
  }

  return Status;
}


/**
  Entry Point

  @param[in] ImageHandle          Handle for this drivers loaded image protocol.
  @param[in] SystemTable          EFI system table.

  @retval EFI_SUCCESS             Returns EFI_SUCCESS if the Label Driver Interface Protocol was installed successfully.
  @retval EFI_UNSUPPORTED         Returns if TwoLmInfoHob not found or PMem size not found.
**/
EFI_STATUS
EFIAPI
NvdimmLabelDriverEntryPoint (
  IN EFI_HANDLE         ImageHandle,
  IN EFI_SYSTEM_TABLE   *SystemTable
  )
{
  TWOLM_INFO_HOB            *TwoLmInfoHob;
  EFI_STATUS                Status;

  TwoLmInfoHob = (TWOLM_INFO_HOB *) GetFirstGuidHob (&gTwoLmInfoHobGuid);
  if (TwoLmInfoHob == NULL) {
    DEBUG ((DEBUG_INFO, "2LM: TwoLmInfoHob not found\n"));
    return EFI_UNSUPPORTED;
  }

  if (TwoLmInfoHob->TwoLmInfo.PMemSize == 0) {
    DEBUG ((DEBUG_INFO, "PMEM Disabled, skip NVDIMM Label Driver Protocol Install\n"));
    return EFI_UNSUPPORTED;
  }

  RegionIdPMem = TwoLmInfoHob->TwoLmInfo.RegionIdPMem;
  DEBUG((DEBUG_INFO, "RegionIdPMem = %x\n", RegionIdPMem));

  Status = gBS->InstallProtocolInterface (
                &ImageHandle,
                &gEfiNvdimmLabelProtocolGuid,
                EFI_NATIVE_INTERFACE,
                &mLabelStorageInterface
                );
  return Status;
}
