/** @file
  This file has SMM wrapper callback which passes FIS command from NVDIMM ASL code.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019-2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include "NvdimmSmm.h"


NVDIMM_NVS            *gNfitNvs;
UINT32                Offset;
UINT32                TransferLength;
UINT64                PMemBase;
UINT64                PMemSize;
UINT16                CmdEffectOpcodeCnt;
UINT16                SupportedModes;
UINT32                FwContext;
UINT8                 InitFwUpdate;
UINT8                 RegionIdPMem;
UpdateFwPacket        FwPacket;

/**
This function returns the status code for DCPMM DSMs as per the mapping table of
FIS to DCPMM error code mapping.

@param[in]      Return code of FIS Mailbox

@param[out]     Return code of DCPMM DSM
**/
UINT32
MapReturnCodeToDD (
  UINT8 FisReturnCode
  )
{
  DEBUG((DEBUG_VERBOSE, "Mapping Return code %d from FIS to DSM \n", FisReturnCode));
  switch(FisReturnCode) {
    case MB_SUCCESS:
      return DD_Success;

    case InvalidCommandParameter:
      return DD_InvalidInputParameters;
    case  DataTransferError:
    case  InternalDeviceError:
      return DD_HwError;

    case  UnsupportedCommand:
      return DD_FunctionNotSupported;

    case DeviceBusy:
    case Aborted:
    case TimeoutOccurred:
      return DD_RetrySuggested;

    case  IncorrectPassphraseOrSecurityNonce:
      return DD_InvalidCurrentPassphraseSupp;

    case  FWAuthenticationFailed:
      // Need to map this to extended status field for FW DSMs
      // For DCPMM DSM it is value 01 - FW Update Context invalid
      return (DD_FunctionSpecificError | (1 << 16));

    case  NoNewFwToExecute:
      // need to map this to extended status field for FW DSMs
      return DD_FunctionSpecificError;

    case  FwUpdateAlreadyOccurred:
      // Need to map this to extended status field for FW DSMs
      // For DCPMM DSM it is value 02 - FW Update already occurred
      return (DD_FunctionSpecificError | (2 << 16));

    case  InvalidSecurityState:
      return DD_InvalidSecurityState;

    case  SystemTimeNotSet:
    case  DataNotSet:
    case  RevisionFailure:
      // NA - doesn't map to DSM, returning success
      return DD_Success;

    case  InjectionNotEnabled:
      // We need to map this to extended status field for Inject Error Data DSMs extended status field set to 1
      return (DD_FunctionSpecificError | (1 << 16));

    case  ConfigLocked:
    case  Incompatible2lmModuleType:
    case  Unsuccessful:
      return DD_UnknownReason;

    case  InvalidAlignment:
      return DD_InvalidInputParameters;

    case  DeviceNotReady:
      return DD_HwNotReady;

    case  MediaDisabled:
      return DD_HwError;

    case  NoResourcesAvailable:
      return DD_RetrySuggestedOutofResources;

    default:
      return DD_Success;
  }
}

/**
This function returns the status code for ACPI Root DSMs as per the mapping table of
FIS to ACPI Root DSM error code mapping.

@param[in]      Return code of FIS Mailbox

@param[out]     Return code of ACPI Root DSM
**/
UINT32
MapReturnCodeToRD (
  UINT8 FisReturnCode
  )
{
  DEBUG((DEBUG_VERBOSE, "Mapping Return code %d from FIS to DSM \n", FisReturnCode));
  switch(FisReturnCode) {
    case MB_SUCCESS:
      return RD_Success;

    case InvalidCommandParameter:
      return RD_InvalidInputParameters;

    case  DataTransferError:
    case  InternalDeviceError:
      return RD_HwError;

    case  UnsupportedCommand:
      return RD_FunctionNotSupported;

    case DeviceBusy:
    case Aborted:
    case TimeoutOccurred:
    case  NoResourcesAvailable:
      return RD_RetrySuggested;

    case  IncorrectPassphraseOrSecurityNonce:
    case  InvalidSecurityState:
    case  ConfigLocked:
    case  Incompatible2lmModuleType:
    case  Unsuccessful:
      return RD_UnknownReason;

    case  FWAuthenticationFailed:
    case  NoNewFwToExecute:
    case  FwUpdateAlreadyOccurred:
    case  InjectionNotEnabled:
      // No mapping required for the extended status field for ACPI Root DSMs
      return RD_FunctionSpecificError;

    case  SystemTimeNotSet:
    case  DataNotSet:
    case  RevisionFailure:
      // NA - doesn't map to DSM, returning success
      return RD_Success;

    case  InvalidAlignment:
      return RD_InvalidInputParameters;

    case  DeviceNotReady:
    case  MediaDisabled:
      return RD_HwError;

    default:
      return RD_Success;
  }
}

/**
This is a call back function for PMEM DSMs. This will issue the mailbox command
to far memory device if required.

@param[in]      EFI DispatchHandle
@param[in]      current context
@param[in,out]  command buffer
@param[in,out]  command buffer size

@retval EFI_SUCCESS    if callback is successful.
**/
EFI_STATUS
EFIAPI
FisCommandCallback (
  IN EFI_HANDLE                  DispatchHandle,
  IN CONST VOID                  *Context,
  IN OUT VOID                    *CommBuffer,
  IN OUT UINTN                   *CommBufferSize
  )
{
  UINT8           FisFunction;
  EFI_STATUS      Status;
  MAILBOX_CMD     MbCmd;
  UINT32          OutputData[3];
  UINT64          StatusRegister;
  UINT8           DeviceReturnStatus;
  UINT8           OutputPayloadSizeDw;
  UINT8           InputPayloadSizeDw;
  UINT8           PageOffset;
  UINT16          Type;
  UINT8           Flags;
  UINT64          ArsAddr;
  UINT64          ArsLen;
  UINT32          ArsU32Output[7];
  ARS_OUTPUT_DATA *ArsOutputData;
  UINT16          Index;
  UINT8           FwMbOpcode;
  UINT8           FwMbSubOpcode;


  FisFunction = gNfitNvs->FisFuntionNumber;
  DeviceReturnStatus  = 0;
  PageOffset          = 0;
  ArsAddr             = 0;
  ArsLen              = 0;
  Status              = EFI_SUCCESS;

  switch (FisFunction) {

    case 1: //Get SMART and Health Info
      OutputPayloadSizeDw = 32; // 128 Bytes
      Status = GetSmartAndHealthInfoCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, OutputPayloadSizeDw);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 = MapReturnCodeToDD (DeviceReturnStatus);
        gNfitNvs->Data2 = 128; //length
        for (Index = 0; Index < 32; ++Index) {
        }
      } else {
        gNfitNvs->Data1 = DD_UnknownReason;
      }
      break;

    case 2: // Get Smart Threshold
      OutputPayloadSizeDw = 2; // 7 Bytes
      Status = GetSmartThresholdCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, OutputPayloadSizeDw);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 = MapReturnCodeToDD (DeviceReturnStatus);
        gNfitNvs->Data2 = 8; //8 bytes length
        for (Index = 0; Index < 2; ++Index) {
        }
      } else {
        gNfitNvs->Data1 = DD_UnknownReason;
      }
      break;

    case 31: // Get Label Info / _LSI
      OutputPayloadSizeDw = 3; // 12 Bytes
      Status = GetLabelInfoCommand (&DeviceReturnStatus, (UINT32 *)OutputData, OutputPayloadSizeDw, RegionIdPMem);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus);
        gNfitNvs->Data2 = OutputData[1];
        gNfitNvs->Data3 = OutputData[2];
      }
      break;

    case 32: // Get Label Data / _LSR
      gNfitNvs->Buffer[0]= gNfitNvs->Data1;
      gNfitNvs->Buffer[1]= gNfitNvs->Data2;
      Offset = gNfitNvs->Data1;
      TransferLength = (gNfitNvs->Data2); // In bytes
      OutputPayloadSizeDw = (UINT8)(gNfitNvs->Data2/4);
      Status = GetLabelDataCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, OutputPayloadSizeDw, RegionIdPMem);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus);
        for (Index = 0; Index < OutputPayloadSizeDw; ++Index) {
        }
      } else {
        gNfitNvs->Data1 = DD_UnknownReason;
      }
      break;

    case 33: // Set Label Data / _LSW
      for (Index = 0; Index < 0x40; Index++) {
        gNfitNvs->Buffer[Index + 0x40] = gNfitNvs->Buffer[Index];
      }
      gNfitNvs->Buffer[0]= gNfitNvs->Data1;
      gNfitNvs->Buffer[1]= gNfitNvs->Data2;
      for (Index = 0; Index < (gNfitNvs->Data2/4); ++Index) {
      }
      InputPayloadSizeDw = (UINT8)((256 + gNfitNvs->Data2)/4);
      Status = SetLabelDataCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, InputPayloadSizeDw, RegionIdPMem);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus);
      } else {
         gNfitNvs->Data1 = DD_UnknownReason;
       }
      break;

    case 7: // Get Command Effect Log Info
      OutputPayloadSizeDw = 1; // 4 Bytes
      Status = GetCommandEffectInfoCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, OutputPayloadSizeDw);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 = MapReturnCodeToDD (DeviceReturnStatus);;
        gNfitNvs->Data2 = 4; //length in bytes
        CmdEffectOpcodeCnt = (UINT16)*(UINT32 *)gNfitNvs->Buffer; //cmd count
        gNfitNvs->Buffer[0] = CmdEffectOpcodeCnt << 3; //converting to bytes
      } else {
        gNfitNvs->Data1 = DD_UnknownReason;
      }
      break;

    case 8: // Command Effect Log
      if (CmdEffectOpcodeCnt < 64) {
        OutputPayloadSizeDw = (UINT8)CmdEffectOpcodeCnt << 1 ; // Each cmd is 8 bytes (CmdEffectOpcodeCnt*8/4)
        DEBUG ((DEBUG_INFO, "2LM: Issuing GetCommandEffectLogCommand\n"));
        Status = GetCommandEffectLogCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, OutputPayloadSizeDw, PageOffset);
        if (Status == EFI_SUCCESS) {
          gNfitNvs->Data1 = MapReturnCodeToDD (DeviceReturnStatus);
          gNfitNvs->Data2 = OutputPayloadSizeDw << 2; //length
          gNfitNvs->Data3 = CmdEffectOpcodeCnt;
          for (Index = 0; Index < OutputPayloadSizeDw; ++Index) {
          }
        } else {
          gNfitNvs->Data1 = DD_UnknownReason;
        }
      } else {
        OutputPayloadSizeDw = 128; // 512 byes
        if(Status == EFI_SUCCESS && DeviceReturnStatus == MB_SUCCESS) {
          PageOffset = 1; // Only if Cmd Opcode exceeds 64
          Status = GetCommandEffectLogCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer + 124, OutputPayloadSizeDw, PageOffset);
          if (Status == EFI_SUCCESS) {
            gNfitNvs->Data1 = MapReturnCodeToDD (DeviceReturnStatus);
          }
        }
      }
      break;

    case 9: // Pass Through Command
      DEBUG ((DEBUG_INFO, "2LM: In Pass Through Command handler\n"));
      // BIOS don't need to populate command register. It will be sent as part of input payload.
      MbCmd.BiosNonceValue = 0;
      MbCmd.MbCmd0Reg.Data = 0;
      MbCmd.MbCmd1Reg.Data = 0;
      MbCmd.OutputPayloadPtr = 0;
      MbCmd.OutputPayloadSizeDw = 0;
      MbCmd.InputPayloadPtr = 0;
      MbCmd.InputPayloadSizeDw = 0;
      MbCmd.StatusRegister = &StatusRegister;
      MbCmd.DeviceReturnStatus = &DeviceReturnStatus;
      FwMbOpcode = (gNfitNvs->Data1 & 0xFF);
      FwMbSubOpcode = (gNfitNvs->Data1 >> 8 & 0xFF);
      InputPayloadSizeDw = (UINT8)(gNfitNvs->Data2/4);
      DEBUG ((DEBUG_INFO, "2LM: FwMbOpcode %d FwMbSubOpcode%d\n", FwMbOpcode, FwMbSubOpcode));
      DEBUG ((DEBUG_INFO, "2LM: InputPayloadSizeDw %d\n", InputPayloadSizeDw));
      for (Index = 0; Index < InputPayloadSizeDw; ++Index) {
        DEBUG ((DEBUG_VERBOSE, "Input Index %x value %x\n", Index, gNfitNvs->Buffer[Index]));
      }

      MbCmd.MbCmd0Reg.Data = gNfitNvs->Buffer[0];
      MbCmd.MbCmd1Reg.Data = gNfitNvs->Buffer[1];
      DEBUG ((DEBUG_INFO, "2LM: MbCmd0Reg = 0x%x MbCmd1Reg = 0x%x\n", MbCmd.MbCmd0Reg.Data, MbCmd.MbCmd1Reg.Data));

      // Check if opcodes are matching or not
      if (!((FwMbOpcode == MbCmd.MbCmd1Reg.Bits.FwMbOpcode) && (FwMbSubOpcode == MbCmd.MbCmd1Reg.Bits.FwMbSubOpcode))) {
        DEBUG ((DEBUG_INFO, "2LM: Opcode missmatch for Pass Through Command\n"));
        gNfitNvs->Data1 = DD_InvalidInputParameters;
        break;
      }
      DEBUG ((DEBUG_INFO, "2LM: Issuing Pass Through Command\n"));

      // Populate other required parameters
      MbCmd.BiosNonceValue = 0; // Nonce value is not required for this cmd
      MbCmd.OutputPayloadPtr = (UINT32 *)(gNfitNvs->Buffer+2); // reserve first 2DW for status
      MbCmd.OutputPayloadSizeDw = 128; // by default read 512 bytes
      MbCmd.InputPayloadPtr = (UINT32 *)(gNfitNvs->Buffer+2);
      MbCmd.InputPayloadSizeDw = (UINT8)((gNfitNvs->Data2-8)/4);
      MbCmd.DeviceReturnStatus = &DeviceReturnStatus;
      DEBUG ((DEBUG_INFO, "OutputPayloadSizeDw is %d   InputPayloadSizeDw is %d\n", MbCmd.OutputPayloadSizeDw, MbCmd.InputPayloadSizeDw));
      Status = IssueMailboxCommand(MbCmd);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 = DD_Success;
      } else {
        gNfitNvs->Data1 = DD_RetrySuggested;
      }
      // Add 64bit status register to the output payload.
      *(UINT64 *)gNfitNvs->Buffer = *(MbCmd.StatusRegister);
      DEBUG ((DEBUG_VERBOSE, "2LM: Printing output payload\n"));
      for (Index = 0; Index < MbCmd.OutputPayloadSizeDw; ++Index) {
        DEBUG ((DEBUG_VERBOSE, "Output Index %x value %x\n", Index, gNfitNvs->Buffer[Index]));
      }
      break;

    case 10: // Enable Latch System Shutdown Status
      InputPayloadSizeDw = 1; // 1 Byte
      Status = SetLatchSystemShutdownStateCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, InputPayloadSizeDw);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 = MapReturnCodeToDD (DeviceReturnStatus);
      } else {
        gNfitNvs->Data1 = DD_UnknownReason;
      }
      break;

    case 11: // Get Supported Modes
      gNfitNvs->Buffer[0] = SupportedModes;
      gNfitNvs->Data1 = 0;
      break;

    case 12: // Get Firmware Info
      OutputPayloadSizeDw = 22; // 88 Bytes
      Status = GetFirmwareImageInfoCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, OutputPayloadSizeDw);
      gNfitNvs->Buffer[0] = 1572864; //1.5MB
      gNfitNvs->Buffer[1] = 512; //512
      gNfitNvs->Buffer[2] = 1; //us
      gNfitNvs->Buffer[3] = 5;
      gNfitNvs->Buffer[4] = 1;
      gNfitNvs->Buffer[5] = 1;
      gNfitNvs->Buffer[6] = 1;
      gNfitNvs->Buffer[7] = 0;
      gNfitNvs->Buffer[8] = 1;
      gNfitNvs->Buffer[9] = 0;
      gNfitNvs->Buffer[10] = 0;
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 =  MapReturnCodeToDD (DeviceReturnStatus);
        gNfitNvs->Data2 = 40; //bytes
      } else {
        gNfitNvs->Data1 = DD_UnknownReason;
      }
      break;

    case 17: // Set Smart Threshold Command
      InputPayloadSizeDw = 2; // 7 Bytes
      Status = SetSmartThresholdCommand (&DeviceReturnStatus, (UINT32 *)gNfitNvs->Buffer, InputPayloadSizeDw);
      if (Status == EFI_SUCCESS) {
        gNfitNvs->Data1 =  MapReturnCodeToDD (DeviceReturnStatus);
      } else {
        gNfitNvs->Data1 = DD_InvalidInputParameters; // Invalid Input ParametersReturned If any threshold value requested to be enabled is invalid.
      }
      break;

    case 18: // Inject Error Command
      InputPayloadSizeDw = 2; // 7 Bytes
      //TBD
      break;


    case 21: // Query Address Range Scrub (ARS) Capabilities
      // BIOS doesn't need to issue any FIS command. Client 2LM supports ARS for Persistent Memory and not for Volatile memory region
      ArsAddr = *(UINT64*)gNfitNvs->Buffer;
      ArsLen  = *(UINT64*)(gNfitNvs->Buffer + 2);
      // Check if ARS range falls within PMem region
      if ((ArsAddr >= PMemBase) && ((ArsAddr+ArsLen) <= (PMemBase + PMemSize))) {
        gNfitNvs->Data1 = 0x20000; // Status=0 and Ext Status=0x2, indicates scrub of Persistent Memory is supported.
        gNfitNvs->Buffer[0] = 384; // Max Query ARS Status Output Buffer Size is 384 bytes
        gNfitNvs->Buffer[1] = 512; // Clear Uncorrectable Error Range Length Unit Size
        gNfitNvs->Buffer[2] = 0;   // Flags left as 0
      } else {
        gNfitNvs->Data1 = RD_InvalidInputParameters; // Status = Invalid Input parameters and Ext status= 0
      }
      break;

    case 22: // Set Address Range Scrub
      ArsAddr = *(UINT64*)gNfitNvs->Buffer; //[0];
      ArsLen  = *(UINT64*)(gNfitNvs->Buffer+2);//[2];
      Type    = *(UINT16*)(gNfitNvs->Buffer+4);//[4];;
      Flags   = (UINT8)(gNfitNvs->Buffer[4] & 0x0F00);//[4] one byte from offset 18
      if (!((Type == 0x2) && (ArsAddr >= PMemBase) && ((ArsAddr+ArsLen) <= (PMemBase + PMemSize)))) {
        gNfitNvs->Data1 = DD_InvalidInputParameters;
        break;
      }
      InputPayloadSizeDw = 5; // 20 Bytes
      // Populate input buffer required by FIS
      gNfitNvs->Buffer[0] = (RegionIdPMem << 8) | 0x1 ; // byte0: enable: 0x1, byte1: RegionId: for Pmem
      *(UINT64*)(gNfitNvs->Buffer +1) = (ArsAddr - PMemBase); //send the region offset to Willard
      *(UINT64*)(gNfitNvs->Buffer +3) = (ArsAddr+ArsLen- PMemBase);
      Status = SetAddressRangeScrubCommand (&DeviceReturnStatus, (UINT32*)gNfitNvs->Buffer, InputPayloadSizeDw);
      if (EFI_ERROR(Status)) {
        gNfitNvs->Data1 = RD_UnknownReason; //5 : Error, Unknown Reason
        gNfitNvs->Data2 = 0;
        break;
      }
      if ((DeviceReturnStatus != MB_SUCCESS) && (DeviceReturnStatus != DeviceBusy)) {
        gNfitNvs->Data1 =  MapReturnCodeToRD (DeviceReturnStatus); //map FIS return status to Root DSM return status
        gNfitNvs->Data2 = 0;
        break;
      }
      if ((DeviceReturnStatus != MB_SUCCESS) && (DeviceReturnStatus == DeviceBusy)) {
        // If Device status is DeviceBusy then issue long operation and check if ARS is alreday in progress
        DEBUG ((DEBUG_INFO, "2LM: Issue long operation status cmd to see if ARS already in progress\n"));
        OutputPayloadSizeDw = 31; // 124 bytes
        Status = GetLongOperationStatusCommand (&DeviceReturnStatus, (UINT32*)gNfitNvs->Buffer, OutputPayloadSizeDw);
        if (EFI_ERROR(Status)) {
          gNfitNvs->Data1 = RD_UnknownReason; //5  Error  Unknown Reason
          gNfitNvs->Data2 = 0;
          DEBUG ((DEBUG_INFO, "2LM: Error in MB Cmd get long operation status\n"));
          break;
        }
        if (DeviceReturnStatus != MB_SUCCESS) {
            DEBUG ((DEBUG_INFO, "2LM: Error in FIS command get long operation status \n"));
            gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus); //map FIS return status to Root DSM return status
            gNfitNvs->Data2 = 0;
            break;
        }
        //check the opcode for long operation
        DEBUG ((DEBUG_VERBOSE, "2LM: Got the long operartion status for cmd opcode 0x%x\n", *(UINT16*)gNfitNvs->Buffer));
        DEBUG ((DEBUG_VERBOSE, "2LM: Percent Complete %d\n", (gNfitNvs->Buffer[0] & 0xFFFF0000)));
        DEBUG ((DEBUG_VERBOSE, "2LM: Estimated Time to Completion %d\n", gNfitNvs->Buffer[1]));
        if (*(UINT16*)gNfitNvs->Buffer == 0x405) {
          DEBUG ((DEBUG_INFO, "2LM: ARS already in progress \n"));
          gNfitNvs->Data1 = 6; //6  ARS already in progress
          //gNfitNvs->Data1 = 0x020006; // status: 6  Function-Specific Error Code Ext Status 2  No ARS performed for current boot.
          gNfitNvs->Data2 = gNfitNvs->Buffer[1]; //Estimated time
          break;
        }
        gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus); //map FIS return status to Root DSM return status
        gNfitNvs->Data2 = 0;
        break;
      } // End of device busy condition

      // Now at this point MB Status and DeviceReturnStatus both are successful
      // Issue long operation status cmd to get the estimated time for ARS
      DEBUG ((DEBUG_INFO, "2LM: Issue long operation status cmd to get the estimated time for ARS\n"));
      OutputPayloadSizeDw = 128; // 9 bytes
      Status = GetLongOperationStatusCommand (&DeviceReturnStatus, (UINT32*)gNfitNvs->Buffer, OutputPayloadSizeDw);
      if (EFI_ERROR(Status)) {
        gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus); //map FIS return status to Root DSM return status
        gNfitNvs->Data2 = 0;
        DEBUG ((DEBUG_INFO, "2LM: Error in get long operation status\n"));
        break;
      }
      if (DeviceReturnStatus != MB_SUCCESS) {
        DEBUG ((DEBUG_INFO, "2LM: Error in FIS command \n"));
        gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus); //map FIS return status to Root DSM return status
        gNfitNvs->Data2 = 0;
        break;
      }
      DEBUG ((DEBUG_VERBOSE, "2LM: Got the long operartion status for cmd opcode 0x%x\n", *(UINT16*)gNfitNvs->Buffer));
      DEBUG ((DEBUG_VERBOSE, "2LM: Percent Complete %d\n", (gNfitNvs->Buffer[0] & 0xFFFF0000)));
      DEBUG ((DEBUG_VERBOSE, "2LM: Estimated Time to Completion %d\n", gNfitNvs->Buffer[1]));
      gNfitNvs->Data1 = RD_Success;
      gNfitNvs->Data2 = gNfitNvs->Buffer[1];
    break;

    case 23: // Get Address Range Scrub (Query ARS Status)
      //Condition chk -> addresses should be part of PMem address range
      ArsOutputData = (ARS_OUTPUT_DATA *)ArsU32Output;
      OutputPayloadSizeDw = 7; // 28 Bytes
      DEBUG ((DEBUG_INFO, "2LM: Issuing GetAddressRangeScrubCommand\n"));
      Status = GetAddressRangeScrubCommand (&DeviceReturnStatus, (UINT32*) ArsU32Output, OutputPayloadSizeDw);
      if (EFI_ERROR(Status)) {
         gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus); //map FIS return status to Root DSM return status
         gNfitNvs->Data2 = 0;
         break;
       }
       if (DeviceReturnStatus != MB_SUCCESS) {
         DEBUG ((DEBUG_INFO, "2LM: DeviceReturnStatus is %d\n", DeviceReturnStatus));
         if (DeviceReturnStatus == DataNotSet) {
           DEBUG ((DEBUG_INFO, "2LM: ARS Data Not Set\n"));
           gNfitNvs->Data1 = 0x020006; // status: 6  Function-Specific Error Code Ext Status 2  No ARS performed for current boot.
           gNfitNvs->Data2 = 0;
         }

         DEBUG ((DEBUG_INFO, "2LM: Error in FIS command \n"));
         gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus); //map FIS return status to Root DSM return status
         gNfitNvs->Data2 = 0;
         break;
       }
       // Check if ARS is in progress
       //If Enable field is 1, Fill extended status with ARS in progress. Fill zero for rest of the fields
       if (ArsOutputData->Enable) {
         DEBUG ((DEBUG_INFO, "ARS is still in progress for RegionID 0x%x\n",ArsOutputData->RegionId));
         OutputPayloadSizeDw = 31; // 124 bytes for ARS
         ZeroMem ((VOID *)(UINTN)gNfitNvs->Buffer, (OutputPayloadSizeDw*4));
         gNfitNvs->Data1 = 0x10000; // Return Ex status:1  ARS in progress.
         gNfitNvs->Data2 = 0; // Length
         break;
       }
       // ARS is complete
       DEBUG ((DEBUG_INFO, "2LM: ARS is complete \n"));
       DEBUG ((DEBUG_INFO, "DpaStartAddr = 0x%lx\n", ArsOutputData->DpaStartAddr));
       DEBUG ((DEBUG_INFO, "DpaCurrAddr = 0x%lx\n", ArsOutputData->DpaCurrAddr));
       DEBUG ((DEBUG_INFO, "DpaEndAddr = 0x%lx\n", ArsOutputData->DpaEndAddr));

       // Issue long operation status cmd to get the Error status
       DEBUG ((DEBUG_INFO, "2LM: Issue long operation status cmd to get the details for ARS\n"));
       OutputPayloadSizeDw = 31; // 124 bytes for ARS
       UINT8 TempBuff[124]; //temp buff for long op status
       Status = GetLongOperationStatusCommand (&DeviceReturnStatus, (UINT32*)TempBuff, OutputPayloadSizeDw);
       if (EFI_ERROR(Status)) {
         gNfitNvs->Data1 = RD_UnknownReason; //5  Error  Unknown Reason
         gNfitNvs->Data2 = 0;
         DEBUG ((DEBUG_INFO, "2LM: Error in get long operation status\n"));
         break;
       }
       DEBUG ((DEBUG_VERBOSE, "2LM: Got the long operartion status for cmd opcode 0x%x\n", *(UINT16*)TempBuff));
       DEBUG ((DEBUG_VERBOSE, "2LM: Estimated Time to Completion 0x%x\n", *(UINT32*)(TempBuff+4)));
       DEBUG ((DEBUG_VERBOSE, "2LM: Output of long operation\n"));
       for (Index = 0; Index < 124; ++Index) {
          DEBUG ((DEBUG_VERBOSE, "output Index %x value %x\n", Index, TempBuff[Index]));
       }
       // check if long operation was ARS (opcode 5, Sub opcode 4)
       if ((TempBuff[0] == 5) && (TempBuff[1] == 4)) {
          DEBUG ((DEBUG_INFO, "Long operation was ARS\n"));
          // Status Code: The completed mailbox status code of the long operation.
          // Once the long operation is complete, the status code will be located here instead of the status register
          DeviceReturnStatus = TempBuff[8];

          if (DeviceReturnStatus != MB_SUCCESS) {
            DEBUG ((DEBUG_INFO, "2LM: Error in FIS command DeviceReturnStatus 0x%x\n", DeviceReturnStatus));
            gNfitNvs->Data1 = MapReturnCodeToRD (DeviceReturnStatus); //map FIS return status to Root DSM return status
            gNfitNvs->Data2 = 0;
            break;
          }

          UINT8 ErrCnt = TempBuff[9];  //Byte 9
          if (TempBuff[10]) {
            DEBUG ((DEBUG_INFO, "ARS Ended early \n"));
          }
          DEBUG ((DEBUG_INFO, "ARS was done for RegionId 0x%x\n", TempBuff[11] & 0x1F));
          DEBUG ((DEBUG_VERBOSE, "All region indicator bit is 0x%x\n", TempBuff[11] & 0x80));

          DEBUG ((DEBUG_INFO, "ErrCnt = 0x%x\n", ErrCnt));
          DEBUG ((DEBUG_INFO, "DpaStartAddr = 0x%lx\n", ArsOutputData->DpaStartAddr));
          DEBUG ((DEBUG_INFO, "DpaCurrAddr = 0x%lx\n", ArsOutputData->DpaCurrAddr));
          DEBUG ((DEBUG_INFO, "DpaEndAddr = 0x%lx\n", ArsOutputData->DpaEndAddr));

          ArsOutputData->DpaStartAddr = ArsOutputData->DpaStartAddr + PMemBase;
          ArsOutputData->DpaCurrAddr = ArsOutputData->DpaCurrAddr + PMemBase;
          ArsOutputData->DpaEndAddr = ArsOutputData->DpaEndAddr + PMemBase;

          DEBUG ((DEBUG_INFO, "DpaStartAddr = 0x%lx\n", ArsOutputData->DpaStartAddr));
          DEBUG ((DEBUG_INFO, "DpaCurrAddr = 0x%lx\n", ArsOutputData->DpaCurrAddr));
          DEBUG ((DEBUG_INFO, "DpaEndAddr = 0x%lx\n", ArsOutputData->DpaEndAddr));

          // Populating ARS status o/p data
          gNfitNvs->Buffer[0] = 44 + (ErrCnt*24) ; // Size of Output Buffer in bytes, including this field.   44 + Number of error records * 24
          *(UINT64*)(gNfitNvs->Buffer+1) = ArsOutputData->DpaStartAddr; //Start SPA 8 4 In bytes  DPA start address from FIS command
          *(UINT64*)(gNfitNvs->Buffer+3) = ArsOutputData->DpaCurrAddr - ArsOutputData->DpaStartAddr; //DPA Current address  DPA Start address from FIS command
          if (ArsOutputData->DpaCurrAddr != ArsOutputData->DpaEndAddr) {
            *(UINT64*)(gNfitNvs->Buffer+5) = ArsOutputData->DpaCurrAddr;
          } else {
            *(UINT64*)(gNfitNvs->Buffer+5) = 0;
          }
          *(UINT64*)(gNfitNvs->Buffer+7) = ArsOutputData->DpaEndAddr - ArsOutputData->DpaCurrAddr; //Restart ARS Length
          gNfitNvs->Buffer[9] = 0x2;  //Flags = 0, Type = 0x2
          gNfitNvs->Buffer[10] = ErrCnt;  //Num Errors found  from Long operation command
          //populating ARS Error Records
          gNfitNvs->Buffer[11] = 1;     //NFIT Handle
          gNfitNvs->Buffer[12] = 0;     //Reserved
          *(UINT64*)(gNfitNvs->Buffer+13) = *(UINT64*)(TempBuff+12); //DPA error address from Long operation command
          *(UINT64*)(gNfitNvs->Buffer+15) = 512; //bytes

          //Fill in return status
          // Ext Status: 0  ARS complete
          // 1  ARS in progress. Any returned ARS data shall be all zeros.
          // 2  No ARS performed for current boot. Any returned ARS data shall be all zeros.
          // 3  ARS Stopped Prematurely  This may occur when the implementation reaches the maximum number of errors that can be reported.
          // 4 ..0xFFFF Reserved. Any returned ARS Data shall be all zeros.
          if (TempBuff[10] == 0x1) { //Ended early
            gNfitNvs->Data1 = 0x30000; //status:00, Ext Status: 3  ARS Stopped Prematurely
            gNfitNvs->Data2 = 0;
          } else {
            gNfitNvs->Data1 = RD_Success; //status:00, Ext Status:  0  ARS complete
            gNfitNvs->Data2 = 44;
          }
       } else { ////2  No ARS performed for current boot. Any returned ARS data shall be all zeros??
         gNfitNvs->Data1 = RD_UnknownReason; //5  Error  Unknown Reason
         gNfitNvs->Data2 = 0;
       }
       gNfitNvs->Data1 = RD_Success;
       gNfitNvs->Data2 = 44;
      break;

    case 24: // Clear Uncorrectable Error
      /***
       * There is no corresponding FIS command. Bios to write value 0 to the address range
       * specified by input. BIOS should return error if the input range doesn't fall within PMEM range.
       *
       * Status
       * 0  Success
       * 2  If the input range doesn't fall with in PMEM
       * If the input SPA is not 512byte aligned.
       * If the input SPA length is not in the units of 512bytes
       */
      DEBUG ((DEBUG_VERBOSE, "2LM: Inside Clear Uncorrectable Errors\n"));
      // Check if the input range doesn't fall with in PMEM
      ArsAddr = *(UINT64*)gNfitNvs->Buffer; //[0];
      ArsLen  = *(UINT64*)(gNfitNvs->Buffer+2);//[2];
      DEBUG ((DEBUG_VERBOSE, "ARSA = 0x%lx  ARSL = 0x%lx\n", ArsAddr,ArsLen));
      // Check if ARS range falls within PMem region
      if ((ArsAddr >= PMemBase) && ((ArsAddr+ArsLen) <= (PMemBase + PMemSize))) {
        DEBUG ((DEBUG_INFO, "2LM:  Clear Uncorrectable Error is within PMem range\n"));
        gNfitNvs->Data1 = RD_Success;
      } else {
        gNfitNvs->Data1 = DD_InvalidInputParameters; //2  Invalid Input Parameters
      }
      break;
  }
  return EFI_SUCCESS;
}

/**
  Find the operation region in NVDIMM ACPI table by given Name and Size,
  and initialize it if the region is found.

  @param[in, out] Table          The NVDIMM item in ACPI table.
  @param[in]      Name           The name string to find in TPM table.
  @param[in]      Size           The size of the region to find.

  @return                        The allocated address for the found region.

**/
VOID *
AssignOpRegion (
  EFI_ACPI_DESCRIPTION_HEADER    *Table,
  UINT32                         Name,
  UINT16                         Size
  )
{
  EFI_STATUS                     Status;
  AML_OP_REGION_32_8             *OpRegion;
  EFI_PHYSICAL_ADDRESS           MemoryAddress;

  MemoryAddress = SIZE_4GB - 1;

  //
  // Patch some pointers for the ASL code before loading the SSDT.
  //
  for (OpRegion  = (AML_OP_REGION_32_8 *) (Table + 1);
       OpRegion <= (AML_OP_REGION_32_8 *) ((UINT8 *) Table + Table->Length);
       OpRegion  = (AML_OP_REGION_32_8 *) ((UINT8 *) OpRegion + 1)) {
    if ((OpRegion->OpRegionOp  == AML_EXT_REGION_OP) &&
        (OpRegion->NameString  == Name) &&
        (OpRegion->DWordPrefix == AML_DWORD_PREFIX) &&
        (OpRegion->BytePrefix  == AML_WORD_PREFIX)) {

      Status = gBS->AllocatePages(AllocateMaxAddress, EfiACPIMemoryNVS, EFI_SIZE_TO_PAGES (Size), &MemoryAddress);
      ASSERT_EFI_ERROR (Status);
      ZeroMem ((VOID *)(UINTN)MemoryAddress, Size);
      OpRegion->RegionOffset = (UINT32) (UINTN) MemoryAddress;
      OpRegion->RegionLen    = (UINT8) Size;
      break;
    }
  }

  return (VOID *) (UINTN) MemoryAddress;
}

/**
  The driver's entry point.

  It install callbacks for NVDIMM FIS commands

  @param[in] ImageHandle  The firmware allocated handle for the EFI image.
  @param[in] SystemTable  A pointer to the EFI System Table.

  @retval EFI_SUCCESS     The entry point is executed successfully.
  @retval Others          Some error occurs when executing this entry point.

**/
EFI_STATUS
EFIAPI
InitializeNvdimmSmm (
  IN EFI_HANDLE                  ImageHandle,
  IN EFI_SYSTEM_TABLE            *SystemTable
  )
{
  EFI_STATUS                     Status;
  EFI_SMM_SW_DISPATCH2_PROTOCOL  *SwDispatch;
  EFI_SMM_SW_REGISTER_CONTEXT    SwContext;
  EFI_HANDLE                     SwHandle;
  EFI_ACPI_DESCRIPTION_HEADER    *Table;
  UINTN                          TableSize;
  EFI_ACPI_TABLE_PROTOCOL        *AcpiTable;
  UINTN                          TableKey;
  TWOLM_INFO_HOB                 *TwoLmInfoHob;

  if (IsTwoLmEnabled ()== FALSE) {
    return EFI_UNSUPPORTED;
  }

  DEBUG((DEBUG_INFO, "InitializeNvdimmSmm\n"));
  TwoLmInfoHob = (TWOLM_INFO_HOB *) GetFirstGuidHob (&gTwoLmInfoHobGuid);
  if (TwoLmInfoHob == NULL) {
    DEBUG ((DEBUG_INFO, "2LM: TwoLmInfoHob not found\n"));
    return EFI_UNSUPPORTED;
  }

  if (TwoLmInfoHob->TwoLmInfo.PMemSize == 0) {
    DEBUG ((DEBUG_VERBOSE, "NVDIMM ACPI table installation for PMEM size Zero\n"));
  }
  DEBUG((DEBUG_VERBOSE, "PMemModeSupported %d MemoryModeSupported %d \n", TwoLmInfoHob->TwoLmInfo.PMemModeSupported,TwoLmInfoHob->TwoLmInfo.MemoryModeSupported));
  SupportedModes |= (TwoLmInfoHob->TwoLmInfo.PMemModeSupported << 1) |
                    (TwoLmInfoHob->TwoLmInfo.MemoryModeSupported);
  DEBUG((DEBUG_VERBOSE, "SupportedModes 0x%x \n", SupportedModes));

  PMemBase = TwoLmInfoHob->TwoLmInfo.PMemBase;
  PMemSize = LShiftU64(TwoLmInfoHob->TwoLmInfo.PMemSize, 20); // In bytes
  RegionIdPMem = TwoLmInfoHob->TwoLmInfo.RegionIdPMem;
  DEBUG((DEBUG_VERBOSE, "PMemBase = 0x%lx \t PMemSize 0x%lx RegionIdPMem = %x\n", PMemBase, PMemSize, RegionIdPMem));

  Status = GetSectionFromFv (
             &gEfiCallerIdGuid,
             EFI_SECTION_RAW,
             0,
             (VOID **) &Table,
             &TableSize
             );
  ASSERT_EFI_ERROR (Status);
  gNfitNvs = AssignOpRegion (Table, SIGNATURE_32 ('N', 'V', 'D', 'M'), (UINT16) sizeof (NVDIMM_NVS));
  DEBUG((DEBUG_VERBOSE, "gNfitNvs Address  %x\n", gNfitNvs));
  ASSERT (gNfitNvs != NULL);

  //
  // Get the Sw dispatch protocol and register SMI callback functions.
  //
  Status = gSmst->SmmLocateProtocol (&gEfiSmmSwDispatch2ProtocolGuid, NULL, (VOID**)&SwDispatch);
  ASSERT_EFI_ERROR (Status);
  SwContext.SwSmiInputValue = (UINTN) -1;
  Status = SwDispatch->Register (SwDispatch, FisCommandCallback, &SwContext, &SwHandle);
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return Status;
  }
  gNfitNvs->FisSwSmiNumber = (UINT8) SwContext.SwSmiInputValue;
  DEBUG((DEBUG_VERBOSE, "Nfit SMM SW SMI number = %x\n", gNfitNvs->FisSwSmiNumber));

  //
  // Publish the NVDIMM ACPI table. Table is re-checksumed.
  //
  Status = gBS->LocateProtocol (&gEfiAcpiTableProtocolGuid, NULL, (VOID **) &AcpiTable);
  ASSERT_EFI_ERROR (Status);

  TableKey = 0;
  Status = AcpiTable->InstallAcpiTable (
                        AcpiTable,
                        Table,
                        TableSize,
                        &TableKey
                        );
  ASSERT_EFI_ERROR (Status);

  return EFI_SUCCESS;
}
