/** @file
  VR post initializations.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019-2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/

#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/CpuMailboxLib.h>
#include <Library/ConfigBlockLib.h>
#include <Ppi/SiPolicy.h>
#include <CpuAccess.h>
#include <Library/PciSegmentLib.h>
#include <Register/Cpuid.h>
#include <Library/CpuPolicyLib.h>
#include <Library/HobLib.h>
#include <Library/VoltageRegulatorDomains.h>
#include <Library/VoltageRegulatorCommands.h>
#include <Library/PeiVrLib.h>
#include <CpuDataHob.h>
#include <Library/PeiVrDomainLib.h>
#include <Library/CpuCommonLib.h>

/**
  Print the Vr Config block.

  @param[IN] CpuPowerMgmtVrConfig  - CPU Power Management VR Config Block
**/
VOID
PrintVrOverrides (
  IN CPU_POWER_MGMT_VR_CONFIG *CpuPowerMgmtVrConfig
  )
{
  UINT16      MaxNumVrs;
  UINT32      Index = 0;

  ///
  /// Define Maximum Number of Voltage Regulator Domains.
  ///
  MaxNumVrs = GetMaxNumVrs ();

  DEBUG ((DEBUG_INFO, "------------------ CPU Power Mgmt VR Config ------------------\n"));

  for (Index = 0; Index < MaxNumVrs; Index++) {
    DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : VrConfigEnable[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->VrConfigEnable[Index]));
    if (CpuPowerMgmtVrConfig->VrConfigEnable[Index] == 1) {
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : TdcCurrentLimit[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->TdcCurrentLimit[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : AcLoadline[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->AcLoadline[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : DcLoadline[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->DcLoadline[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : Psi1Threshold[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->Psi1Threshold[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : Psi2Threshold[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->Psi2Threshold[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : Psi3Threshold[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->Psi3Threshold[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : Psi3Enable[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->Psi3Enable[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : Psi4Enable[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->Psi4Enable[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : ImonSlope[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->ImonSlope[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : ImonOffset[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->ImonOffset[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : IccMax[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->IccMax[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : VrVoltageLimit[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->VrVoltageLimit[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : TdcEnable[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->TdcEnable[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : TdcTimeWindow[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->TdcTimeWindow1[Index]));
      DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : TdcLock[%X] : 0x%X\n", Index, CpuPowerMgmtVrConfig->TdcLock[Index]));
    }
    DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG:: FastPkgCRampDisable for Index = %d : 0x%X\n", Index, CpuPowerMgmtVrConfig->FastPkgCRampDisable[Index]));
    DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG:: SlowSlewRate for Index = %d : 0x%X\n", Index, CpuPowerMgmtVrConfig->SlowSlewRate[Index]));
  }
  DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : PsysSlope : 0x%X\n", CpuPowerMgmtVrConfig->PsysSlope));
  DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : PsysOffset : 0x%X\n", CpuPowerMgmtVrConfig->PsysOffset));
  DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : AcousticNoiseMitigation : 0x%X\n", CpuPowerMgmtVrConfig->AcousticNoiseMitigation));
  DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG:: SendVrMbxCmd : 0x%X\n", CpuPowerMgmtVrConfig->SendVrMbxCmd));
  DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : FivrRfiFrequency : 0x%X\n", CpuPowerMgmtVrConfig->FivrRfiFrequency));
  DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : FivrSpreadSpectrum : 0x%X\n", CpuPowerMgmtVrConfig->FivrSpreadSpectrum));
  DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : EnableMinVoltageOverride : 0x%X\n", CpuPowerMgmtVrConfig->EnableMinVoltageOverride));
  if (CpuPowerMgmtVrConfig->EnableMinVoltageOverride == 1) {
    DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : MinVoltageRuntime : 0x%X\n", CpuPowerMgmtVrConfig->MinVoltageRuntime));
    DEBUG ((DEBUG_INFO, " CPU_POWER_MGMT_VR_CONFIG : MinVoltageC8 : 0x%X\n", CpuPowerMgmtVrConfig->MinVoltageC8));
  }
}

/**
  Programs the VR parameters for the external VR's which support SVID communication.

  @param[IN OUT] SiPolicyPpi      The SI Policy PPI instance
  @param[in]  SelectedCtdpLevel   Ctdp Level Selected in BIOS
**/
VOID
ConfigureSvidVrs (
  IN OUT SI_POLICY_PPI *SiPolicyPpi,
  IN UINT16            SelectedCtdpLevel
  )
{
  EFI_STATUS                  Status;
  UINT32                      MailboxData;
  UINT32                      MailboxCmd;
  UINT32                      MailboxStatus;
  UINT32                      MailboxType;
  UINT64                      TempAcLoadline;
  UINT64                      TempDcLoadline;
  UINT8                       TempVrAddress;
  UINT8                       SvidEnabled;
  UINT8                       ConvertedTimeWindow;
  UINT16                      ConvertedPsysOffset;
  UINTN                       VrIndex;
  CPU_POWER_MGMT_VR_CONFIG    *CpuPowerMgmtVrConfig;
  CPU_TEST_CONFIG             *CpuTestConfig;
  CPU_POWER_MGMT_PSYS_CONFIG  *CpuPowerMgmtPsysConfig;
  VR_DOMAIN_TOPOLOGY          VrDomainTopology;
  CPU_DATA_HOB                CpuDataHob;
  VOID                        *Hob;
  BOOLEAN                     IsTimeWindowInSeconds;

  DEBUG ((DEBUG_INFO, "VR: Configure SVID Vr's\n"));

  ///
  /// Get RC Policy settings through the SiPolicy PPI
  ///
  Status = PeiServicesLocatePpi (
             &gSiPolicyPpiGuid,
             0,
             NULL,
             (VOID **)&SiPolicyPpi
             );
  if (Status != EFI_SUCCESS) {
    //
    // SI_POLICY_PPI must be installed at this point
    //
    ASSERT (FALSE);
    return;
  }

  Status = GetConfigBlock((VOID *) SiPolicyPpi, &gCpuPowerMgmtVrConfigGuid, (VOID *)&CpuPowerMgmtVrConfig);
  if (CpuPowerMgmtVrConfig == NULL) {
    ASSERT_EFI_ERROR (Status);
    return;
  }

  Status = GetConfigBlock((VOID *) SiPolicyPpi, &gCpuTestConfigGuid, (VOID *)&CpuTestConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock((VOID *) SiPolicyPpi, &gCpuPowerMgmtPsysConfigGuid, (VOID *)&CpuPowerMgmtPsysConfig);
  ASSERT_EFI_ERROR (Status);

  DEBUG ((DEBUG_INFO, "VR: Update VR overrides\n"));
  UpdateVrOverrides (CpuPowerMgmtVrConfig, SelectedCtdpLevel);
  PrintVrOverrides (CpuPowerMgmtVrConfig);

  ///
  /// Send command for MPS IMPV8 VR if requested
  ///
  if (CpuPowerMgmtVrConfig->SendVrMbxCmd == MPS_VR_CMD_REQUESTED) {
    MailboxType = MAILBOX_TYPE_PCODE;
    MailboxCmd  = WRITE_MPS_VR_IMPV8_CMD;
    MailboxData = 0x1;
    DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_MPS_VR_IMPV8_CMD\n"));
    Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
    if (EFI_ERROR(Status) || (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS)) {
      DEBUG ((DEBUG_ERROR, "VR: Error writing MPS VR IMPV8 command. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
    }
  }

  ///
  /// Send command for PS4 exit failures on specific VRs if requested
  ///
  if (CpuPowerMgmtVrConfig->SendVrMbxCmd == PS4_EXIT_VR_CMD_REQUESTED) {
    MailboxType = MAILBOX_TYPE_PCODE;
    MailboxCmd  = WRITE_PS4_EXIT_VR_CMD;
    MailboxData = 0x0;
    DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_PS4_EXIT_VR_CMD\n"));
    Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
    if (EFI_ERROR(Status) || (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS)) {
      DEBUG ((DEBUG_ERROR, "VR: Error writing PS4 Exit VR command. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
    }
  }

  ///
  /// CPU VR MSR mailbox
  ///
  MailboxType = MAILBOX_TYPE_VR_MSR;

  ///
  ///  Configure Platform Level controls
  ///  PSYS Config
  ///
  /// -PsysOffset is defined as S16.7.8 fixed point
  /// -PsysSlope is defined as U16.1.15 fixed point
  /// -Policy Psys offset is defined in 1/1000 increments
  /// -Policy Psys slope is defined in 1/100 increments
  /// -Mailbox ImonOffset = (PlatPolicyPsysOffset * 2^8)/1000
  /// -Mailbox PsysSlope = (PlatPolicyPsysSlope * 2^15)/100
  /// -Adding half of divisor to dividend to account for rounding errors in fixed point arithmetic.
  ///
  if (CpuPowerMgmtVrConfig->PsysOffset1 != 0 || CpuPowerMgmtVrConfig->PsysSlope != 0 ) {
    ConvertedPsysOffset = (UINT16) (((CpuPowerMgmtVrConfig->PsysOffset1 * (1 << 8) + 500) / 1000) & VR_PSYS_OFFSET_MASK);
    MailboxData =  (UINT32) (ConvertedPsysOffset |
                   (((CpuPowerMgmtVrConfig->PsysSlope * (1 << 15) + 50)/100) << VR_PSYS_SLOPE_OFFSET));
    MailboxCmd = WRITE_PSYS_CONFIG_CMD;
    DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_PSYS_CONFIG_CMD\n"));
    DEBUG ((DEBUG_INFO, "(MAILBOX) PsysOffset            = %d (1/1000)\n", CpuPowerMgmtVrConfig->PsysOffset1));
    DEBUG ((DEBUG_INFO, "(MAILBOX) PsysSlope             = %d (1/100)\n", CpuPowerMgmtVrConfig->PsysSlope));
    DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Data    = %d\n", MailboxData));
    Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
    if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
      DEBUG ((DEBUG_ERROR, "VR: Error Writing PSYS Config. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
    }
  }

  ///
  /// PSYS Pmax
  /// -PMax is defined as U16.10.6 fixed point
  /// -Policy Pmax is defined in 1/8 W increments
  /// -Mailbox Pmax = (PlatPolicyPmax * 2^6)/8
  ///
  if (CpuPowerMgmtPsysConfig->PsysPmax != 0) {
    MailboxData =  (UINT32)((CpuPowerMgmtPsysConfig->PsysPmax * (1<<6))/8);
    MailboxCmd = WRITE_PSYS_PMAX_CMD;
    DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_PSYS_PMAX_CMD\n"));
    DEBUG ((DEBUG_INFO, "(MAILBOX) PsysPmax              = %d (1/8 Watt)\n", CpuPowerMgmtPsysConfig->PsysPmax));
    DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Data    = %d\n", MailboxData));
    Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
    if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
      DEBUG ((DEBUG_ERROR, "VR: Error Writing Psys PMAX. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
    }
  }

  ///
  /// Support Min Voltage override command. Min Voltage Runtime Data[7:0] and Min Voltage for C8 Data[15:8].
  /// - Mailbox Voltage limit defined in U8.1.7 volts in fixed point. Range 4120 to 1.999V
  /// - Policy defined in mV, Range 0 - 1999mV.
  ///
  if (CpuPowerMgmtVrConfig->EnableMinVoltageOverride == 1) {
    MailboxData =  (UINT32) ((CpuPowerMgmtVrConfig->MinVoltageRuntime) * ((1<<7) /1000)        ///< Min Voltage Runtime Data[7:0]
                            | (((CpuPowerMgmtVrConfig->MinVoltageC8) * ((1<<7) /1000)) << 8)); ///< Min Voltage C8 Data[15:8]
    MailboxCmd = WRITE_VCCIN_MIN_VOLTAGE;
    DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_VCCIN_MIN_VOLTAGE\n"));
    DEBUG ((DEBUG_INFO, "(MAILBOX) MinVoltageRuntime     = %d mV\n",CpuPowerMgmtVrConfig->MinVoltageRuntime));
    DEBUG ((DEBUG_INFO, "(MAILBOX) MinVoltageC8          = %d mV\n", CpuPowerMgmtVrConfig->MinVoltageC8));
    DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Data    = %d\n", MailboxData));
    Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
    if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
      DEBUG ((DEBUG_ERROR, "VR: Error Writing Min Voltage Override Command. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
    }
  }

  ///
  /// Get CPU VR topology domains which are fru dependent
  ///
  GetCpuVrTopology (&VrDomainTopology);

  ///
  /// Initial cpu data into one hob.
  ///
  ZeroMem (&CpuDataHob, sizeof (CPU_DATA_HOB));

  for (VrIndex = VrDomainTopology.MinVrIndex; VrIndex < VrDomainTopology.MaxVrIndex; VrIndex++) {
    TempVrAddress = VrDomainTopology.VrAddress[VrIndex];
    SvidEnabled = VrDomainTopology.SvidEnabled[VrIndex];

    DEBUG ((DEBUG_INFO, " VR Address for Vr Index %x = 0x%x\n", VrIndex, VrDomainTopology.VrAddress[VrIndex]));
    DEBUG ((DEBUG_INFO, " VR SvidEnabled for Vr Index %x = 0x%x\n", VrIndex, VrDomainTopology.SvidEnabled[VrIndex]));

    ///
    /// Update Vr Address for CPU Data HOB
    ///
    CpuDataHob.VrAddress[VrIndex] = VrDomainTopology.VrAddress[VrIndex];
    CpuDataHob.SvidEnabled = CpuDataHob.SvidEnabled + (VrDomainTopology.SvidEnabled[VrIndex] << VrIndex);

    if (CpuPowerMgmtVrConfig->VrConfigEnable[VrIndex] == 1 && SvidEnabled == 1) {
      ///
      /// AC/DC Loadline
      ///
      if (CpuPowerMgmtVrConfig->AcLoadline[VrIndex] != 0 || CpuPowerMgmtVrConfig->DcLoadline[VrIndex] != 0) {
        ///
        ///  Check max AC/DC loadline boundary. Max allowed is 6249 (62.49 mOhm)
        ///
        if (CpuPowerMgmtVrConfig->AcLoadline[VrIndex] > AC_DC_LOADLINE_MAX){
          CpuPowerMgmtVrConfig->AcLoadline[VrIndex] = AC_DC_LOADLINE_MAX;
        } else if (CpuPowerMgmtVrConfig->DcLoadline[VrIndex] > AC_DC_LOADLINE_MAX){
          CpuPowerMgmtVrConfig->DcLoadline[VrIndex] = AC_DC_LOADLINE_MAX;
        }

        ///
        ///  Loadline is 1/100 mOhm units. Mailbox interface requires Loadline in U-4.20 Ohms format.
        ///  After multiplying by 2^20, we must divide the result by 100,000 to convert to Ohms.
        ///  Adding half of divisor to dividend to account for rounding errors in fixed point arithmetic.
        ///
        TempAcLoadline = MultU64x64(CpuPowerMgmtVrConfig->AcLoadline[VrIndex], LShiftU64 (1, 20));
        TempAcLoadline = DivU64x32(TempAcLoadline + 50000, 100000);
        TempDcLoadline = MultU64x64(CpuPowerMgmtVrConfig->DcLoadline[VrIndex], LShiftU64 (1, 20));
        TempDcLoadline = DivU64x32(TempDcLoadline + 50000, 100000);

        MailboxData = (UINT32) (TempAcLoadline | (LShiftU64 (TempDcLoadline, DC_LOADLINE_OFFSET)));
        MailboxCmd = WRITE_ACDC_LOADLINE_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
        DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_ACDC_LOADLINE_CMD\n"));
        DEBUG ((DEBUG_INFO, "(MAILBOX) AcLoadline            = %d (1/100 mOhms)\n",CpuPowerMgmtVrConfig->AcLoadline[VrIndex]));
        DEBUG ((DEBUG_INFO, "(MAILBOX) DcLoadline            = %d (1/100 mOhms)\n",CpuPowerMgmtVrConfig->DcLoadline[VrIndex]));
        DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Data    = %d\n", MailboxData));
        Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
        if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
          DEBUG ((DEBUG_ERROR, "VR: Error Writing AC/DC Loadline. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
        }
      }

      ///
      /// PS Cutoff Current
      ///
      MailboxData =  (UINT32)(CpuPowerMgmtVrConfig->Psi1Threshold[VrIndex] & PSI_THRESHOLD_MASK) |
                     ((CpuPowerMgmtVrConfig->Psi2Threshold[VrIndex] & PSI_THRESHOLD_MASK) << PSI2_THRESHOLD_OFFSET) |
                     ((CpuPowerMgmtVrConfig->Psi3Threshold[VrIndex] & PSI_THRESHOLD_MASK) << PSI3_THRESHOLD_OFFSET) |
                     ((~CpuPowerMgmtVrConfig->Psi3Enable[VrIndex] & BIT0) << PSI3_ENABLE_OFFSET) |
                     ((~CpuPowerMgmtVrConfig->Psi4Enable[VrIndex] & BIT0) << PSI4_ENABLE_OFFSET);
      MailboxCmd = WRITE_PSI_CUTOFF_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
      DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_PSI_CUTOFF_CMD\n"));
      DEBUG ((DEBUG_INFO, "(MAILBOX) Psi1Threshold         = %d (1/4 Amp)\n",CpuPowerMgmtVrConfig->Psi1Threshold[VrIndex]));
      DEBUG ((DEBUG_INFO, "(MAILBOX) Psi2Threshold         = %d (1/4 Amp)\n",CpuPowerMgmtVrConfig->Psi2Threshold[VrIndex]));
      DEBUG ((DEBUG_INFO, "(MAILBOX) Psi3Threshold         = %d (1/4 Amp)\n",CpuPowerMgmtVrConfig->Psi3Threshold[VrIndex]));
      DEBUG ((DEBUG_INFO, "(MAILBOX) Psi3Enable            = %d, Psi4Enable            = %d\n",CpuPowerMgmtVrConfig->Psi3Enable[VrIndex], CpuPowerMgmtVrConfig->Psi4Enable[VrIndex]));
      DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Data    = %d\n", MailboxData));
      Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Writing PS Cutoff Current. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
      }

      /// -ImonOffset is defined as S16.7.8 fixed point
      /// -ImonSlope is defined as U16.1.15 fixed point
      /// -Policy Imon offset is defined in 1/1000 increments
      /// -Policy Imon slope is defined in 1/100 increments
      /// -Mailbox ImonOffset = (PlatPolicyImonOffset * 2^8)/1000
      /// -Mailbox ImonSlope = (PlatPolicyImonSlope * 2^15)/100
      /// -Adding half of divisor to dividend to account for rounding errors in fixed point arithmetic.
      ///
      MailboxData =  (UINT32)(((CpuPowerMgmtVrConfig->ImonOffset[VrIndex] * (1 << 8) + 500) / 1000))
                             | (((CpuPowerMgmtVrConfig->ImonSlope[VrIndex] * (1 << 15) + 50) / 100) << VR_IMON_SLOPE_OFFSET);
      MailboxCmd = WRITE_IMON_CONFIG_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
      DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_IMON_CONFIG_CMD\n"));
      DEBUG ((DEBUG_INFO, "(MAILBOX) ImonOffset            = %d (1/1000)\n",CpuPowerMgmtVrConfig->ImonOffset[VrIndex]));
      DEBUG ((DEBUG_INFO, "(MAILBOX) ImonSlope             = %d (1/100)\n",CpuPowerMgmtVrConfig->ImonSlope[VrIndex]));
      DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Data    = %d\n", MailboxData));
      Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Writing IMON Config. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
      }

      ///
      /// Icc Max
      ///
      if (CpuPowerMgmtVrConfig->IccMax[VrIndex] != 0) {
        MailboxData =  (UINT32)CpuPowerMgmtVrConfig->IccMax[VrIndex];
        MailboxCmd = WRITE_VR_ICC_MAX_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
        DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_VR_ICC_MAX_CMD\n"));
        DEBUG ((DEBUG_INFO, "(MAILBOX) IccMax                = %d (1/4 A)\n",CpuPowerMgmtVrConfig->IccMax[VrIndex]));
        Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
        if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
          DEBUG ((DEBUG_ERROR, "VR: Error Writing IccMax. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
        }
      }

      ///
      /// VR Voltage Limit
      /// -Mailbox Voltage Limit defined as U16.3.13, Range 0-7.999V
      /// -Policy defined in mV, Range 0-7999mV
      /// -Adding half of divisor to dividend to account for rounding errors in fixed point arithmetic.
      ///
      if (CpuPowerMgmtVrConfig->VrVoltageLimit[VrIndex] != 0) {
        MailboxData =  (UINT32)((CpuPowerMgmtVrConfig->VrVoltageLimit[VrIndex] * (1 << 13) + 500) / 1000) & VR_VOLTAGE_LIMIT_MASK;
        MailboxCmd = WRITE_VR_VOLTAGE_LIMIT_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
        DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_VR_VOLTAGE_LIMIT_CMD\n"));
        DEBUG ((DEBUG_INFO, "(MAILBOX) VrVoltageLimit        = %d mV\n",CpuPowerMgmtVrConfig->VrVoltageLimit[VrIndex]));
        DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Data    = %d\n", MailboxData));
        Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
        if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
          DEBUG ((DEBUG_ERROR, "VR: Error Writing VR Voltage Limit. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
        }
      }

      ///
      /// VR TDC Settings
      /// -Mailbox TDC Current Limit defined as U15.12.3, Range 0-4095A
      ///    -Policy defined in 1/8 A increments
      /// -Time window mailbox format, in binary, is xxyyyyy, time window = (1+X/4)*pow(2,Y)
      ///    -Set to 1ms default (mailbox value = 0000000b)
      ///

      ///
      /// Ensure Time Window Value is within the supported range.
      ///
      if (CpuPowerMgmtVrConfig->TdcTimeWindow[VrIndex] <= VR_TDC_TIME_WINDOW_MAX) {
        IsTimeWindowInSeconds = (CpuPowerMgmtVrConfig->TdcTimeWindow1[VrIndex] / 1000) >= 1;
        if (IsTimeWindowInSeconds) {
          ConvertedTimeWindow = GetConvertedTime ((CpuPowerMgmtVrConfig->TdcTimeWindow1[VrIndex] / 1000), SecondsTimeWindowConvert);
        } else {
          ConvertedTimeWindow = GetConvertedTime (CpuPowerMgmtVrConfig->TdcTimeWindow1[VrIndex], MilliSecondsTimeWindowConvert);
        }
      } else {
        ConvertedTimeWindow = 0;
      }

      MailboxData =  (UINT32) ((CpuPowerMgmtVrConfig->TdcCurrentLimit[VrIndex] & VR_TDC_CURRENT_LIMIT_MASK) |
                              ((CpuPowerMgmtVrConfig->TdcEnable[VrIndex] & BIT0) << VR_TDC_ENABLE_OFFSET) |
                              ((ConvertedTimeWindow & VR_TDC_TIME_WINDOW_MASK) << VR_TDC_TIME_WINDOW_OFFSET) |
                              ((CpuPowerMgmtVrConfig->TdcLock[VrIndex] & BIT0) << VR_TDC_LOCK_OFFSET));

     if (CpuPowerMgmtVrConfig->Irms[VrIndex] == 1) {
       DEBUG ((DEBUG_INFO, "(MAILBOX) Irms Enabled\n"));
       MailboxData |= (UINT32) ((CpuPowerMgmtVrConfig->Irms[VrIndex]) << VR_TDC_IRMS_OFFSET);
     }
      MailboxCmd = WRITE_VR_TDC_CONFIG_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_TDC_ADDRESS_OFFSET);

      DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_VR_TDC_CONFIG_CMD\n"));
      DEBUG ((DEBUG_INFO, "(MAILBOX) TdcPowerLimit         = %d (1/8A)\n",CpuPowerMgmtVrConfig->TdcCurrentLimit[VrIndex]));
      Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Writing VR TDC Config. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
      }
    }

    ///
    /// Acoustic Noise Mitigation
    ///
    if (CpuPowerMgmtVrConfig->AcousticNoiseMitigation) {
      ///
      /// Set Fast and Slow Slew Rate for Deep Package C States.
      ///
      MailboxCmd  = WRITE_SVID_SET_DISABLE_FAST_PKGC_RAMP_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
      MailboxData = CpuPowerMgmtVrConfig->FastPkgCRampDisable[VrIndex];
      DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_SVID_SET_DISABLE_FAST_PKGC_RAMP_CMD\n"));
      Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Writing disable Fast Deep Package C States for VrIndex = %x. EFI_STATUS = %r, Mailbox Status = %X\n", VrIndex , Status, MailboxStatus));
      }

      MailboxCmd  = WRITE_SVID_SET_VR_SLEW_RATE_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
      MailboxData = CpuPowerMgmtVrConfig->SlowSlewRate[VrIndex];
      DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Write Command = WRITE_SVID_SET_VR_SLEW_RATE_CMD\n"));
      DEBUG((DEBUG_INFO, "(MAILBOX) SlowSlewRate[%x] = %d (0: Fast/2</b>; 1: Fast/4; 2: Fast/8; 3: Fast/16)\n", VrIndex, CpuPowerMgmtVrConfig->SlowSlewRate[VrIndex]));
      Status = MailboxWrite (MailboxType, MailboxCmd, MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Writing Slow Slew Rate for VrIndex = %x. EFI_STATUS = %r, Mailbox Status = %X\n", VrIndex , Status, MailboxStatus));
      }
    }
  }
  //
  //  To control settled time of Richtek VR which is enabled for VccSA to meet IMVP8 VR settled time.
  //
  VrSettledTimerRt (VrIndex);

  ///
  /// Update Min and Max value for CPU Data HOB
  ///
  CpuDataHob.MaxVrIndex            = VrDomainTopology.MaxVrIndex;
  CpuDataHob.MinVrIndex            = VrDomainTopology.MinVrIndex;

  DEBUG ((DEBUG_INFO, " VR SvidEnabled = 0x%x\n", CpuDataHob.SvidEnabled));
  ///
  /// Build CPU Data HOB
  ///

  Hob = BuildGuidDataHob (
          &gCpuDataHobGuid,
          &CpuDataHob,
          (UINTN) sizeof (CPU_DATA_HOB)
          );
  ASSERT (Hob != NULL);

  ///
  /// Print VR Topology data
  ///
  DEBUG ((DEBUG_INFO, "VR: VR Topology data = 0x%x\n", VrDomainTopology));
  DEBUG ((DEBUG_INFO, "    VR Type 0 = SVID, VR Type 1 = non-SVID\n"));

  ///
  /// RFI Mitigation
  ///
  UpdateFivrRfiSettings (CpuPowerMgmtVrConfig);
}
