/** @file
  PCH PCIe Bus Device Function Library.
  All functions from this library are available in PEI, DXE, and SMM,
  But do not support UEFI RUNTIME environment call.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Base.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/PchInfoLib.h>
#include <Library/PchPcieRpLib.h>
#include <Register/PchRegs.h>
#include <PchBdfAssignment.h>

/**
  Check if a Device is present for PCH FRU
  If the data is defined for PCH RFU return it
  If the data is not defined (Device is NOT present) assert.

  @param[in]  DataToCheck       Device or Function number to check

  @retval Device or Function number value if defined for PCH FRU
          0xFF if not present in PCH FRU
**/
UINT8
CheckAndReturn (
  UINT8 DataToCheck
  )
{
  if (DataToCheck == NOT_PRESENT) {
    ASSERT (FALSE);
  }
  return DataToCheck;
}

/**
  Get CNVI controller PCIe Device Number

  @retval CNVI controller PCIe Device Number
**/
UINT8
CnviDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_CNVI_WIFI);
}

/**
  Get CNVI controller PCIe Function Number

  @retval CNVI controller PCIe Function Number
**/
UINT8
CnviFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_CNVI_WIFI);
}

/**
  Get CNVI controller address that can be passed to the PCI Segment Library functions.

  @retval CNVI controller address in PCI Segment Library representation
**/
UINT64
CnviPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            CnviDevNumber (),
            CnviFuncNumber (),
            0
            );
}

/**
  Get eSPI controller address that can be passed to the PCI Segment Library functions.

  @retval eSPI controller address in PCI Segment Library representation
**/
UINT64
EspiPciCfgBase (
  VOID
  )
{
  ASSERT (PCI_DEVICE_NUMBER_PCH_ESPI != NOT_PRESENT);

  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           PCI_DEVICE_NUMBER_PCH_ESPI,
           PCI_FUNCTION_NUMBER_PCH_ESPI,
           0
           );
}

/**
  Returns Gigabit Ethernet PCI Device Number

  @retval  GbE device number
**/
UINT8
GbeDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_GBE);
}

/**
  Returns Gigabit Ethernet PCI Function Number

  @retval  GbE function number
**/
UINT8
GbeFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_GBE);
}

/**
  Get GbE controller address that can be passed to the PCI Segment Library functions.

  @retval GbE controller address in PCI Segment Library representation
**/
UINT64
GbePciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            GbeDevNumber (),
            GbeFuncNumber (),
            0
            );
}

/**
  Get HDA PCI device number

  @retval PCI dev number
**/
UINT8
HdaDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_HDA);
}

/**
  Get HDA PCI function number

  @retval PCI fun number
**/
UINT8
HdaFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_HDA);
}

/**
  Get HDA controller address that can be passed to the PCI Segment Library functions.

  @retval HDA controller address in PCI Segment Library representation
**/
UINT64
HdaPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            HdaDevNumber (),
            HdaFuncNumber (),
            0
            );
}

/**
  Get IEH PCI device number

  @retval PCI dev number
**/
UINT8
IehDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_IEH);
}

/**
  Get IEH PCI function number

  @retval PCI fun number
**/
UINT8
IehFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_IEH);
}

/**
  Get IEH controller address that can be passed to the PCI Segment Library functions.

  @retval IEH controller address in PCI Segment Library representation
**/
UINT64
IehPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            IehDevNumber (),
            IehFuncNumber (),
            0
            );
}

/**
  Get P2SB PCI device number

  @retval PCI dev number
**/
UINT8
P2sbDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_P2SB);
}

/**
  Get P2SB PCI function number

  @retval PCI fun number
**/
UINT8
P2sbFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_P2SB);
}

/**
  Get P2SB controller address that can be passed to the PCI Segment Library functions.

  @retval P2SB controller address in PCI Segment Library representation
**/
UINT64
P2sbPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           P2sbDevNumber (),
           P2sbFuncNumber (),
           0
           );
}

/**
  Get ISH Device Number

  @retval  ISH Device Number
**/
UINT8
IshDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_ISH);
}

/**
  Get ISH Function Number

  @retval  ISH Function Number
**/
UINT8
IshFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_ISH);
}

/**
  Get ISH controller address that can be passed to the PCI Segment Library functions.

  @retval ISH controller address in PCI Segment Library representation
**/
UINT64
IshPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            IshDevNumber (),
            IshFuncNumber (),
            0
            );
}

/**
  Get PCH Trace Hub PCI device number

  @retval PCI dev number
**/
UINT8
PchTraceHubDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_TRACE_HUB);
}

/**
  Get PCH Trace Hub PCI function number

  @retval PCI fun number
**/
UINT8
PchTraceHubFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_TRACE_HUB);
}

/**
  Get PCH Trace Hub PCI config space base address

  @retval PCI config space base address
**/
UINT64
PchTraceHubPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
          DEFAULT_PCI_SEGMENT_NUMBER_PCH,
          DEFAULT_PCI_BUS_NUMBER_PCH,
          PchTraceHubDevNumber (),
          PchTraceHubFuncNumber (),
          0
          );
}

/**
  Get PMC PCI device number

  @retval PCI dev number
**/
UINT8
PmcDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PMC);
}

/**
  Get PMC PCI function number

  @retval PCI fun number
**/
UINT8
PmcFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_PMC);
}

/**
  Get PMC controller address that can be passed to the PCI Segment Library functions.

  @retval PMC controller address in PCI Segment Library representation
**/
UINT64
PmcPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            PmcDevNumber (),
            PmcFuncNumber (),
            0
            );
}

/**
  Get PMC SSRAM controller address that can be passed to the PCI Segment Library functions.

  @retval PMC SSRAM controller address in PCI Segment Library representation
**/
UINT64
PmcSsramPciCfgBase (
  VOID
  )
{
  ASSERT (PCI_DEVICE_NUMBER_PCH_PMC_SSRAM != NOT_PRESENT);

  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            PCI_DEVICE_NUMBER_PCH_PMC_SSRAM,
            PCI_FUNCTION_NUMBER_PCH_PMC_SSRAM,
            0
            );
}

/**
  Get Tsn PCI device number

  @retval PCI dev number
**/
UINT8
TsnDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_TSN);
}

/**
  Get Tsn PCI function number

  @retval PCI fun number
**/
UINT8
TsnFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_TSN);
}

/**
  Get TSN controller address that can be passed to the PCI Segment Library functions.

  @retval TSN controller address in PCI Segment Library representation
**/
UINT64
TsnPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            TsnDevNumber (),
            TsnFuncNumber (),
            0
            );
}

/**
  Returns PCH SPI Device number

  @retval UINT8   PCH SPI Device number
**/
UINT8
SpiDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SPI);
}

/**
  Returns PCH SPI Function number

  @retval UINT8   PCH SPI Function number
**/
UINT8
SpiFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SPI);
}

/**
  Returns PCH SPI PCI Config Space base address

  @retval  UINT64  PCH SPI Config Space base address
**/
UINT64
SpiPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           SpiDevNumber (),
           SpiFuncNumber (),
           0
           );
}

/**
  Get XHCI controller PCIe Device Number

  @retval XHCI controller PCIe Device Number
**/
UINT8
PchXhciDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_XHCI);
}

/**
  Get XHCI controller PCIe Function Number

  @retval XHCI controller PCIe Function Number
**/
UINT8
PchXhciFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_XHCI);
}

/**
  Get XHCI controller address that can be passed to the PCI Segment Library functions.

  @retval XHCI controller address in PCI Segment Library representation
**/
UINT64
PchXhciPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
          DEFAULT_PCI_SEGMENT_NUMBER_PCH,
          DEFAULT_PCI_BUS_NUMBER_PCH,
          PchXhciDevNumber (),
          PchXhciFuncNumber (),
          0
          );
}

/**
  Get XDCI controller PCIe Device Number

  @retval XDCI controller PCIe Device Number
**/
UINT8
PchXdciDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_XDCI);
}

/**
  Get XDCI controller PCIe Function Number

  @retval XDCI controller PCIe Function Number
**/
UINT8
PchXdciFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_XDCI);
}

/**
  Get XDCI controller address that can be passed to the PCI Segment Library functions.

  @retval XDCI controller address in PCI Segment Library representation
**/
UINT64
PchXdciPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
          DEFAULT_PCI_SEGMENT_NUMBER_PCH,
          DEFAULT_PCI_BUS_NUMBER_PCH,
          PchXdciDevNumber (),
          PchXdciFuncNumber (),
          0
          );
}

/**
  Return Smbus Device Number

  @retval Smbus Device Number
**/
UINT8
SmbusDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SMBUS);
}

/**
  Return Smbus Function Number

  @retval  Smbus Function Number
**/
UINT8
SmbusFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SMBUS);
}

/**
  Get SMBUS controller address that can be passed to the PCI Segment Library functions.

  @retval SMBUS controller address in PCI Segment Library representation
**/
UINT64
SmbusPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            SmbusDevNumber (),
            SmbusFuncNumber (),
            0
            );
}

/**
  Return DMA Smbus Device Number

  @retval DMA Smbus Device Number
**/
UINT8
SmbusDmaDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_DMA_SMBUS);
}

/**
  Return DMA Smbus Function Number

  @retval  DMA Smbus Function Number
**/
UINT8
SmbusDmaFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_DMA_SMBUS);
}

/**
  Get DMA SMBUS controller address that can be passed to the PCI Segment Library functions.

  @retval DMA SMBUS controller address in PCI Segment Library representation
**/
UINT64
SmbusDmaPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            SmbusDmaDevNumber (),
            SmbusDmaFuncNumber (),
            0
            );
}

/**
  Get SD Card controller PCIe Device Number

  @retval SD Card controller PCIe Device Number
**/
UINT8
ScsSdCardDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SCS_SDCARD);
}

/**
  Get SD Card controller PCIe Function Number

  @retval SD Card controller PCIe Function Number
**/
UINT8
ScsSdCardFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SCS_SDCARD);
}

/**
  Get SCS SD Card controller address that can be passed to the PCI Segment Library functions.

  @retval SCS SD Card controller address in PCI Segment Library representation
**/
UINT64
ScsSdCardPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
          DEFAULT_PCI_SEGMENT_NUMBER_PCH,
          DEFAULT_PCI_BUS_NUMBER_PCH,
          ScsSdCardDevNumber (),
          ScsSdCardFuncNumber (),
          0
          );
}

/**
  Get EMMC controller PCIe Device Number

  @retval EMMC controller PCIe Device Number
**/
UINT8
ScsEmmcDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SCS_EMMC);
}

/**
  Get EMMC controller PCIe Function Number

  @retval EMMC controller PCIe Function Number
**/
UINT8
ScsEmmcFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SCS_EMMC);
}

/**
  Get SCS EMMC controller address that can be passed to the PCI Segment Library functions.

  @retval SCS EMMC controller address in PCI Segment Library representation
**/
UINT64
ScsEmmcPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
          DEFAULT_PCI_SEGMENT_NUMBER_PCH,
          DEFAULT_PCI_BUS_NUMBER_PCH,
          ScsEmmcDevNumber (),
          ScsEmmcFuncNumber (),
          0
          );
}

/**
  Get UFS controller PCIe Device Number

  @param[in]  UfsIndex     Index of the UFS controller

  @retval UFS controller PCIe Device Number
**/
UINT8
ScsUfsDevNumber (
  IN  UINT8  UfsIndex
  )
{
  switch (UfsIndex) {
    case 0:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SCS_UFS0);
    case 1:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SCS_UFS1);
    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get UFS controller PCIe Function Number

  @param[in]  UfsIndex     Index of the UFS controller

  @retval UFS controller PCIe Function Number
**/
UINT8
ScsUfsFuncNumber (
  IN  UINT8  UfsIndex
  )
{
  switch (UfsIndex) {
    case 0:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SCS_UFS0);
    case 1:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SCS_UFS1);
    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get SCS EMMC controller address that can be passed to the PCI Segment Library functions.

  @param[in]  UfsIndex     Index of the UFS controller

  @retval SCS EMMC controller address in PCI Segment Library representation
**/
UINT64
ScsUfsPciCfgBase (
  IN  UINT8  UfsIndex
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            ScsUfsDevNumber (UfsIndex),
            ScsUfsFuncNumber (UfsIndex),
            0
            );
}

/**
  Get SATA controller PCIe Device Number

  @param[in]  SataCtrlIndex       SATA controller index

  @retval SATA controller PCIe Device Number
**/
UINT8
SataDevNumber (
  IN UINT32  SataCtrlIndex
  )
{
  ASSERT (SataCtrlIndex < MAX_SATA_CONTROLLER);

  if (SataCtrlIndex == 0) {
    return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SATA_1);
  } else if (SataCtrlIndex == 1) {
    return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SATA_2);
  } else if (SataCtrlIndex == 2) {
    return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SATA_3);
  } else {
    ASSERT (FALSE);
    return 0xFF;
  }
}

/**
  Get SATA controller PCIe Function Number

  @param[in]  SataCtrlIndex       SATA controller index

  @retval SATA controller PCIe Function Number
**/
UINT8
SataFuncNumber (
  IN UINT32  SataCtrlIndex
  )
{
  ASSERT (SataCtrlIndex < MAX_SATA_CONTROLLER);

  if (SataCtrlIndex == 0) {
    return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SATA_1);
  } else if (SataCtrlIndex == 1) {
    return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SATA_2);
  } else if (SataCtrlIndex == 2) {
    return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SATA_3);
  } else {
    ASSERT (FALSE);
    return 0xFF;
  }
}

/**
  Get SATA controller address that can be passed to the PCI Segment Library functions.

  @param[in]  SataCtrlIndex       SATA controller index

  @retval SATA controller address in PCI Segment Library representation
**/
UINT64
SataPciCfgBase (
  IN UINT32  SataCtrlIndex
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           SataDevNumber (SataCtrlIndex),
           SataFuncNumber (SataCtrlIndex),
           0
           );
}

/**
  Get LPC controller PCIe Device Number

  @retval LPC controller PCIe Device Number
**/
UINT8
LpcDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_LPC);
}

/**
  Get LPC controller PCIe Function Number

  @retval LPC controller PCIe Function Number
**/
UINT8
LpcFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_LPC);
}

/**
  Returns PCH LPC device PCI base address.

  @retval                   PCH LPC PCI base address.
**/
UINT64
LpcPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           LpcDevNumber (),
           LpcFuncNumber (),
           0
           );
}

/**
  Get Thermal Device PCIe Device Number

  @retval Thermal Device PCIe Device Number
**/
UINT8
ThermalDevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_THERMAL);
}

/**
  Get Thermal Device PCIe Function Number

  @retval Thermal Device PCIe Function Number
**/
UINT8
ThermalFuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_THERMAL);
}

/**
  Returns Thermal Device PCI base address.

  @retval                   Thermal Device PCI base address.
**/
UINT64
ThermalPciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           ThermalDevNumber (),
           ThermalFuncNumber (),
           0
           );
}

/**
  Get Serial IO I2C controller PCIe Device Number

  @param[in]  I2cNumber       Serial IO I2C controller index

  @retval Serial IO I2C controller PCIe Device Number
**/
UINT8
SerialIoI2cDevNumber (
  IN UINT8       I2cNumber
  )
{
  if (GetPchMaxSerialIoI2cControllersNum () <= I2cNumber) {
    ASSERT (FALSE);
    return 0xFF;
  }
  switch (I2cNumber) {
    case 0:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C0);
    case 1:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C1);
    case 2:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C2);
    case 3:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C3);
    case 4:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C4);
    case 5:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C5);
    case 6:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C6);
    case 7:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C7);
    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get Serial IO I2C controller PCIe Function Number

  @param[in]  I2cNumber       Serial IO I2C controller index

  @retval Serial IO I2C controller PCIe Function Number
**/
UINT8
SerialIoI2cFuncNumber (
  IN UINT8       I2cNumber
  )
{
  if (GetPchMaxSerialIoI2cControllersNum () <= I2cNumber) {
    ASSERT (FALSE);
    return 0xFF;
  }
  switch (I2cNumber) {
    case 0:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C0);
    case 1:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C1);
    case 2:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C2);
    case 3:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C3);
    case 4:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C4);
    case 5:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C5);
    case 6:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C6);
    case 7:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C7);
    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get Serial IO I2C controller address that can be passed to the PCI Segment Library functions.

  @param[in]  I2cNumber       Serial IO I2C controller index

  @retval Serial IO I2C controller address in PCI Segment Library representation
**/
UINT64
SerialIoI2cPciCfgBase (
  IN UINT8        I2cNumber
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           SerialIoI2cDevNumber (I2cNumber),
           SerialIoI2cFuncNumber (I2cNumber),
           0
           );
}

/**
  Get Serial IO SPI controller PCIe Device Number

  @param[in]  I2cNumber       Serial IO SPI controller index

  @retval Serial IO SPI controller PCIe Device Number
**/
UINT8
SerialIoSpiDevNumber (
  IN UINT8       SpiNumber
  )
{
  if (GetPchMaxSerialIoSpiControllersNum () <= SpiNumber) {
    ASSERT (FALSE);
    return 0xFF;
  }
  if (IsPchN ()) {
    switch (SpiNumber) {
      case 0:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_N_SERIAL_IO_SPI0);
      case 1:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_N_SERIAL_IO_SPI1);
      case 2:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_N_SERIAL_IO_SPI2);
      case 3:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_N_SERIAL_IO_SPI3);
      case 4:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_N_SERIAL_IO_SPI4);
      case 5:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_N_SERIAL_IO_SPI5);
      case 6:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_N_SERIAL_IO_SPI6);
      default:
        ASSERT (FALSE);
        return 0xFF;
    }
  }
  if (IsPchH ()) {
    switch (SpiNumber) {
      case 0:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI0);
      case 1:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI1);
      case 2:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI2);
      case 3:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI3);
      case 4:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI4);
      case 5:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_H_SERIAL_IO_SPI5);
      case 6:
        return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_H_SERIAL_IO_SPI6);
      default:
        ASSERT (FALSE);
        return 0xFF;
    }
  }
  switch (SpiNumber) {
    case 0:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI0);
    case 1:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI1);
    case 2:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI2);
    case 3:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI3);
    case 4:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI4);
    case 5:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI5);
    case 6:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI6);
    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get Serial IO SPI controller PCIe Function Number

  @param[in]  SpiNumber       Serial IO SPI controller index

  @retval Serial IO SPI controller PCIe Function Number
**/
UINT8
SerialIoSpiFuncNumber (
  IN UINT8       SpiNumber
  )
{
  if (GetPchMaxSerialIoSpiControllersNum () <= SpiNumber) {
    ASSERT (FALSE);
    return 0xFF;
  }
  if (IsPchN ()) {
    switch (SpiNumber) {
      case 0:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_N_SERIAL_IO_SPI0);
      case 1:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_N_SERIAL_IO_SPI1);
      case 2:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_N_SERIAL_IO_SPI2);
      case 3:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_N_SERIAL_IO_SPI3);
      case 4:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_N_SERIAL_IO_SPI4);
      case 5:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_N_SERIAL_IO_SPI5);
      case 6:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_N_SERIAL_IO_SPI6);
      default:
        ASSERT (FALSE);
        return 0xFF;
    }
  }
  if (IsPchH ()) {
    switch (SpiNumber) {
      case 0:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI0);
      case 1:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI1);
      case 2:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI2);
      case 3:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI3);
      case 4:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI4);
      case 5:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_H_SERIAL_IO_SPI5);
      case 6:
        return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_H_SERIAL_IO_SPI6);
      default:
        ASSERT (FALSE);
        return 0xFF;
    }
  }
  switch (SpiNumber) {
    case 0:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI0);
    case 1:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI1);
    case 2:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI2);
    case 3:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI3);
    case 4:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI4);
    case 5:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI5);
    case 6:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI6);
    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get Serial IO SPI controller address that can be passed to the PCI Segment Library functions.

  @param[in]  SpiNumber       Serial IO SPI controller index

  @retval Serial IO SPI controller address in PCI Segment Library representation
**/
UINT64
SerialIoSpiPciCfgBase (
  IN UINT8        SpiNumber
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           SerialIoSpiDevNumber (SpiNumber),
           SerialIoSpiFuncNumber (SpiNumber),
           0
           );
}

/**
  Get Serial IO UART controller PCIe Device Number

  @param[in]  UartNumber       Serial IO UART controller index

  @retval Serial IO UART controller PCIe Device Number
**/
UINT8
SerialIoUartDevNumber (
  IN UINT8       UartNumber
  )
{
  if (GetPchMaxSerialIoUartControllersNum () <= UartNumber) {
    ASSERT (FALSE);
    return 0xFF;
  }
  switch (UartNumber) {
    case 0:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART0);
    case 1:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART1);
    case 2:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART2);
    case 3:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART3);
    case 4:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART4);
    case 5:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART5);
    case 6:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART6);
    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get Serial IO UART controller PCIe Function Number

  @param[in]  UartNumber       Serial IO UART controller index

  @retval Serial IO UART controller PCIe Function Number
**/
UINT8
SerialIoUartFuncNumber (
  IN UINT8       UartNumber
  )
{
  if (GetPchMaxSerialIoUartControllersNum () <= UartNumber) {
    ASSERT (FALSE);
    return 0xFF;
  }
  switch (UartNumber) {
    case 0:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART0);
    case 1:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART1);
    case 2:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART2);
    case 3:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART3);
    case 4:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART4);
    case 5:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART5);
    case 6:
      return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART6);
    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get Serial IO UART controller address that can be passed to the PCI Segment Library functions.

  @param[in]  UartNumber       Serial IO UART controller index

  @retval Serial IO UART controller address in PCI Segment Library representation
**/
UINT64
SerialIoUartPciCfgBase (
  IN UINT8        UartNumber
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
           DEFAULT_PCI_SEGMENT_NUMBER_PCH,
           DEFAULT_PCI_BUS_NUMBER_PCH,
           SerialIoUartDevNumber (UartNumber),
           SerialIoUartFuncNumber (UartNumber),
           0
           );
}

/**
  Get PCH PCIe controller PCIe Device Number

  @param[in]  RpIndex       Root port physical number. (0-based)

  @retval PCH PCIe controller PCIe Device Number
**/
UINT8
PchPcieRpDevNumber (
  IN  UINTN   RpIndex
  )
{
  if (RpIndex >= GetPchMaxPciePortNum ()) {
    ASSERT (FALSE);
    return 0xFF;
  }
  switch (RpIndex) {
    case 0:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_1);
    case 1:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_2);
    case 2:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_3);
    case 3:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_4);
    case 4:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_5);
    case 5:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_6);
    case 6:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_7);
    case 7:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_8);
    case 8:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_9);
    case 9:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_10);
    case 10:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_11);
    case 11:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_12);
    case 12:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_13);
    case 13:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_14);
    case 14:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_15);
    case 15:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_16);
    case 16:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_17);
    case 17:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_18);
    case 18:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_19);
    case 19:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_20);
    case 20:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_21);
    case 21:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_22);
    case 22:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_23);
    case 23:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_24);
    case 24:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_25);
    case 25:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_26);
    case 26:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_27);
    case 27:
      return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_28);

    default:
      ASSERT (FALSE);
      return 0xFF;
  }
}

/**
  Get PCH PCIe controller PCIe Function Number
  Note:
  For Client PCH generations Function Number can be various
  depending on "Root Port Function Swapping". For such cases
  Function Number  MUST be obtain from proper register.
  For Server PCHs we have no "Root Port Function Swapping"
  and we can return fixed Function Number.
  To address this difference in this, PCH generation independent,
  library we should call specific function in PchPcieRpLib.

  @param[in]  RpIndex       Root port physical number. (0-based)

  @retval PCH PCIe controller PCIe Function Number
**/
UINT8
PchPcieRpFuncNumber (
  IN  UINTN   RpIndex
  )
{
  UINTN   Device;
  UINTN   Function;

  GetPchPcieRpDevFun (RpIndex, &Device, &Function);

  return (UINT8)Function;
}

/**
  Get PCH PCIe controller address that can be passed to the PCI Segment Library functions.

  @param[in]  RpIndex       PCH PCIe Root Port physical number. (0-based)

  @retval PCH PCIe controller address in PCI Segment Library representation
**/
UINT64
PchPcieRpPciCfgBase (
  IN  UINT32   RpIndex
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
          DEFAULT_PCI_SEGMENT_NUMBER_PCH,
          DEFAULT_PCI_BUS_NUMBER_PCH,
          PchPcieRpDevNumber (RpIndex),
          PchPcieRpFuncNumber (RpIndex),
          0
          );
}

/**
  Get HECI1 PCI device number

  @retval PCI dev number
**/
UINT8
PchHeci1DevNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_DEVICE_NUMBER_PCH_HECI1);
}

/**
  Get HECI1 PCI function number

  @retval PCI fun number
**/
UINT8
PchHeci1FuncNumber (
  VOID
  )
{
  return CheckAndReturn (PCI_FUNCTION_NUMBER_PCH_HECI1);
}

/**
  Get HECI1 controller address that can be passed to the PCI Segment Library functions.

  @retval HECI1 controller address in PCI Segment Library representation
**/
UINT64
PchHeci1PciCfgBase (
  VOID
  )
{
  return PCI_SEGMENT_LIB_ADDRESS (
            DEFAULT_PCI_SEGMENT_NUMBER_PCH,
            DEFAULT_PCI_BUS_NUMBER_PCH,
            PchHeci1DevNumber (),
            PchHeci1FuncNumber (),
            0
            );
}
