###############################################################
# TPM 2.0 Sample AUX Deletion Script w/Algorithm Agility
# Copyright Intel Corporation 2016
# Last update: Jan 04, 2017
#   Jan 2017: Allow AUX and PS to have different name algorithms
###############################################################
# NOTE: Deleting the AUX index requires writing to the PS Index to set the
# AuxDeletionEnable bit and then invoking the ACM's ResetAux function
###########################################################################
# Usage:                                                                  #
#     Tpm2DeleteAuxAA.nsh  <PS alg>   <PS fileNamePrefix>  <AUX alg> [-O] #
#  Where:                                                                 #
#    <PS alg> is the PS Name Algorithm                                    #
#       Such as SHA1, SHA256, SHA384, SHA512, SM3                         #
#       <PS alg> will be substituted for %1 in Tpm2ProvTool command lines #
#    <fileNamePrefix> is the file name prefix you used to create the      #
#       PS def file - typically a project name or platform name           #
#      (e.g., MyFirst; Example; Purley)                                   #
#       <fileNamePrefix> will be substituted for %2 in command lines      #
#       For example %2Ps%1.iDef --> MyFirstPsSha256.iDef                  #
#            %2PsFinalOR%1.pDef --> MyFirstPsFinalORSha256.pdef           #
#    <AUX alg> is the AUX Name Algorithm  (substituted for %3)            #
#    "-o" is an optional parameter, when specified, deletes original      #
#       AUX index handle(0x01800003) instead of the new (0x01C10102)      #
###########################################################################

echo -OFF 
# Copyrights:
# You may copy, distribute, and make derivatives of this script under the 
# following provisions:
#  You must not remove or modify the Intel copyright notice nor these requirements
#  The script must only be used to provision a TPM for use with Intel Technologies
#    For any other use you must get written permission from Intel Corporation
 
##########################################################################
# This script will:
# Check if the AUX Index exists 
# If so:
#   Modify the PS Index to enable AUX Deletion
#   Delete the AUX PS NV Index if it exists
#   Modify the PS Index to disable AUX Deletion
#
##########################################################################



##########################################################################
# Here is a list of files that are needed
## //User Modified Files -----------[example for %1=SHA256, %2=MyFirst]
## %2SecretOR%1.pDef                  MyFirstSecretOrSha256.pDef
## %2PsFinalOR%1.pDef                 MyFirstPsFinalORSha256.pdef
## %2Ps%1.iDef                        MyFirstPsSha256.iDef
## Intel Provided Files -----------------------------------------------------
## Aux%1.iDef                         AuxSha256.iDef
## Aux-o%1.iDef                       Aux-oSha256.iDef
## AuxPolicyOR%1.pDef                 AuxPolicyORSha256.pDef
## EmptyAuthPwSession.sDef
## getsec64.efi    -- part of Intel TXT Toolkit  
## Locality3&4.pDef
## Policy%1.sDef                      PolicySha256.sDef
## PsDelAux.iDef
## PsDelAux-o.iDef
## UndefinespaceSpecial.pDef
##########################################################################
# This script echoes progress markers to the screen and also to
#   a log file (*.log) which also captures provisioning details 
##########################################################################

#Check that Alg parameter is present
if ZZ%1  == ZZ   then
	echo Algorithm parameter missing
	echo Syntax: %0 <alg> <PlatformName>
	echo   Examples: 
	echo     %0 SHA256 Example 
	echo     %0 SHA384 MyFirst
	echo     %0 SHA512 Grantley
	echo     %0 SHA256 Old -o
	goto EOF
endif
# Check if PS alg parameter supported
if NOT EXIST Policy%1.sDef then 
	echo PS Algorithm not supported - Policy%1.sDef not found
	goto EOF
endif
# Check if platform parameter supported
if NOT EXIST %2Ps%1.iDef then 
	echo %2Ps%1.iDef not found
	goto EOF
endif
# Check if AUXS alg parameter and Param4 are correct
if NOT EXIST Aux%4%3.iDef then  
	echo Aux%4%3.iDef not found
	goto EOF
endif

# Now copy this file to the error log
#echo TPM2 AUX Deletion Script %1 %2 %3 %4 > Tpm2DeleteAux.log
#if EXIST %0     then
#	type %0     >> Tpm2DeleteAux.log
#endif
#if EXIST %0.nsh then 
#	type %0.nsh >> Tpm2DeleteAux.log
#endif

##########################################################################

:START
echo ****** Deleting Aux Index *****
# Uses Session 0 for AuthPolicy
# Uses Session 1 for Empty AuthValue
# Note that session 0 uses "Don't Continue" so no need to flush on successful use

echo **** Checking if AUX index exists
echo **** Checking if AUX index exists >> Tpm2DeleteAux.log
Tpm2ProvTool NvReadPublic Aux%4%3.iDef >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo Aux Index does not exist - nothing to delete
	goto DONE
endif

# Start a PW Session to use for PhAuth
echo **** Starting PW Session for PlatformAuth and IndexAuth
echo **** Starting PW Session for PlatformAuth and IndexAuth >> Tpm2DeleteAux.log
Tpm2ProvTool StartSession EmptyAuthPwSession.sDef 1 >> Tpm2DeleteAux.log

# This script requires that PlatformAuth be EMPTY in order to delete NV Indexes
# Lets check if PlatformAuth is EMPTY by attemping to change it from EMPTY to EMPTY
Tpm2ProvTool HierarchyChangeAuth PH EmptyAuth.dat 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo ERROR: PlatformAuth is not EMPTY (or Platfrom Hierarchy not enabled)
	echo Try running ResetPlatformAuth.nsh %1 %2
	goto ERROR
endif


echo Deleting the AUX Index
echo Rewriting PS Data to set Aux-Delete-Enable

:PSREWRITE1
echo *** Start Policy Session to Write PS
echo *** Start Policy Session to write PS >> Tpm2DeleteAux.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2DeleteAux.log
# Satisfy Branch 2
echo **** Policy Branch 2 
echo **** Policy Branch 2 >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2DeleteAux.log

echo *** Writing PS Data to set AuxDeltionEnable bit
echo *** Writing NV Data to set AuxDeltionEnable bit >> Tpm2DeleteAux.log
Tpm2ProvTool NvWrite PsDelAux%4.iDef 0 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	goto ERROR		# Didn't satisfy PS Index authPolicy
endif

##########################################################################
:AUXDELETE
#echo ***** Deleting AUX Index *****
echo "ServerGetSec -l ent -fn res"
getsec64_1.3.2.RC1.efi -l ent -fn res >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo Unable to delete Aux index
	goto ERROR		# Didn't delete AUX Index 
endif

##########################################################################
:PSREWRITE2
echo *** Start Policy Session to re-write PS
echo *** Start Policy Session to re-write PS >> Tpm2DeleteAux.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2DeleteAux.log
# Satisfy Branch 2
echo **** Policy Branch 2 
echo **** Policy Branch 2 >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2DeleteAux.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2DeleteAux.log

echo *** Writing PS NV Data (resetting AuxDeletionEnable bit)
echo *** Writing PS NV Data (resetting AuxDeletionEnable bit) >> Tpm2DeleteAux.log
Tpm2ProvTool NvWrite %2Ps%1.iDef 0 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	Unable to rewrite PS Index
	goto ERROR		# Didn't write PS Index 
endif

# Need to check to make sure AUX Deletion was really successful
echo **** Checking if AUX index was deleted
echo **** Checking if AUX index was deleted >> Tpm2TxtProv.log
Tpm2ProvTool NvReadPublic Aux%4%3.iDef >> Tpm2TxtProv.log
if %lasterror% == 0 then
	echo Aux Index was not deleted
	goto ERROR		# AUX Index still exists
endif

:DONE
echo  ***************************************************
echo  ***********   AUX Deleted Successfully  ***********
echo  ***************************************************

goto EOF

:ERROR
echo  ***************************************************
echo  *************** ERROR Deleting AUX  ***************
echo  ***************************************************
Tpm2ProvTool FlushSession 0 >> Tpm2DeleteAux.log

goto EOF

:EOF

