##########################################################################
# TPM 2.0 Sample PPI Provisioning Script 
# Copyright Intel Corporation 2015
# Last update: Aug 30, 2017
#
#
##########################################################################
##########################################################################
# Usage:                                                                 #
#     Tpm2PpiProvDev.nsh <alg> <fileNamePrefix> [-D]                     #
#  Where:                                                                #
#      <alg> is a supported hash algorithm                              #
#         Such as SHA1, SHA256, SHA384, SHA512, SM3                      #
#         <alg> will be substituted for %1 in Tpm2ProvTool command lines #
#      <fileNamePrefix> is the file name prefix you used to create the   #
#         def files - typically a project name or platform name          #
#        (e.g., MyFirst; Example; Purley)                                #
#         <fileNamePrefix> will be substituted for %2 in command lines   #
#       For example %2Ppi%1.iDef --> ExamplePpiSha256.iDef               #
#                   %2Ppi%1.iDef --> MyFirstPpiSha384.iDef               #
#                   %2Ppi%1.iDef --> OldPpiSha384.iDef                   #
#     "-D" option tells the script to delete the NV Index                #
##########################################################################

echo -OFF 
# Copyrights:
# You may copy, distribute, and make derivatives of this script under the 
# following provisions:
#  You must not remove or modify the Intel copyright notice nor these requirements
#  The script must only be used to provision a TPM for use with Intel Technologies
#    For any other use you must get written permission from Intel corporation
 
##########################################################################
# This script will:
# Delete the index if the -D option is specified, otherwise it will
# Check if the PPI NV Index already exists
# - if the index does not exist, creates the NV Index 
# - If it does exist, script checks the index's attributes 
#   -- if wrong attributes - deletes the index, recreates it 
##########################################################################
# Here is a list of files that are needed
## //User Modified Files -----------[example for %1=SHA256, %2=MyFirst]
## %2SecretOR%1.pDef                  MyFirstSecretOrSha256.pDef
## %2Ppi%1.iDef                       MyFirstPpiSha256.iDef
## Intel Provided Files -----------------------------------------------------
## EmptyAuthPwSession.sDef
## Policy%1.sDef                      PolicySha256.sDef
## UndefinespaceSpecial.pDef
##########################################################################
# This script echoes progress markers to the screen and also to
#   a log file (Tpm2PpiProv.log) which also captures provisioning details 
#
#  Note that there are commands indicated by "# For testing on NEX"  
#    These never get executed when the script is run in the EFI environment
#     - they are for ease of manually testing using the NSH Exerciser
#     - Those lines may be deleted  
##########################################################################

#Check that Alg parameter is present
if ZZ%1  == ZZ   then
	echo Algorithm parameter missing
	echo Syntax: %0 <alg> <PlatformName>
	echo   Examples: 
	echo     %0 SHA256 Example 
	echo     %0 SHA384 MyFirst
	echo     %0 SHA512 Grantley
	goto EOF
endif
# Check if alg parameter supported
if NOT EXIST Policy%1.sDef then  
	echo Algorithm not supported
	goto EOF
endif
# Check if platform parameter supported
if NOT EXIST %2PPI%1.iDef then  
	echo PlatformName not valid
	goto EOF
endif

# Now copy this file to the log
echo TPM2 provisioning Script %1 %2 %3  > Tpm2PpiProv.log
if EXIST %0     then
	type %0     >> Tpm2PpiProv.log
endif
if EXIST %0.nsh then 
	type %0.nsh >> Tpm2PpiProv.log
endif

goto START
##########################################################################
# #####For testing on NEX #####
# The following commands do not need to be sent to a TPM on a 
#   target platform because BIOS starts the TPM  
#   -- rather they are here for testing on a simulator or TTV
#   --   using the NSH Exerciser
#
#  On Simulator start-up or after inserting a TPM in the TTV
Tpm2ProvTool  PowerUp      # for testing on NEX
Tpm2ProvTool  StartUpClear # for testing on NEX
#
# Before removing a TPM from the TTV
Tpm2ProvTool ShutdownClear # for testing on NEX
Tpm2ProvTool PowerDown     # for testing on NEX
##########################################################################

:START
echo ****** Provisioning NV Indexes *****
# Uses Session 0 for AuthPolicy
# Uses Session 1 for Empty AuthValue PW session
# Note that session 0 uses "Don't Continue" so no need to flush on successful use

# This script requires that PlatformAuth be EMPTY in order to create the index

# Start a PW Session to use for PhAuth and Index Read Auth
echo **** Start PW Session for PlatformAuth & Index Read Auth
echo **** Start PW Session for PlatformAuth & Index Read Auth >> Tpm2PpiProv.log
Tpm2ProvTool StartSession EmptyAuthPwSession.sDef 1 >> Tpm2PpiProv.log

# This script requires PlatformAuth be EMPTY to create or delete NV Indexes
# Lets check if PlatformAuth is EMPTY
Tpm2ProvTool HierarchyChangeAuth PH EmptyAuth.dat 1 >> Tpm2DeleteAux.log
if not %lasterror% == 0 then
	echo PlatformAuth is not EMPTY
	echo First run ResetPlatformAuth.nsh %1 %2
	goto ERROR
endif



:PPICHECK
# ***************************************************************
echo *************  Provisioning PPI Index  *********************
# ***************************************************************

echo **** Checking if PPI Index exists
echo **** Checking if PPI Index exists >> Tpm2PpiProv.log
Tpm2ProvTool NvReadPublic %2PPI%1.iDef >> Tpm2PpiProv.log
if not %lasterror% == 0 then
	echo PPI Index does not exist
        if p%3 == "p-D" then
            echo Cannot find index to delete
		    goto EOF
        endif
	goto CREATE		# Index does not exist 
endif

# We are here becase the NV Index already exists
if p%3 == "p-D" then
  goto DELETE
endif

echo **** Comparing attributes against definition
echo **** Comparing attributes against definition >> Tpm2PpiProv.log
Tpm2ProvTool NvVerifyDef %2PPI%1.iDef >> Tpm2PpiProv.log
if %lasterror% == 0 then
	goto DONE
endif

:DELETE
# Index attributes are incorrect
echo ****** Deleting PPI Index *****
echo ****** Deleting PPI Index ***** >> Tpm2PpiProv.log
echo **** Start Policy Session for Index auth
echo **** Start Policy Session for Index auth >> Tpm2PpiProv.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2PpiProv.log
	
# Satisfy the PPI authPolicy
# There are 4 possible policy styles
#	A. single branch with 1 assertion:    PolCommandCode(NV_UndefinespaceSpecial)
#	B. multiple branches with 1 of them:  PolCommandCode(NV_UndefinespaceSpecial)
#	C. single branch with 2 assertions:   PolOR(0,SecretValue) && PolCommandCode(NV_UndefinespaceSpecial)
#	D. multiple branches with 1 of them:  PolOR(0,SecretValue) && PolCommandCode(NV_UndefinespaceSpecial)

# Let's try A & B first
	
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace)
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace) >> Tpm2PPIProv.log
Tpm2ProvTool AssertPolicy UnDefineSpaceSpecial.pDef 0 >> Tpm2PPIProv.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2PPIProv.log

# this is all that is needed for Style A - so let's try to delete the index
echo *** Execute UndefineSpecial for the Index
echo *** Execute UndefineSpecial for the Index >> Tpm2PPIProv.log
Tpm2ProvTool NvUndefineSpecial %2PPI%1.iDef 0 1 >> Tpm2PPIProv.log
if %lasterror% == 0 then
    goto DELETED
endif

# you get here if that did not work,  So lt's assume style B
#  so send the final PolicyOR assertion and try again

echo *** Policy OR (Branch 1, Branch 2, Branch 3)
echo *** Policy OR (Branch 1, Branch 2, Branch 3) >> Tpm2PPIProv.log
Tpm2ProvTool AssertPolicy %2PPIFinalOR%1.pDef 0 >> Tpm2PPIProv.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2PPIProv.log

echo *** Execute UndefineSpecial for the Index (Style B)
echo *** Execute UndefineSpecial for the Index (Style B) >> Tpm2PPIProv.log
Tpm2ProvTool NvUndefineSpecial %2PPI%1.iDef 0 1 >> Tpm2PPIProv.log
if %lasterror% == 0 then
    goto DELETED
endif

# We get here if that did not work.  So let's assume style C
# First we have to reset the Policy Session
echo **** Restart session
Tpm2ProvTool PolicyRestart 0 >> Tpm2PPIProv.log

echo *** Policy OR (0, %2SecretValue%1)
echo *** Policy OR (0, %2SecretValue%1) >> Tpm2PPIProv.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2PPIProv.log
#Tpm2ProvTool StorePolDigest 0 6 >> Tpm2PPIProv.log

echo *** Policy Command Code (TPM_CC_NV_UndefineSpace)
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace) >> Tpm2PPIProv.log
Tpm2ProvTool AssertPolicy UnDefineSpaceSpecial.pDef 0 >> Tpm2PPIProv.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2PPIProv.log

# this is all that is needed for PS2 - so let's see
echo *** Execute UndefineSpaceSpecial 
echo *** Execute UndefineSpaceSpecial  >> Tpm2PPIProv.log
Tpm2ProvTool NvUndefineSpecial %2PPI%1.iDef 0 1 >> Tpm2PPIProv.log
if %lasterror% == 0 then
    goto DELETED
endif

# you get here if that did not work, 
#  so send the final PolicyOR assertion and try again

echo **** Policy OR (Branch 1, Branch 2, Branch 3)
echo **** Policy OR (Branch 1, Branch 2, Branch 3) >> Tpm2PPIProv.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2PPIProv.log
Tpm2ProvTool StorePolDigest 0 6 >> Tpm2DeletePS.log

echo **** UndefineSpecial PS Index
echo **** UndefineSpecial PS Index >> Tpm2PPIProv.log
Tpm2ProvTool NvUndefineSpecial %PPI%1.iDef 0 1 >> Tpm2PPIProv.log
if not %lasterror% == 0 then
    echo Did not satisfy index authPolicy for deleting the index
	goto ERROR		# Didn't satisfy Index authPolicy
endif


:DELETED
if p%3 == p-D then
  goto EOF
endif


:CREATE
echo ***** Creating PPI Index *****
echo **** NV_DefineSpace for PPI Index
echo **** NV_DefineSpace for PPI Index >> Tpm2PpiProv.log
Tpm2ProvTool NvDefineSpace %2PPI%1.iDef 1 >> Tpm2PpiProv.log
if not %lasterror% == 0 then
    echo Requires PlatformAuth be EMPTY  
	echo If PlatformAuth is not EMPTY, then first run ResetPlatformAuth.nsh %1 %2
	goto ERROR		# Didn't create the NV Index
endif

goto DONE
goto PPICHECK  # for testing on NEX



:DONE
echo  ***************************************************
echo  ******* Provisioning Completed Successfully *******
echo  ***************************************************

goto EOF

:ERROR
echo  ***************************************************
echo  *************** Provisioning FAILED ***************
echo  ***************************************************
Tpm2ProvTool FlushSession 0 >> Tpm2PpiProv.log

goto EOF
goto START # for testing on NSH

:EOF
# Tpm2ProvTool FlushSession 1



