##########################################################################
# TPM 2.0 Sample TXT Provisioning Script  
# Copyright Intel Corporation 2018
# Last update: Mar 27, 2018
#   Mar 2018 Modified for Converged BtG and TXT (CBnT)
##########################################################################
##########################################################################
# Usage:                                                                 #
#     Tpm2_CBnT_Prov.nsh   <alg>                                         #
#  Where:                                                                #
#      <alg> is  a supported hash algorithm                              #
#         Such as SHA1, SHA256, SHA384, SHA512, SM3                      #
#         <alg> will be substituted for %1 in Tpm2ProvTool command lines #
##########################################################################
# This script only creates the AUX index because PS index is no longer used
#   It checks to see if the index already exists and if so, is it correct, 
#   if not, deletes the index so it can be re-created
#   However, for CBnT, there is a different process for deleating the AUX Index

echo -OFF 
# Copyrights:
# You may copy, distribute, and make derivatives of this script under the 
# following provisions:
#  You must not remove or modify the Intel copyright notice nor these requirements
#  The script must only be used to provision a TPM for use with Intel Technologies
#    For any other use you must get written permission from Intel Corporation
 
##########################################################################
# This script will:
# Check if the AUX Index already exists
# - if AUX does not exist, creates the AUX Index 
# - If AUX does exist, it checks the index's attributes 
#   -- if wrong attributes - deletes the index, recreates it, 

##########################################################################
# Here is a list of files that are needed
## Intel pProvided files -----------[example for %1=SHA256]
## Aux%1.iDef                         AuxSha256.iDef
## AuxPolicyOR%1.pDef                 AuxPolicyORSha256.pDef
## EmptyAuthPwSession.sDef
## getsec64.efi                    -- part of Intel TXT Toolkit  
## Locality3&4.pDef
## Policy%1.sDef                      PolicySha256.sDef
## UndefinespaceSpecial.pDef
##########################################################################
# This script echoes progress markers to the screen and also to
#   a log file (Tpm2CBnTProv.log) which also captures provisioning details 
#
#  Note that there are commands indicated by "# For testing on NEX"  
#    These never get executed when the script is run in the EFI environment
#     - they are for ease of manually testing using the NSH Exerciser
#     - Those lines may be deleted  
##########################################################################

#Check that Alg parameter is present
if ZZ%1  == ZZ   then
	echo Algorithm parameter missing
	echo Syntax: %0 <alg>
	echo   Example: 
	echo     %0 SHA256   
	echo     %0 SHA384  
	echo     %0 SM3  
	goto EOF
endif
# Check if alg parameter supported
if NOT EXIST Policy%1.sDef then  
	echo Algorithm not supported - Policy%1.sDef not found
	goto EOF
endif

# Now copy this file to the error log
# since batch file can be invoked with or without extension - try both forms
#echo TPM2 provisioning Script %1 %2 %3 > TPM2_CBnT_Prov.log
#if EXIST %0     then
#	type %0     >> TPM2_CBnT_Prov.log
#endif
#if EXIST %0.nsh then 
#	type %0.nsh >> TPM2_CBnT_Prov.log
#endif

goto START
##########################################################################
# #####For testing on NEX #####
# The following commands do not need to be sent to a TPM on a 
#   target platform because BIOS starts the TPM  
#   -- rather they are here for testing on a simulator or TTV
#   --   using the NSH Exerciser
#
#  On Simulator start-up or after inserting a TPM in the TTV
Tpm2ProvTool  PowerUp      # for testing on NEX
Tpm2ProvTool  StartUpClear # for testing on NEX
#
# Before removing a TPM from the TTV
Tpm2ProvTool ShutdownClear # for testing on NEX
Tpm2ProvTool PowerDown     # for testing on NEX
##########################################################################

:START
echo ****** Provisioning AUX NV Index *****
# Uses Session 0 for AuthPolicy
# Uses Session 1 for Empty AuthValue PW session
# Note that session 0 uses "Don't Continue" so no need to flush on successful use

# Start a PW Session to use for PhAuth and Index Read Auth
echo **** Start PW Session for PlatformAuth & Index Read Auth
echo **** Start PW Session for PlatformAuth & Index Read Auth >> TPM2_CBnT_Prov.log
Tpm2ProvTool StartSession EmptyAuthPwSession.sDef 1 >> TPM2_CBnT_Prov.log

# This script requires PlatformAuth be EMPTY to create or delete NV Indexes
# Lets check if PlatformAuth is EMPTY by attemping to change it to EMPTY
Tpm2ProvTool HierarchyChangeAuth PH EmptyAuth.dat 1 >> TPM2_CBnT_Prov.log
if not %lasterror% == 0 then
	echo PlatformAuth is not EMPTY
	echo First run ResetPlatformAuth.nsh
	goto ERROR
endif

##########################################################################
# ***************************************************************
# *************************AUX INDEX ****************************
# ***************************************************************

echo **** Checking if AUX index exists
echo **** Checking if AUX index exists >> TPM2_CBnT_Prov.log
Tpm2ProvTool NvReadPublic Aux%1.iDef >> TPM2_CBnT_Prov.log
if not %lasterror% == 0 then
	echo Aux Index does not exist
	goto AUXCREATE		# Index does not exist 
endif

echo **** AUX already exists, check if provisioned correctly 
echo **** AUX already exists, check if provisioned correctly  >> TPM2_CBnT_Prov.log
Tpm2ProvTool NvVerifyDef Aux%1.iDef >> TPM2_CBnT_Prov.log
if %lasterror% == 0 then 
	echo Aux Index provisioned correctly
	goto DONE
endif

##########################################################################
echo **** Need to delete AUX Index
echo **** Need to delete AUX Index >> TPM2_CBnT_Prov.log

## Here is where things change for CBnT
## the bit to authorize deleting the AUX index is now in the BPM instead of the PS Index

echo It is required that the BPM have the AUX Delete bit in the BPM (IBB Element FLAGS) 

:AUXDELETE
#echo ***** Deleting AUX Index *****
echo **** "ServerGetSec -l ent -fn res"
echo **** "ServerGetSec -l ent -fn res" >> TPM2_CBnT_Prov.log
getsec64.efi -l ent -fn res >> TPM2_CBnT_Prov.log
if not %lasterror% == 0 then
	echo Unable to delete Aux index
	goto ERROR		# Didn't delete Index 
endif
goto CONTINUE
##########################################################################
# Delete AUX on Simulator or TTV
echo **** Start Policy Session 
Tpm2ProvTool StartSession Policy%1.sDef 0  # For testing on NEX
echo **** Locality3&4 Assertion 
Tpm2ProvTool AssertPolicy Locality3&4.pDef 0  # For testing on NEX
echo **** CommandCode assertion
Tpm2ProvTool AssertPolicy UndefineSpaceSpecial.pDef 0  # For testing on NEX
echo **** PolicyOR assertion
Tpm2ProvTool AssertPolicy AuxPolicyOR%1.pDef 0  # For testing on NEX
echo **** change to locality 3 
Tpm2ProvTool SetActiveLocality 3 # For testing on NEX
echo **** UndefineSpaceSpecial for AUX
Tpm2ProvTool NvUndefineSpecial Aux%1.iDef 0 1  # For testing on NEX
if not %lasterror% == 0 then
    Tpm2ProvTool SetActiveLocality 0
	echo Cannot delete AUX index
	set FAIL = TRUE
	goto ERROR		
endif
Tpm2ProvTool SetActiveLocality 0 # For testing on NEX

##########################################################################
:CONTINUE
#
# Need to check to make sure AUX Deletion was really successful
echo **** Checking if AUX index was deleted
echo **** Checking if AUX index was deleted >> TPM2_CBnT_Prov.log
Tpm2ProvTool NvReadPublic Aux%1.iDef >> TPM2_CBnT_Prov.log
if %lasterror% == 0 then
	echo Aux Index was not deleted
	goto ERROR		# Index still exists
endif
##########################################################################
:AUXCREATE
echo ******** Creating Aux Index ********
echo **** AUX NV_DefineSpace
echo **** AUX NV_DefineSpace >> TPM2_CBnT_Prov.log
Tpm2ProvTool NvDefineSpace Aux%1.iDef 1 >> TPM2_CBnT_Prov.log
if not %lasterror% == 0 then
	echo Unable to create AUX Index
	goto ERROR		
endif
goto DONE
goto AUXCHECK # for testing on NEX
##########################################################################


:DONE
echo  ***************************************************
echo  ******* Provisioning Completed Successfully *******
echo  ***************************************************

goto EOF

:ERROR
echo  ***************************************************
echo  *************** Provisioning FAILED ***************
echo  ***************************************************
Tpm2ProvTool FlushSession 0 >> TPM2_CBnT_Prov.log

goto EOF

:EOF



