// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file capsule_validation.h
 * @brief Perform validation on a firmware update capsule.
 */

#ifndef WHITLEY_INC_CAPSULE_VALIDATION_H
#define WHITLEY_INC_CAPSULE_VALIDATION_H

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "authentication.h"
#include "cpld_update.h"
#include "firmware_update.h"
#include "pbc.h"
#include "pfr_pointers.h"


/**
 * @brief This function validates a PBC structure.
 * For example, this function checks the tag, version and page size.
 *
 * @param pbc pointer to a PBC structure.
 *
 * @return alt_u32 1 if all checks passed; 0, otherwise.
 */
static alt_u32 is_pbc_valid(PBC_HEADER* pbc)
{
    // Check Magic number
    if (pbc->tag != PBC_EXPECTED_TAG)
    {
        return 0;
    }

    // Check version
    if (pbc->version != PBC_EXPECTED_VERSION)
    {
        return 0;
    }

    // only 4kB pages are valid for SPI NOR flash used in PFR systems.
    if (pbc->page_size != PBC_EXPECTED_PAGE_SIZE)
    {
        return 0;
    }

    // Number of bits in bitmap must be multiple of 8
    // This is because Nios processes bitmap byte by byte.
    if (pbc->bitmap_nbit % 8 != 0)
    {
        return 0;
    }

    return 1;
}

/**
 * @brief Perform authentication on a signed capsule.
 * This includes verifying signature of the capsule and
 * signature of the signed PFM inside the capsule.
 *
 * @param signed_capsule start address of a signed capsule
 * @return alt_u32 1 if the signed capsule is authentic; 0, otherwise.
 */
static alt_u32 are_signatures_in_capsule_valid(alt_u32* signed_capsule)
{
    // Verify the signature of the capsule
    if (is_signature_valid((KCH_SIGNATURE*) signed_capsule))
    {
        // Verify the signature of the PFM inside the capsule
        KCH_SIGNATURE *signed_pfm_addr = (KCH_SIGNATURE*) incr_alt_u32_ptr(signed_capsule, SIGNATURE_SIZE);
        return is_signature_valid(signed_pfm_addr);
    }
    return 0;
}

/**
 * @brief Perform validation on a signed capsule.
 * This includes validating expected data in the PBC structure and
 * verifying signatures in the capsule and PFM.
 *
 * @param signed_capsule start address of a signed capsule
 * @return alt_u32 1 if the signed capsule is valid; 0, otherwise.
 */
static alt_u32 is_capsule_valid(alt_u32* signed_capsule)
{
    // Check the two signatures in a firmware update capsule
    if (are_signatures_in_capsule_valid(signed_capsule))
    {
        // Check the compression structure definition
        return is_pbc_valid(get_pbc_ptr_from_signed_capsule(signed_capsule));
    }
    return 0;
}

/**
 * @brief Pre-process the update capsule prior to performing the FW or CPLD update.
 *
 * This function first checks the capsule signature. If that passes, Nios continues to check its content.
 * If this is a key cancellation certificate, Nios cancels the key.
 * If this is a CPLD update capsule, this function returns 1 if its SVN is valid.
 * If this is a firmware update capsule, Nios proceeds to check the signature of PFM. Once that passes,
 * SVN is then checked. The SVN checks here are more involved. The capsule PFM SVN is validated against the
 * SVN policy stored in UFM. Also, if this is an active firmware update, its SVN must be the same as the SVN
 * of recovery image. If user wishes to bump the SVN, a recovery firmware update, which update both active and
 * recovery firmware, must be triggered.
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 * @param signed_capsule pointer to the start address of a signed capsule
 * @param is_cpld_update 1 if this is a CPLD update; 0 if this is a firmware update
 * @param is_fw_recovery_update 1 if this is an update to the recovery firmware; 0, otherwise.
 *
 * @return alt_u32 1 if Nios should proceed to perform the CPLD or FW update with the @p signed_capsule; 0, otherwise.
 *
 * @see act_on_update_intent
 * @see is_svn_valid
 */
static alt_u32 check_capsule_before_update(
        SPI_FLASH_TYPE_ENUM spi_flash_type, alt_u32* signed_capsule, alt_u32 is_cpld_update, alt_u32 is_fw_recovery_update)
{
    // This minor error code will be posted to Mailbox in case of check failure
    alt_u32 update_event_minor_error = MINOR_ERROR_FW_UPDATE_AUTH_FAILED;
    if (is_cpld_update)
    {
        update_event_minor_error = MINOR_ERROR_CPLD_UPDATE_AUTH_FAILED;
    }

    // Validate Capsule signature
    if (is_signature_valid((KCH_SIGNATURE*) signed_capsule))
    {
        KCH_BLOCK0* b0 = (KCH_BLOCK0*) signed_capsule;
        alt_u32* capsule_pc = incr_alt_u32_ptr(signed_capsule, SIGNATURE_SIZE);

        if (b0->pc_type & KCH_PC_TYPE_KEY_CAN_CERT_MASK)
        {
            // If this is a key cancellation certificate, proceed to cancel this key.
            cancel_key(get_kch_pc_type(b0), ((KCH_CAN_CERT*) capsule_pc)->csk_id);

            // There's no update or error logging to do. Exit now.
            return 0;
        }
        else if (b0->pc_type == KCH_PC_PFR_CPLD_UPDATE_CAPSULE)
        {
            // This is a CPLD update
            if (is_svn_valid(UFM_SVN_POLICY_CPLD, ((CPLD_UPDATE_PC*) capsule_pc)->svn))
            {
                // Yes, this is a valid CPLD update capsule. Proceed with this CPLD update
                return 1;
            }

            // Failed SVN check
            update_event_minor_error = MINOR_ERROR_CPLD_UPDATE_INVALID_SVN;
        }
        else
        {
            // This is a firmware update

            // Validate PFM signature
            if (is_signature_valid((KCH_SIGNATURE*) capsule_pc))
            {
                // Validate SVN
                alt_u8 new_svn = get_capsule_pfm(signed_capsule)->svn;

                // Check whether this SVN is allowed by looking at the UFM SVN policy
                if ((get_kch_pc_type(b0) == KCH_PC_PFR_PCH_UPDATE_CAPSULE) && is_svn_valid(UFM_SVN_POLICY_PCH, new_svn))
                {
                    if (is_fw_recovery_update || (read_from_mailbox(MB_PCH_PFM_RECOVERY_SVN) == new_svn))
                    {
                        // Yes, this is a valid firmware update capsule. Proceed with the update.
                        fw_active_update_svn_for_pch = new_svn;
                        return 1;
                    }
                }
                else if ((get_kch_pc_type(b0) == KCH_PC_PFR_BMC_UPDATE_CAPSULE) && is_svn_valid(UFM_SVN_POLICY_BMC, new_svn))
                {
                    if (is_fw_recovery_update || (read_from_mailbox(MB_BMC_PFM_RECOVERY_SVN) == new_svn))
                    {
                        // Yes, this is a valid firmware update capsule. Proceed with the update.
                        fw_active_update_svn_for_bmc = new_svn;
                        return 1;
                    }
                }

                update_event_minor_error = MINOR_ERROR_FW_UPDATE_INVALID_SVN;
            }
        }
    }

    // If Nios reaches this point, it means that this incoming update capsule is rejected.
    // Update number of failed attempts
    if (is_cpld_update)
    {
        num_failed_cpld_update_attempts++;
    }
    else // This is a firmware update
    {
        incr_fw_update_failed_attempts(spi_flash_type);
    }

    // Log the error
    log_update_failure(spi_flash_type, update_event_minor_error);

    return 0;
}

#endif /* WHITLEY_INC_CAPSULE_VALIDATION_H */
