// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file cpld_reconfig.h
 * @brief Functions that perform CPLD reconfiguration.
 */

#ifndef WHITLEY_INC_CPLD_RECONFIG_H_
#define WHITLEY_INC_CPLD_RECONFIG_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "gen_gpi_signals.h"
#include "gen_gpo_controls.h"
#include "timer_utils.h"
#include "utils.h"

/**
 * @brief Preparation before reconfiguring CPLD.
 */
static void prep_for_reconfig()
{
    // Clear Deep SX. This will drive Z onto PWRGD_DSW_PWROK_R
    clear_bit(U_GPO_1_ADDR, GPO_1_PWRGD_DSW_PWROK_R);

    // Add some delays between DSW_PWROK and PFR_SLP_SUS
    // The max time for this transition is 100ms. Set a delay of 110ms to be safe.
    // If this delay is not added or is too small, platform would enter
    //   a power loop (constantly shutting down and then power up).
    sleep_ms(110);
}

/**
 * @brief Perform switching of CPLD image to use either CFM0 or CFM1
 *
 * @param cfm_one_or_zero an enum value indicating which CFM to switch to
 */
static void perform_cfm_switch(CPLD_CFM_TYPE_ENUM cfm_one_or_zero)
{
    // Set ConfigSelect register to either CFM1 or CFM0, bit[1] represent CFM selection, bit[0] commit to register
    IOWR(U_DUAL_CONFIG_BASE, 1, (cfm_one_or_zero << 1) | 1);

    // Check if written new ConfigSelect value being consumed
    poll_on_dual_config_ip_busy_bit();

    // We need to deassert deep sleep before starting reconfiguration
    // We will lose power when all pins to go high-Z otherwise
    prep_for_reconfig();

    // Trigger reconfiguration
    IOWR(U_DUAL_CONFIG_BASE, 0, 0x1);
}

#endif /* WHITLEY_INC_CPLD_RECONFIG_H_ */
