// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file cpld_recovery.h
 * @brief Responsible for updating CPLD bitstream in the case of a forced recovery.
 */

#ifndef WHITLEY_INC_CPLD_RECOVERY_H_
#define WHITLEY_INC_CPLD_RECOVERY_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "global_state.h"
#include "transition.h"
#include "spi_ctrl_utils.h"
#include "ufm_utils.h"
#include "authentication.h"
#include "tmin1_routines.h"


static void perform_cpld_recovery(alt_u32 oldest_backup_addr_in_flash, alt_u32 newest_backup_addr_in_flash, alt_u32 num_valid_backups)
{
    switch_spi_flash(SPI_FLASH_BMC);
    alt_u32 bmc_staging_offset = get_ufm_pfr_data()->bmc_staging_region;
    alt_u32 cpld_factory_default_image_addr_in_flash = bmc_staging_offset + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET;
    alt_u32* cpld_recovery_capsule_ptr = get_spi_flash_ptr_with_offset(cpld_factory_default_image_addr_in_flash);

    // Assume that we entered recovery because the newest backup image
    // Erase first page to show that it is invalid
    erase_spi_region(newest_backup_addr_in_flash, SPI_FLASH_PAGE_SIZE);

    if (num_valid_backups == 2)
    {
        // Skip the first word because it is the internal version number
        cpld_recovery_capsule_ptr = get_spi_flash_ptr_with_offset(oldest_backup_addr_in_flash + 4);
        if(!is_signature_valid((KCH_SIGNATURE*) cpld_recovery_capsule_ptr))
        {
            num_valid_backups--;
        }
    }
    if (num_valid_backups == 1 || num_valid_backups == 0)
    {
        if (num_valid_backups == 1 )
        {
            // Assume that we entered recovery because the oldest backup image
            // Erase first page to show that it is invalid
            erase_spi_region(oldest_backup_addr_in_flash, SPI_FLASH_PAGE_SIZE);
        }
        // There is no internal version number on the factory capsule
        cpld_recovery_capsule_ptr = get_spi_flash_ptr_with_offset(cpld_factory_default_image_addr_in_flash);
        if(!is_signature_valid((KCH_SIGNATURE*) cpld_recovery_capsule_ptr))
        {
            // Brick device
            // TBD on what do do in this state
            never_exit_loop();
        }
    }

    CPLD_UPDATE_PC* cpld_update_protected_content = (CPLD_UPDATE_PC*) incr_alt_u32_ptr(cpld_recovery_capsule_ptr, SIGNATURE_SIZE);
    alt_u32* spi_cpld_bitstream_ptr = cpld_update_protected_content->cpld_bitstream;

    // Copy the new image into cfm backwards so that we don't have a partially functional image in the case of a power outage
    ufm_erase_sector(CMF1_TOP_HALF_SECTOR_ID);
    ufm_erase_sector(CMF1_BOTTOM_HALF_SECTOR_ID);
    reset_hw_watchdog();

    // Pointer to the top of CFM1
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = (UFM_CPLD_ACTIVE_IMAGE_LENGTH / 4 - 1); i > 0; i--)
    {
        cfm1_ptr[i] = spi_cpld_bitstream_ptr[i];
    }
    cfm1_ptr[0] = spi_cpld_bitstream_ptr[0];
    reset_hw_watchdog();

    // The CPLD recovery is done. Run the recovered image.
    perform_cfm_switch(CPLD_CFM1);
}

#endif /* WHITLEY_INC_CPLD_RECOVERY_H_ */
