// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file cpld_update.h
 * @brief Responsible for updating CPLD bitstream and triggering reconfiguration.
 */

#ifndef WHITLEY_INC_CPLD_UPDATE_H_
#define WHITLEY_INC_CPLD_UPDATE_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "authentication.h"
#include "cpld_reconfig.h"
#include "global_state.h"
#include "key_cancellation.h"
#include "pfr_pointers.h"
#include "platform_log.h"
#include "spi_ctrl_utils.h"
#include "ufm_rw_utils.h"
#include "ufm_utils.h"

// Keeping track of the number of failed CPLD updates
static alt_u32 num_failed_cpld_update_attempts = 0;

typedef enum {
    UPDATE_SOURCE_BMC = 0,
    UPDATE_SOURCE_PCH = 1,
} CPLD_UPDATE_SOURCE_ENUM;

#define RECONFIG_CPLD_UPDATE 0b010

/*!
 * Describe the format of the protected content in the CPLD update capsule
 */
typedef struct
{
    alt_u32 svn;
    alt_u32 cpld_bitstream[UFM_CPLD_ACTIVE_IMAGE_LENGTH / 4];
    alt_u32 padding[31];
} CPLD_UPDATE_PC;


/**
 * @brief Trigger the CPLD update by switching to the recovery image.
 *
 * A bread crumb is left in the last word of CFM1, to indicate that a
 * CPLD update is in progress. It's assumed that this last word is not
 * being used. This bread crumb will be cleaned up upon a successful
 * CPLD update.
 */
static void trigger_cpld_update()
{
    log_platform_state(PLATFORM_STATE_CPLD_UPDATE);

    // Leave bread crumb to indicate that the CFM switch was caused by CPLD update
    alt_u32* cfm1_breadcrumb = get_ufm_ptr_with_offset(CFM1_BREAD_CRUMB);
    *cfm1_breadcrumb = 0;

    // Switch the recovery image to perform the update
    perform_cfm_switch(CPLD_CFM0);
}

/**
 * @brief Perform the second half of the CPLD update flow from within the recovery image.
 * This will write into the configuration bitstream into CFM0 and
 * backup the update capsule in BMC staging to the backup area
 * This should not be called within the main PFR code.
 *
 * @param oldest_backup_addr The address in BMC CPI Flash that contains the oldest update backup
 * This gets overwritten during backup
 * @param newest_backup_addr The address in BMC CPI Flash that contains the newest update backup
 * This is used to figure out what the new internal verion number should be
 */
static void perform_update_post_reconfig(alt_u32 oldest_backup_addr, alt_u32 newest_backup_addr)
{
    alt_u32* cpld_update_capsule_ptr = get_spi_flash_ptr_with_offset(
            get_ufm_pfr_data()->bmc_staging_region +
            BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET
    );

    // Pointers and values in the BMC SPI flash cpld update staging area
    CPLD_UPDATE_PC* cpld_update_protected_content = (CPLD_UPDATE_PC*)incr_alt_u32_ptr(cpld_update_capsule_ptr, SIGNATURE_SIZE);
    alt_u32 svn = cpld_update_protected_content->svn;
    alt_u32* spi_cpld_bitstream_ptr = cpld_update_protected_content->cpld_bitstream;

    // Points to the start of the BMC SPI Flash backup areas
    alt_u32* oldest_backup_ptr = get_spi_flash_ptr_with_offset(oldest_backup_addr);
    alt_u32* current_internal_version_num_ptr = get_spi_flash_ptr_with_offset(newest_backup_addr);

    // Return to active image if authentication fails
    if (svn < get_ufm_svn(UFM_SVN_POLICY_CPLD) ||
            !is_signature_valid((KCH_SIGNATURE*) cpld_update_capsule_ptr))
    {
        perform_cfm_switch(CPLD_CFM1);
        return;
    }

    // CFM1 is stored within 2 sectors
    ufm_erase_sector(CMF1_TOP_HALF_SECTOR_ID);
    ufm_erase_sector(CMF1_BOTTOM_HALF_SECTOR_ID);

    // We need to assign an internal version number to the backup to keep track of the newest backup
    alt_u32 next_internal_version_num;
    if (*current_internal_version_num_ptr == 0xFFFFFFFF)
    {
        next_internal_version_num = 0;
    }
    else
    {
        next_internal_version_num = *current_internal_version_num_ptr + 1;
    }

    // Overwrite the oldest backup slot with the update + internal version number
    // the CPLD update capsule is stored right after the internal version number
    erase_spi_region(oldest_backup_addr, MAX_CPLD_UPDATE_CAPSULE_SIZE);
    *oldest_backup_ptr = next_internal_version_num;
    reset_hw_watchdog();
    alt_u32_memcpy(oldest_backup_ptr + 1, cpld_update_capsule_ptr, MAX_CPLD_UPDATE_CAPSULE_SIZE - 4);
    reset_hw_watchdog();

    // Pointer to the top of CFM1
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    // Copy the new image into cfm backwards so that we don't have a partially functional image in the case of a power outage
    for (alt_u32 i = (UFM_CPLD_ACTIVE_IMAGE_LENGTH / 4 - 1); i > 0; i--)
    {
        cfm1_ptr[i] = spi_cpld_bitstream_ptr[i];
    }
    reset_hw_watchdog();

    // Write_ufm_svn will only update the SVN if the input SVN in larger than what is in memory
    write_ufm_svn(svn, UFM_SVN_POLICY_CPLD);

    // Copying the first word to CFM0 should be the last thing we do
    // This prevents the situation where a powercycle mid update
    // causes us to have a partially working CPLD image
    cfm1_ptr[0] = spi_cpld_bitstream_ptr[0];

    // The CPLD update is done. Run the new image.
    perform_cfm_switch(CPLD_CFM1);
}

#endif /* WHITLEY_INC_CPLD_UPDATE_H_ */
