// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file firmware_update.h
 * @brief Responsible for firmware updates, as specified in the BMC/PCH update intent, in T-1 mode.
 */

#ifndef WHITLEY_INC_FIRMWARE_UPDATE_H
#define WHITLEY_INC_FIRMWARE_UPDATE_H

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "crypto.h"
#include "decompression.h"
#include "gen_gpo_controls.h"
#include "key_cancellation.h"
#include "keychain_utils.h"
#include "platform_log.h"
#include "spi_ctrl_utils.h"
#include "spi_flash_state.h"
#include "utils.h"
#include "watchdog_timers.h"

// Tracking number of failed firmware update attempts
static alt_u32 num_failed_pch_update_attempts = 0;
static alt_u32 num_failed_bmc_update_attempts = 0;

// Tracking the SVN for the active update in progress
static alt_u32 fw_active_update_svn_for_pch = 0;
static alt_u32 fw_active_update_svn_for_bmc = 0;

/**
 * @brief Check whether firmware update from the @p spi_flash_type flash has exceeded
 * max number of failed attempts. The maximum number of failed firmware update attempts
 * is defined in pfr_sys.h.
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 * @return alt_u32 1 if firmware update has exceeded max number of failed attempts
 */
static alt_u32 block_fw_update_attempt(SPI_FLASH_TYPE_ENUM spi_flash_type)
{
    if (spi_flash_type == SPI_FLASH_PCH)
    {
        return num_failed_pch_update_attempts >= MAX_PCH_FW_UPDATE_FAIL_ATTEMPTS;
    }
    // spi_flash_type == SPI_FLASH_BMC
    return num_failed_bmc_update_attempts >= MAX_BMC_FW_UPDATE_FAIL_ATTEMPTS;
}

/**
 * @brief Increment the global variable that tracks number of failed firmware update
 * attempt from the @p spi_flash_type flash.
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 */
static void incr_fw_update_failed_attempts(SPI_FLASH_TYPE_ENUM spi_flash_type)
{
    if (spi_flash_type == SPI_FLASH_PCH)
    {
        num_failed_pch_update_attempts++;
    }
    else
    {
        // spi_flash_type == SPI_FLASH_BMC
        num_failed_bmc_update_attempts++;
    }
}

/**
 * @brief Reset the global variable that tracks number of failed firmware update
 * attempt from the @p spi_flash_type flash.
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 */
static void reset_fw_update_failed_attempts(SPI_FLASH_TYPE_ENUM spi_flash_type)
{
    if (spi_flash_type == SPI_FLASH_PCH)
    {
        num_failed_pch_update_attempts = 0;
    }
    else
    {
        // spi_flash_type == SPI_FLASH_BMC
        num_failed_bmc_update_attempts = 0;
    }
}

/**
 * @brief Perform firmware update for the @p spi_flash_type flash.
 * If @p is_recovery_update is set to 1, do active update first and then set a flag
 * that will later trigger the recovery firmware update flow in T0 mode.
 *
 * Nios firmware needs to verify that the firmware from the update capsule passes authentication
 * and can boot up correctly, before promoting that firmware to the recovery region. That's why
 * active firmware update is required before performing recovery firmware update.
 *
 * Active firmware update is done by decompressing the static SPI regions from the update capsule. If
 * @p is_recovery_update is set to 1, then both static and dynamic SPI regions are being decompressed
 * from the update capsule. Active firmware PFM will also be overwritten by the PFM in the update capsule.
 *
 * If @p is_recovery_update is set to 1, Nios firmware will apply write protection on the
 * staging region. Hence, the staging update capsule won't be changed from active firmware update to
 * recovery firmware update.
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 * @param update_intent the value of the update intent register
 *
 * @see decompress_capsule
 * @see post_update_routine
 * @see perform_firmware_recovery_update
 */
static void perform_firmware_update(SPI_FLASH_TYPE_ENUM spi_flash_type, alt_u32 update_intent)
{
    log_firmware_update(spi_flash_type);

    // Make sure we are updating the right flash device
    switch_spi_flash(spi_flash_type);

    // Check if this is a recovery update
    alt_u32 is_recovery_update = update_intent & MB_UPDATE_INTENT_BMC_RECOVERY_MASK;
    if (spi_flash_type == SPI_FLASH_PCH)
    {
        is_recovery_update = update_intent & MB_UPDATE_INTENT_PCH_RECOVERY_MASK;
    }

    // Clear the number of failed attempts
    reset_fw_update_failed_attempts(spi_flash_type);

    // Get pointers to staging capsule
    alt_u32* signed_staging_capsule = get_spi_staging_region_ptr(spi_flash_type);

    // Set active update in progress flag
    set_spi_flash_state(spi_flash_type, SPI_FLASH_STATE_ACTIVE_UPDATE_IN_PROGRESS_MASK);

    // Only overwrite static regions in active update
    DECOMPRESSION_TYPE_MASK_ENUM decomp_event = DECOMPRESSION_STATIC_REGIONS_MASK;

    if (is_recovery_update)
    {
        // The actual update to the recovery region will be executed after timed boot is completed
        // Disable write access on the staging region
        set_spi_flash_state(spi_flash_type, SPI_FLASH_STATE_RECOVERY_UPDATE_IN_PROGRESS_MASK);

        // Overwrite both static and dynamic regions in recovery update
        decomp_event = DECOMPRESSION_STATIC_AND_DYNAMIC_REGIONS_MASK;
    }
    else if (update_intent & MB_UPDATE_INTENT_UPDATE_DYNAMIC_MASK)
    {
        // Overwrite both static and dynamic regions in recovery update
        decomp_event = DECOMPRESSION_STATIC_AND_DYNAMIC_REGIONS_MASK;
    }

    // Perform decompression
    decompress_capsule(signed_staging_capsule, spi_flash_type, decomp_event);

    // Re-enable watchdog timers in case they were disabled after 3 WDT timeouts.
    if (spi_flash_type == SPI_FLASH_PCH)
    {
        wdt_enable_status |= WDT_ENABLE_PCH_TIMERS_MASK;
    }
    else // spi_flash_type == SPI_FLASH_BMC
    {
        wdt_enable_status |= WDT_ENABLE_BMC_TIMER_MASK;
    }

}

/**
 * @brief Perform recovery firmware update for @p spi_flash_type flash.
 *
 * If the staging capsule is authentic, then copy the staging capsule to overwrite
 * the recovery capsule.
 *
 * The write protection on the staging region of the flash is removed at the end.
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 *
 * @see perform_firmware_update
 * @see post_update_routine
 * @see process_pending_recovery_update
 */
static void perform_firmware_recovery_update(SPI_FLASH_TYPE_ENUM spi_flash_type)
{
    switch_spi_flash(spi_flash_type);

    alt_u32* staging_capsule = get_spi_staging_region_ptr(spi_flash_type);
    if (is_signature_valid((KCH_SIGNATURE*) staging_capsule))
    {
        // If the staging capsule is still authentic, then proceed to overwrite recovery capsule
    	memcpy_signed_payload(get_recovery_region_offset(spi_flash_type), staging_capsule);
    }

    clear_spi_flash_state(spi_flash_type, SPI_FLASH_STATE_RECOVERY_UPDATE_IN_PROGRESS_MASK);
}

#endif /* WHITLEY_INC_FIRMWARE_UPDATE_H */
