// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file flash_validation.h
 * @brief Perform validations on active/recovery/staging regions of the SPI flash memory in T-1 mode.
 */

#ifndef WHITLEY_INC_FLASH_VALIDATION_H
#define WHITLEY_INC_FLASH_VALIDATION_H

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "authentication.h"
#include "capsule_validation.h"
#include "firmware_update.h"
#include "firmware_recovery.h"
#include "keychain.h"
#include "keychain_utils.h"
#include "mailbox_utils.h"
#include "pfm_validation.h"
#include "pfr_pointers.h"
#include "spi_flash_state.h"
#include "ufm_utils.h"
#include "ufm.h"

/**
 * @brief Perform validation on the active region.
 * First, it verifies the signature of the PFM. Then, it verifies
 * the content (e.g. SPI region and SMBus rule definitions) of the PFM.
 *
 * @param active_addr start address of an active region
 * @return 1 if the active region is valid; 0, otherwise.
 */
static alt_u32 is_active_region_valid(alt_u32* active_addr)
{
    // Verify the signature of the PFM first, then SPI region definitions and other content in PFM.
    return is_signature_valid((KCH_SIGNATURE*) active_addr) &&
            is_pfm_valid((PFM*) incr_alt_u32_ptr(active_addr, SIGNATURE_SIZE));
}

/**
 * @brief Log any authentication failure in the mailbox major and minor error registers.
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 * @param is_active_invalid 1 if signed active PFM failed authentication
 * @param is_recovery_invalid 1 if signed recovery capsule failed authentication
 */
static void log_auth_results(SPI_FLASH_TYPE_ENUM spi_flash_type, alt_u32 is_active_invalid, alt_u32 is_recovery_invalid)
{
    if (is_active_invalid)
    {
        log_auth_failure(spi_flash_type, MINOR_ERROR_AUTH_ACTIVE);

        if (is_recovery_invalid)
        {
            log_auth_failure(spi_flash_type, MINOR_ERROR_AUTH_ACTIVE_AND_RECOVERY);
        }
    }
    else if (is_recovery_invalid)
    {
        log_auth_failure(spi_flash_type, MINOR_ERROR_AUTH_RECOVERY);
    }
}

/**
 * @brief Authenticate the active/recovery/staging region for @p spi_flash_type flash and 
 * perform recovery where appropriate. 
 * 
 * Nios authenticates the signed active PFM and signed recovery capsule. If any failure is seen, 
 * Nios proceeds to perform recovery according to the Recovery Matrix in the High-level Architecture Spec. 
 * 
 * Active | Recovery | Staging | Action
 * 0      | 0        | 0       | unrecoverable; do not allow the corresponding device to boot
 * 0      | 0        | 1       | Copy Staging capsule over to Recovery region; then perform an active recovery
 * 0      | 1        | 0       | Perform an active recovery
 * 0      | 1        | 1       | Perform an active recovery
 * 1      | 0        | 0       | Allow the device to boot but restrict active update; only allow recovery udpate
 * 1      | 0        | 1       | Copy the Staging capsule over to Recovery region
 * 1      | 1        | 0       | No action needed
 * 1      | 1        | 1       | No action needed
 *
 * Recovery of active image:
 * Nios performs a static recovery. Static recovery only recover the static regions (i.e. regions that allow read
 * but not write) of the active firmware. This recovery can be triggered by either authentication failure or forced
 * recovery.
 *
 * Recovery of recovery image:
 * If the staging image is valid, Nios copies it over to the recovery area. It's assumed that this scenario may occur
 * only after a power failure.
 *
 * If the active firmware is authentic, Nios applies the SPI and SMBus filtering rules. Only SPI filters are enabled
 * here. SMBus filters are enabled only when BMC has completed boot. 
 * 
 * Nios writes the PFMs' information to mailbox after all the above tasks are done. 
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 */
static void authenticate_and_recover_spi_flash(SPI_FLASH_TYPE_ENUM spi_flash_type)
{
#ifdef USE_SYSTEM_MOCK
    if (SYSTEM_MOCK::get()->should_exec_code_block(
            SYSTEM_MOCK::CODE_BLOCK_TYPES::SKIP_FLASH_AUTHENTICATION))
    {
        return;
    }
#endif

    // Log platform state
    if (spi_flash_type == SPI_FLASH_BMC)
    {
        log_platform_state(PLATFORM_STATE_BMC_FLASH_AUTHENTICATION);
    }
    else // spi_flash_type == SPI_FLASH_PCH
    {
        log_platform_state(PLATFORM_STATE_PCH_FLASH_AUTHENTICATION);
    }

    // Switch to the right flash
    switch_spi_flash(spi_flash_type);

    alt_u32* active_pfm_ptr = get_spi_active_pfm_ptr(spi_flash_type);
    alt_u32* recovery_region_ptr = get_spi_recovery_region_ptr(spi_flash_type);
    alt_u32* staging_region_ptr = get_spi_staging_region_ptr(spi_flash_type);

    /*
     * Authentication of Active region and Recovery region
     */
    // Verify the signature and content of the active section PFM
    alt_u32 is_active_valid = is_active_region_valid(active_pfm_ptr);

    // Verify the signature of the recovery section capsule
    alt_u32 is_recovery_valid = is_capsule_valid(recovery_region_ptr);

    // Check for FORCE_RECOVERY GPI signal
    alt_u32 require_force_recovery = !check_bit(U_GPI_1_ADDR, GPI_1_FM_PFR_FORCE_RECOVERY_N);

    // Log the authentication results
    log_auth_results(spi_flash_type, !is_active_valid, !is_recovery_valid);

    /*
     * Perform recovery where applied
     * Please refer to Recovery Matrix in the High-level Architecture Spec
     *
     * Possible regions state (1 means valid; 0 means invalid)
     * Active | Recovery | Staging
     * 0      | 0        | 0
     * 0      | 0        | 1
     * 0      | 1        | 0
     * 0      | 1        | 1
     * 1      | 0        | 0
     * 1      | 0        | 1
     * 1      | 1        | 0 (no action required)
     * 1      | 1        | 1 (no action required)
     */
    // Deal with the cases where Recovery image is invalid first
    if (!is_recovery_valid)
    {
        if (is_capsule_valid(staging_region_ptr))
        {
            if (spi_flash_type == SPI_FLASH_BMC)
            {
                log_tmin1_recovery(LAST_RECOVERY_BMC_RECOVERY_FAIL);
            }
            else // spi_flash_type == SPI_FLASH_PCH
            {
                log_tmin1_recovery(LAST_RECOVERY_PCH_RECOVERY_FAIL);
            }

            /* Scenarios
             * Active | Recovery | Staging
             * 0      | 0        | 1
             * 1      | 0        | 1
             *
             * Corrective action required: copy Staging over to Recovery, then perform recovery action
             */
            memcpy_signed_payload(get_recovery_region_offset(spi_flash_type), staging_region_ptr);
            is_recovery_valid = 1;
        }
        // Unable to recovery recovery image
        else if (is_active_valid)
        {
            /* Scenarios
             * Active | Recovery | Staging
             * 1      | 0        | 0
             *
             * Cannot recover the recovery image. Save the flash state.
             */
            set_spi_flash_state(spi_flash_type, SPI_FLASH_STATE_RECOVERY_FAILED_AUTH_MASK);
        }
        else
        {
            /* Scenarios
             * Active | Recovery | Staging
             * 0      | 0        | 0
             *
             * Hence, Active/Recovery/Staging images are all bad
             * Cannot recover the Active and Recovery images. Save the flash state.
             */
            set_spi_flash_state(spi_flash_type, SPI_FLASH_STATE_ALL_REGIONS_FAILED_AUTH_MASK);

            // Critical platform error
            log_auth_failure(spi_flash_type, MINOR_ERROR_AUTH_ALL_REGIONS);
        }
    }

    /* Scenarios
     * Active | Recovery | Staging
     * 0      | 1        | 0
     * 0      | 1        | 1
     *
     * Simply recover active image if needed
     */
    if (is_recovery_valid)
    {
        if (!is_active_valid || require_force_recovery)
        {
            // Logging
            if (require_force_recovery)
            {
                log_platform_state(PLATFORM_STATE_TMIN1_FORCED_ACTIVE_FW_RECOVERY);
                log_recovery(LAST_RECOVERY_FORCED_ACTIVE_FW_RECOVERY);
            }
            else
            {
                if (spi_flash_type == SPI_FLASH_BMC)
                {
                    log_tmin1_recovery(LAST_RECOVERY_BMC_ACTIVE_FAIL);
                }
                else // spi_flash_type == SPI_FLASH_PCH
                {
                    log_tmin1_recovery(LAST_RECOVERY_PCH_ACTIVE_FAIL);
                }
            }

            // When recovery image is valid and active image needs to be recovered,
            // perform recovery for all static regions in the active firmware.
            decompress_capsule(recovery_region_ptr, spi_flash_type, DECOMPRESSION_STATIC_REGIONS_MASK);
            is_active_valid = 1;
        }
    }

    if (is_active_valid)
    {
        // If the active firmware passed authentication, then apply the protection specified in the active PFM.
        apply_spi_write_protection_and_smbus_rules(spi_flash_type);
    }

    // Print the active & recovery PFM information to mailbox
    // Do this last in case there was some recovery action.
    mb_write_pfm_info(spi_flash_type);
}


#endif /* WHITLEY_INC_FLASH_VALIDATION_H */
