// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file pfm_validation.h
 * @brief Perform validation on PFM, including the PFM SPI region definition and SMBus rule definition.
 */

#ifndef WHITLEY_INC_PFM_VALIDATION_H_
#define WHITLEY_INC_PFM_VALIDATION_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "crypto.h"
#include "pfm.h"
#include "pfm_utils.h"
#include "pfr_pointers.h"

/**
 * @brief Check if the SMBus device address in a rule definition is valid.
 * The address is valid when it matches the I2C address for the given bus ID
 * and rule ID based on gen_smbus_relay_config.h
 *
 * @return 1 if success else 0
 */
static PFR_ALT_INLINE alt_u32 PFR_ALT_ALWAYS_INLINE is_device_addr_valid(PFM_SMBUS_RULE_DEF* rule_def)
{
    return rule_def->device_addr == smbus_device_addr[rule_def->bus_id - 1][rule_def->rule_id - 1];
}

/**
 * @brief Perform validation on a PFM SMBus rule definition
 *
 * @return 1 if success else 0
 */
static alt_u32 is_smbus_rule_valid(PFM_SMBUS_RULE_DEF* rule_def)
{
    // Check rule ID and Bus ID
    if ((rule_def->rule_id > MAX_I2C_ADDRESSES_PER_RELAY) || (rule_def->bus_id > NUM_RELAYS))
    {
        return 0;
    }

    // Check device address
    return is_device_addr_valid(rule_def);
}

/**
 * @brief Perform validation on a PFM defined SPI region
 *
 * @return 1 if success else 0
 */
static alt_u32 is_spi_region_valid(PFM_SPI_REGION_DEF* region_def)
{
    // Only check hash for static region
    if ((region_def->hash_algorithm & PFM_HASH_ALGO_SHA256_MASK) &&
            is_spi_region_static(region_def))
    {
        return verify_sha(region_def->region_hash,
                get_spi_flash_ptr_with_offset(region_def->start_addr),
                region_def->end_addr - region_def->start_addr);
    }
    return 1;
}

/**
 * @brief Iterate through PFM body to validate SPI region definition and SMBus rule definition.
 *
 * @return 1 if success else 0
 */
static alt_u32 is_pfm_body_valid(alt_u32* pfm_body_ptr)
{
    // Go through the PFM Body
    while (1)
    {
        alt_u8 def_type = *((alt_u8*) pfm_body_ptr);
        if (def_type == SMBUS_RULE_DEF_TYPE)
        {
            PFM_SMBUS_RULE_DEF* rule_def = (PFM_SMBUS_RULE_DEF*) pfm_body_ptr;
            if (!is_smbus_rule_valid(rule_def))
            {
                return 0;
            }
            pfm_body_ptr = incr_alt_u32_ptr(pfm_body_ptr, SMBUS_RULE_DEF_SIZE);
        }
        else if (def_type == SPI_REGION_DEF_TYPE)
        {
            PFM_SPI_REGION_DEF* region_def = (PFM_SPI_REGION_DEF*) pfm_body_ptr;
            if (!is_spi_region_valid(region_def))
            {
                return 0;
            }
            pfm_body_ptr = get_end_of_spi_region_def(region_def);
        }
        else
        {
            // Break when there is no more region/rule definition in PFM body
            break;
        }
    }

    return 1;
}

/**
 * @brief Perform validation on a PFM data
 *
 * @param pfm_ptr a pointer to the start of a PFM data
 * @return 1 if success else 0
 */
static alt_u32 is_pfm_valid(PFM* pfm)
{
    if (pfm->tag == PFM_MAGIC)
    {
        // Iterate through PFM SPI region and SMBus rule definitions and validate them
        return is_pfm_body_valid(pfm->pfm_body);
    }

    return 0;
}


#endif /* WHITLEY_INC_PFM_VALIDATION_H_ */
