// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file recovery_main.h
 * @brief Mainline function
 */

// Includes

// Always include pfr_sys first
#include "pfr_sys.h"

#include "initialization.h"
#include "transition.h"
#include "cpld_update.h"
#include "cpld_recovery.h"


/**
 * @brief Mainline of Recovery system. Called from main()
 *
 * [detailed description]
 *
 * @return None
 */
static PFR_ALT_INLINE void PFR_ALT_ALWAYS_INLINE recovery_main()
{
#ifndef PLATFORM_WILSON_CITY_FAB1
    // Assert sleep suspend right away
    // We do this to ensure we maintain power because we don't know what state the machine was in before reconfig
    set_bit(U_GPO_1_ADDR, GPO_1_FM_PFR_SLP_SUS_N);
#endif

    // Wait for ready from common core
    log_platform_state(PLATFORM_STATE_CPLD_NIOS_WAITING_TO_START);
    while (!check_ready_for_nios_start())
    {
        reset_hw_watchdog();
    }

    // Nios starts now
    log_platform_state(PLATFORM_STATE_CPLD_NIOS_STARTED);

    // Initialize the system
    initialize_system();

    // Prepare for transition to T-1 mode
    perform_entry_to_tmin1();

    prep_for_reconfig();

    // Check to see if we are in recovery because of a powercycle or a cpld update request
    // Note that the bread crumb may not be cleaned up if CPLD update failed (e.g. authentication failure)
    // If it was because of authentication failure, recovery image will just always perform authentication
    // on the CPLD update capsule and then switch to CFM1 once that fails. Authentication on CPLD update capsule
    // should be pretty quick.
    alt_u32* cfm1_breadcrumb = get_ufm_ptr_with_offset(CFM1_BREAD_CRUMB);
    if (read_reconfig_reason() == RECONFIG_CPLD_UPDATE && *cfm1_breadcrumb != 0)
    {
        // Switch to the active image if Nios gets here from a powercycle.
        perform_cfm_switch(CPLD_CFM1);
    }

    switch_spi_flash(SPI_FLASH_BMC);
    alt_u32 bmc_staging_offset = get_ufm_pfr_data()->bmc_staging_region;
    alt_u32 cpld_slota_image_offset_in_flash = bmc_staging_offset + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET;
    alt_u32 cpld_slotb_image_offset_in_flash = bmc_staging_offset + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET;

    alt_u32* slota_backup_ptr = get_spi_flash_ptr_with_offset(cpld_slota_image_offset_in_flash);
    alt_u32* slotb_backup_ptr = get_spi_flash_ptr_with_offset(cpld_slotb_image_offset_in_flash);

    // oldest_backup_addr and newest_backup_addr are expected to be overwritten.
    alt_u32 oldest_backup_addr = 0;
    alt_u32 newest_backup_addr = 0;
    alt_u32 num_valid_backups = 0;

    // Determine the state of the backups in BMC SPI flash
    // TODO this can be optimized by reading the internal version numbers into local ram because reads from spi flash are more expensive
    if (*slota_backup_ptr == 0xFFFFFFFF && *slotb_backup_ptr == 0xFFFFFFFF) // Both slots empty
    {
        oldest_backup_addr = cpld_slota_image_offset_in_flash;
        newest_backup_addr = cpld_slota_image_offset_in_flash;
    }
    else if (*slota_backup_ptr == 0xFFFFFFFF) // Only slot B valid
    {
        oldest_backup_addr = cpld_slota_image_offset_in_flash;
        newest_backup_addr = cpld_slotb_image_offset_in_flash;
        num_valid_backups = 1;
    }
    else if (*slotb_backup_ptr == 0xFFFFFFFF) // Only slot A valid
    {
        oldest_backup_addr = cpld_slotb_image_offset_in_flash;
        newest_backup_addr = cpld_slota_image_offset_in_flash;
        num_valid_backups = 1;
    }
    else if (*slota_backup_ptr > *slotb_backup_ptr) // Both slots are valid, figure out which one is oldest
    {
        oldest_backup_addr = cpld_slotb_image_offset_in_flash;
        newest_backup_addr = cpld_slota_image_offset_in_flash;
        num_valid_backups = 2;
    }
    else
    {
        oldest_backup_addr = cpld_slota_image_offset_in_flash;
        newest_backup_addr = cpld_slotb_image_offset_in_flash;
        num_valid_backups = 2;
    }

    if (read_reconfig_reason() == RECONFIG_CPLD_UPDATE)
    {
        log_platform_state(PLATFORM_STATE_CPLD_UPDATE_IN_RECOVERY_MODE);
        perform_update_post_reconfig(oldest_backup_addr, newest_backup_addr);
    }
    else
    {
        log_platform_state(PLATFORM_STATE_CPLD_RECOVERY_IN_RECOVERY_MODE);
        perform_cpld_recovery(oldest_backup_addr, newest_backup_addr, num_valid_backups);
    }
}

