// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file rfnvram_utils.h
 * @brief Utility functions to communicate with RFNVRAM IP.
 */

#ifndef WHITLEY_INC_RFNVRAM_UTILS_H_
#define WHITLEY_INC_RFNVRAM_UTILS_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "pfr_pointers.h"
#include "rfnvram.h"

/**
 * @brief This function reads the PIT password from a specific address in the RFNVRAM
 *
 * @param buffer the buffer where we save the read data
 */
static void read_pit_password_from_rfnvram(alt_u8* buffer)
{
    // Start the sequence (with write)
    // Lowest bit: High -> Read ; Low -> Write
    IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, 0x2DC);

    // Write most significant byte
    IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, RFNVRAM_PIT_PASSWORD_MSB_OFFSET);
    // Write least significant byte
    IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, RFNVRAM_PIT_PASSWORD_LSB_OFFSET);

    // Start to read
    IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, 0x2DD);

    alt_u32 write_word = 0x00;
    for (alt_u32 i = 0; i < RFNVRAM_PIT_PASSWORD_LENGTH; i++)
    {
        // Read another word
        if (i == RFNVRAM_PIT_PASSWORD_LENGTH - 1)
        {
            // Also put RFNVRAM to idle at the last read
            write_word |= RFNVRAM_IDLE_MASK;
        }
        IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, write_word);

        // Poll to get data
        while (IORD_32DIRECT(RFNVRAM_RX_FIFO_BYTES_LEFT, 0) == 0) {}

        // Read and save the word
        buffer[i] = IORD_32DIRECT(RFNVRAM_RX_FIFO, 0);
    }
}

/**
 * @brief This function writes the PIT password at a specific address in the RFNVRAM
 */
static void write_ufm_pit_password_to_rfnvram()
{
    // Start to write
    // Lowest bit: High -> Read ; Low -> Write
    IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, 0x2DC);

    // Write most significant byte
    IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, RFNVRAM_PIT_PASSWORD_MSB_OFFSET);
    // Write least significant byte
    IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, RFNVRAM_PIT_PASSWORD_LSB_OFFSET);

    // get the password from UFM and write it to RFNVRAM
    alt_u8* ufm_pit_password = (alt_u8*) get_ufm_pfr_data()->pit_password;
    for (alt_u32 byte_i = 0; byte_i < RFNVRAM_PIT_PASSWORD_LENGTH - 1; byte_i++)
    {
        IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, ufm_pit_password[byte_i]);
    }

    // Write the last byte and put RFNVRAM to idle
    IOWR_32DIRECT(U_RFNVRAM_SMBUS_MASTER_ADDR, 0, RFNVRAM_IDLE_MASK | ufm_pit_password[RFNVRAM_PIT_PASSWORD_LENGTH - 1]);
}

#endif /* WHITLEY_INC_RFNVRAM_UTILS_H_ */
