// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file t0_provisioning.h
 * @brief Responsible for handling UFM provisioning requests.
 */

#ifndef WHITLEY_INC_T0_PROVISIONING_H_
#define WHITLEY_INC_T0_PROVISIONING_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "cpld_reconfig.h"
#include "mailbox_utils.h"
#include "rfnvram_utils.h"
#include "platform_log.h"
#include "transition.h"
#include "ufm_rw_utils.h"
#include "ufm_utils.h"


/**
 * @brief Clear the Mailbox UFM Command Trigger register.
 */
static PFR_ALT_INLINE void PFR_ALT_ALWAYS_INLINE mb_clear_ufm_cmd_trigger()
{
    write_to_mailbox(MB_UFM_CMD_TRIGGER, 0);
}

/**
 * @brief Check whether bit[0] (execute command) is set in the Mailbox UFM Command Trigger register.
 *
 * @return alt_u32 1 if the bit[0] is set.
 */
static PFR_ALT_INLINE alt_u32 PFR_ALT_ALWAYS_INLINE mb_has_ufm_cmd_trigger()
{
    return read_from_mailbox(MB_UFM_CMD_TRIGGER) & MB_UFM_CMD_EXECUTE_MASK;
}

/**
 * @brief This function checks whether the given UFM command is illegal, given the current state of UFM.
 * Covered cases:
 * 1. If UFM is locked, any UFM command that attempt to modify UFM is illegal.
 * 2. If UFM is locked, Reconfig CPLD UFM command is illegal.
 * 3. If UFM is not provisioned with root key hash and BMC/PCH offsets, lock UFM command is illegal.
 * 4. If PIT password has not been provisioned, enabling PIT L1 is not allowed.
 *
 * @param ufm_cmd a UFM provisioning command read from the mailbox register.
 * @return alt_u32 1 if the given UFM command is illegal; 0, otherwise.
 */
static alt_u32 mb_is_ufm_cmd_illegal(alt_u32 ufm_cmd)
{
    if (is_ufm_locked())
    {
        if ((ufm_cmd == MB_UFM_PROV_RECONFIG_CPLD) ||
                (ufm_cmd == MB_UFM_PROV_ERASE) ||
                (ufm_cmd == MB_UFM_PROV_ROOT_KEY) ||
                (ufm_cmd == MB_UFM_PROV_PIT_PASSWORD) ||
                (ufm_cmd == MB_UFM_PROV_PCH_OFFSETS) ||
                (ufm_cmd == MB_UFM_PROV_BMC_OFFSETS) ||
                (ufm_cmd == MB_UFM_PROV_END) ||
                (ufm_cmd == MB_UFM_PROV_ENABLE_PIT_L1))
        {
            // Cannot modify UFM or reconfig CPLD after UFM has been locked
            // For MB_UFM_PROV_ENABLE_PIT_L1 command, attacker can use this command to get the PIT password if it's provisioned
            // If UFM is locked, it is assumed the PIT password is already provisioned and PIT level 1 is already enabled
            return 1;
        }
    }
    if (!is_ufm_provisioned() && ufm_cmd == MB_UFM_PROV_END)
    {
        // Cannot lock UFM unless root key hash and offsets are provisioned
        return 1;
    }
    if (!check_ufm_status(UFM_STATUS_PROVISIONED_PIT_PASSWORD_BIT_MASK) && ufm_cmd == MB_UFM_PROV_ENABLE_PIT_L1)
    {
        // Cannot enable PIT L1 protection unless PIT password is provisioned
        return 1;
    }

    // Otherwise, this ufm command is legal
    return 0;
}

/**
 * @brief This function processes any write UFM command.
 *
 * If the provisioning data has been provisioned, log an error and exit.
 * For example, once PIT password has been provisioned, it cannot be provisioned
 * again unless a erase provisioning command has been issued.
 *
 * All provisioned data is stored in UFM according to the offset in
 * UFM_PFR_DATA.
 *
 * @param ufm_cmd a UFM provisioning command read from the mailbox register.
 */
static void mb_process_write_ufm_cmd(alt_u32 ufm_cmd)
{
    UFM_PFR_DATA* ufm_data = get_ufm_pfr_data();
    if (ufm_cmd == MB_UFM_PROV_ERASE)
    {
        // Perform page erase on the first page (which contains the UFM PFR data)
        ufm_erase_page(UFM_PFR_DATA_OFFSET);
    }
    else if (ufm_cmd == MB_UFM_PROV_ROOT_KEY)
    {
        if (check_ufm_status(UFM_STATUS_PROVISIONED_ROOT_KEY_HASH_BIT_MASK))
        {
            // Root key hash has been provisioned. Must erase before provisioning again.
            mb_set_ufm_provision_status(MB_UFM_PROV_CMD_ERROR_MASK);
        }
        else
        {
            // Update internal provisioning status
            set_ufm_status(UFM_STATUS_PROVISIONED_ROOT_KEY_HASH_BIT_MASK);

            write_ufm_from_mb_fifo(ufm_data->root_key_hash, PFR_CRYPTO_LENGTH);
        }
    }
    else if (ufm_cmd == MB_UFM_PROV_PIT_PASSWORD)
    {
        if (check_ufm_status(UFM_STATUS_PROVISIONED_PIT_PASSWORD_BIT_MASK))
        {
            // PIT password has been provisioned. Must erase before provisioning again.
            mb_set_ufm_provision_status(MB_UFM_PROV_CMD_ERROR_MASK);
        }
        else
        {
            // Update internal provisioning status
            set_ufm_status(UFM_STATUS_PROVISIONED_PIT_PASSWORD_BIT_MASK);

            // PIT Key Provisioning
            write_ufm_from_mb_fifo(ufm_data->pit_password, RFNVRAM_PIT_PASSWORD_LENGTH);
        }
    }
    else if (ufm_cmd == MB_UFM_PROV_PCH_OFFSETS)
    {
        if (check_ufm_status(UFM_STATUS_PROVISIONED_PCH_OFFSETS_BIT_MASK))
        {
            // PCH offsets have been provisioned. Must erase before provisioning again.
            mb_set_ufm_provision_status(MB_UFM_PROV_CMD_ERROR_MASK);
        }
        else
        {
            set_ufm_status(UFM_STATUS_PROVISIONED_PCH_OFFSETS_BIT_MASK);

            // Read provisioned PCH offsets (Active/PFM, Recovery, Staging)
            // Write the offsets in the above order/ Those offsets are stored in the same order in UFM
            // Each offset is 4-byte long. Read/write 12 bytes in total
            write_ufm_from_mb_fifo(&ufm_data->pch_active_pfm, 12);
        }
    }
    else if (ufm_cmd == MB_UFM_PROV_BMC_OFFSETS)
    {
        if (check_ufm_status(UFM_STATUS_PROVISIONED_BMC_OFFSETS_BIT_MASK))
        {
            // BMC offsets have been provisioned. Must erase before provisioning again.
            mb_set_ufm_provision_status(MB_UFM_PROV_CMD_ERROR_MASK);
        }
        else
        {
            set_ufm_status(UFM_STATUS_PROVISIONED_BMC_OFFSETS_BIT_MASK);

            // Read provisioned BMC offsets (Active/PFM, Recovery, Staging)
            // Write the offsets in the above order/ Those offsets are stored in the same order in UFM
            // Each offset is 4-byte long. Read/write 12 bytes in total
            write_ufm_from_mb_fifo(&ufm_data->bmc_active_pfm, 12);
        }
    }
    else if (ufm_cmd == MB_UFM_PROV_END)
    {
        set_ufm_status(UFM_STATUS_LOCK_BIT_MASK);
        // Update mailbox UFM provisioning status
        mb_set_ufm_provision_status(MB_UFM_PROV_UFM_LOCKED_MASK);
    }

    // Update mailbox UFM provisioning status,
    // Set provisioned bit to 1, when root key hash and the BMC/PCH offsets are provisioned
    if (is_ufm_provisioned())
    {
        mb_set_ufm_provision_status(MB_UFM_PROV_UFM_PROVISIONED_MASK);
    }
    else
    {
        mb_clear_ufm_provision_status(MB_UFM_PROV_UFM_PROVISIONED_MASK);
    }
}

/**
 * @brief This function processes any read UFM command.
 *
 * @param ufm_cmd a UFM provisioning command read from the mailbox register.
 */
static void mb_process_read_ufm_cmd(alt_u32 ufm_cmd)
{
    UFM_PFR_DATA* ufm_data_ptr = get_ufm_pfr_data();
    if (ufm_cmd == MB_UFM_PROV_RD_ROOT_KEY)
    {
        write_mb_fifo_from_ufm((alt_u8*) ufm_data_ptr->root_key_hash, PFR_CRYPTO_LENGTH);
    }
    else if (ufm_cmd == MB_UFM_PROV_RD_PCH_OFFSETS)
    {
        // Read provisioned PCH offsets (Active/PFM, Recovery, Staging)
        // Write the offsets in the above order/ Those offsets are stored in the same order in UFM
        // Each offset is 4-byte long. Read/write 12 bytes in total
        write_mb_fifo_from_ufm((alt_u8*) &ufm_data_ptr->pch_active_pfm, 12);
    }
    else if (ufm_cmd == MB_UFM_PROV_RD_BMC_OFFSETS)
    {
        // Read provisioned BMC offsets (Active/PFM, Recovery, Staging)
        // Write the offsets in the above order/ Those offsets are stored in the same order in UFM
        // Each offset is 4-byte long. Read/write 12 bytes in total
        write_mb_fifo_from_ufm((alt_u8*) &ufm_data_ptr->bmc_active_pfm, 12);
    }
}

/**
 * @brief This function processes the PIT L1/L2 enabling UFM command.
 *
 * @param ufm_cmd a UFM provisioning command read from the mailbox register.
 */
static void mb_process_pit_ufm_cmd(alt_u32 ufm_cmd)
{
    if (ufm_cmd == MB_UFM_PROV_ENABLE_PIT_L1)
    {
        // Save the UFM PIT password to RFNVRAM
        write_ufm_pit_password_to_rfnvram();

        // Start checking PIT password in future T-1 mode
        set_ufm_status(UFM_STATUS_PIT_L1_ENABLE_BIT_MASK);

        // Report to UFM that PIT L1 has been enabled
        mb_set_ufm_provision_status(MB_UFM_PROV_UFM_PIT_L1_ENABLED_MASK);
    }
    else if (ufm_cmd == MB_UFM_PROV_ENABLE_PIT_L2)
    {
        // Once enabled, PIT L2 cannot be enabled again and the associated UFM command is blocked.
        if (!check_ufm_status(UFM_STATUS_PIT_L2_ENABLE_BIT_MASK))
        {
            set_ufm_status(UFM_STATUS_PIT_L2_ENABLE_BIT_MASK);

            // Go into T-1 mode and calculate firmware hash
            perform_platform_reset();
        }
    }
}

/**
 * @brief This function processes the Reconfig CPLD UFM command.
 * Nios firmware transitions to T-1 mode first, then perform CFM switch
 * to the current image. Hence, Nios firmware will be re-ran from the beginning.
 *
 * @param ufm_cmd a UFM provisioning command read from the mailbox register.
 */
static void mb_process_reconfig_cpld_ufm_cmd(alt_u32 ufm_cmd)
{
    if (ufm_cmd == MB_UFM_PROV_RECONFIG_CPLD)
    {
        // Reconfig to the active image (i.e. reboot with the current image)
        perform_entry_to_tmin1();
        perform_cfm_switch(CPLD_CFM1);
    }
}

/**
 * @brief This function responses to UFM command and sets appropriate provisioning status afterwards.
 *
 * If the bit[0] (execute command) is set, then proceed to respond to the UFM provisioning command.
 * First, Nios firmware clears the command trigger register in Mailbox and validate the UFM command.
 * Once validation passed, the UFM command is processed appropriately.
 *
 * UFM status register is updated during the provisioning flow as well. For example, if the UFM command
 * failed any validation check, the command error bit will be set. UFM command busy bit is set before
 * Nios firmware begins to process the UFM command. The busy bit is cleared once the UFM command has been
 * processed.
 */
static void mb_ufm_provisioning_handler()
{
    // Check command trigger
    if (mb_has_ufm_cmd_trigger())
    {
        // Clear done status and then command trigger
        mb_clear_ufm_provision_status(MB_UFM_PROV_CMD_DONE_MASK);
        mb_clear_ufm_cmd_trigger();

        // End if the UFM command is illegal
        alt_u32 ufm_cmd = read_from_mailbox(MB_PROVISION_CMD);
        if (mb_is_ufm_cmd_illegal(ufm_cmd))
        {
            mb_set_ufm_provision_status(MB_UFM_PROV_CMD_ERROR_MASK);
            mb_set_ufm_provision_status(MB_UFM_PROV_CMD_DONE_MASK);
        }
        else
        {
            // Update status: busy with the UFM command
            mb_set_ufm_provision_status(MB_UFM_PROV_CMD_BUSY_MASK);
            mb_clear_ufm_provision_status(MB_UFM_PROV_CMD_ERROR_MASK);

            // Process the UFM command
            mb_process_read_ufm_cmd(ufm_cmd);
            mb_process_reconfig_cpld_ufm_cmd(ufm_cmd);
            mb_process_write_ufm_cmd(ufm_cmd);
            mb_process_pit_ufm_cmd(ufm_cmd);

            // Update status: done with the UFM command
            mb_clear_ufm_provision_status(MB_UFM_PROV_CMD_BUSY_MASK);
            mb_set_ufm_provision_status(MB_UFM_PROV_CMD_DONE_MASK);
        }
    }
}

#endif /* WHITLEY_INC_T0_PROVISIONING_H_ */
