// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file t0_update.h
 * @brief Monitor the BMC and PCH update intent registers and trigger panic event appropriately.
 */

#ifndef WHITLEY_INC_T0_UPDATE_H_
#define WHITLEY_INC_T0_UPDATE_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "firmware_update.h"
#include "transition.h"
#include "mailbox_utils.h"
#include "platform_log.h"
#include "spi_flash_state.h"


/**
 * @brief Monitor update intent register for the @p spi_flash_type flash and transition to T-1 mode
 * to perform the update if necessary.
 *
 * This function reads the corresponding update intent from the mailbox first.
 *
 * If the update intent is not a valid request (i.e. not matching any bit mask), then exit.
 * If the update intent indicates a deferred update, also exit.
 *
 * For the following scenarios, log some error, clear the update intent register and exit.
 * - It's a firmware update and maximum failed update attempts have been reached for that flash device.
 * - It's a CPLD update and maximum failed CPLD update attempts have been reached.
 * - It's a active firmware update and the corresponding flash has a corrupted recovery image.
 *
 * Once all these checks have passed, Nios firmware proceed to trigger the update.
 * If it's a PCH firmware update or a recovery update, Nios firmware performs a platform reset.
 * Otherwise, Nios firmware performs a BMC-only reset. The actual update is done as part of the
 * T-1 operations.
 *
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 *
 * @see act_on_update_intent
 */
static void mb_update_intent_handler_for(SPI_FLASH_TYPE_ENUM spi_flash_type)
{
    MB_REGFILE_OFFSET_ENUM update_intent_offset = MB_PCH_UPDATE_INTENT;
    // Only process PCH firmware update from PCH
    MB_UPDATE_INTENT_MASK_ENUM is_fw_or_cpld_update_mask = MB_UPDATE_INTENT_PCH_FW_UPDATE_MASK;
    STATUS_LAST_PANIC_ENUM panic_event = LAST_PANIC_PCH_UPDATE_INTENT;
    if (spi_flash_type == SPI_FLASH_BMC)
    {
        update_intent_offset = MB_BMC_UPDATE_INTENT;
        // Process PCH/BMC firmware update and CPLD update from BMC
        is_fw_or_cpld_update_mask = MB_UPDATE_INTENT_FW_OR_CPLD_UPDATE_MASK;
        panic_event = LAST_PANIC_BMC_UPDATE_INTENT;
    }

    // Read the update intent
    alt_u32 update_intent = read_from_mailbox(update_intent_offset);

    // Check if there's any update intent.
    // If the update is not a deferred update, proceed to validate the update intent value.
    if (update_intent &&
            ((update_intent & MB_UPDATE_INTENT_UPDATE_AT_RESET_MASK) != MB_UPDATE_INTENT_UPDATE_AT_RESET_MASK))
    {
        /*
         * Validate the update intent
         */
        if ((update_intent & is_fw_or_cpld_update_mask) == 0)
        {
            // This update intent is not actionable or valid.
            log_update_failure(spi_flash_type, MINOR_ERROR_INVALID_UPDATE_INTENT);
            write_to_mailbox(update_intent_offset, 0);
            return;
        }

        if (block_fw_update_attempt(spi_flash_type) &&
                (update_intent & MB_UPDATE_INTENT_FW_UPDATE_MASK))
        {
            // If there are too many failed firmware update attempts, reject this update.
            log_update_failure(spi_flash_type, MINOR_ERROR_FW_UPDATE_EXCEEDED_MAX_FAILED_ATTEMPTS);
            write_to_mailbox(update_intent_offset, 0);
            return;
        }

        if ((num_failed_cpld_update_attempts >= MAX_CPLD_UPDATE_FAIL_ATTEMPTS) &&
                (update_intent & MB_UPDATE_INTENT_CPLD_MASK))
        {
            // If there are too many failed CPLD update attempts, reject this update.
            log_update_failure(spi_flash_type, MINOR_ERROR_CPLD_UPDATE_EXCEEDED_MAX_FAILED_ATTEMPTS);
            write_to_mailbox(update_intent_offset, 0);
            return;
        }

        if ((update_intent & MB_UPDATE_INTENT_FW_ACTIVE_UPDATE_MASK) &&
                check_spi_flash_state(spi_flash_type, SPI_FLASH_STATE_RECOVERY_FAILED_AUTH_MASK))
        {
            // Since both recovery image failed authentication, active update is blocked. Recovery update is required.
            log_update_failure(spi_flash_type, MINOR_ERROR_FW_UPDATE_ACTIVE_UPDATE_NOT_ALLOWED);
            write_to_mailbox(update_intent_offset, 0);
            return;
        }

        /*
         * Proceed to transition to T-1 mode and perform the update
         */
        // Log panic event
        log_panic(panic_event);

        // Enter T-1 mode to do the update there
        if (update_intent & MB_UPDATE_INTENT_REQUIRE_PLATFORM_RESET_MASK)
        {
            // Go into T-1 mode and perform the update there as part of the T-1 work.
            // Then transition back to T0.
            perform_platform_reset();
        }
        else
        {
            // Only bring down BMC if only BMC active firmware is being updated
            perform_bmc_only_reset();
        }
    }
}

/**
 * @brief Monitor the PCH and BMC update intent fields in the mailbox register file.
 *
 * Run mb_update_intent_handler_for on PCH first then on BMC. This is because if there's
 * a PCH firmware update, both PCH and BMC will be transitioned to T-1 mode. If there's any
 * update request(s) from BMC, it can be processed in the same T-1 cycle.
 *
 * @see mb_update_intent_handler_for(SPI_FLASH_TYPE_ENUM)
 */
static void mb_update_intent_handler()
{
    // Process PCH update intent first.
    // If there's also BMC update, it can be processed in the same platform reset T-1 cycle as well
    mb_update_intent_handler_for(SPI_FLASH_PCH);
    mb_update_intent_handler_for(SPI_FLASH_BMC);
}

/**
 * @brief Events after a successful active update.
 *
 * If an active firmware has completed successfully, update the SVN policy in UFM (i.e.
 * bump the minimum SVN to match the SVN of active firmware PFM). In addition, if there's
 * a pending recovery update, trigger platform reset to perform the update as part of the T-1
 * operations.
 *
 * These steps are done for both PCH and BMC flashes.
 */
static void post_update_routine()
{
    alt_u32 do_platform_reset = 0;
    STATUS_LAST_PANIC_ENUM panic_reason = LAST_PANIC_BMC_UPDATE_INTENT;

    // Check if there is any active/recovery update in progress for BMC
    if (check_spi_flash_state(SPI_FLASH_BMC, SPI_FLASH_STATE_ACTIVE_UPDATE_IN_PROGRESS_MASK))
    {
        // Active update was successful. Bump up the SVN.
        write_ufm_svn(fw_active_update_svn_for_bmc, UFM_SVN_POLICY_BMC);
        // Active update is complete. Clear the SPI state.
        clear_spi_flash_state(SPI_FLASH_BMC, SPI_FLASH_STATE_ACTIVE_UPDATE_IN_PROGRESS_MASK);

        // Now that active firmware has booted up successful, it can be promoted to the recovery region.
        // Check if there's any recovery update in progress
        if (check_spi_flash_state(SPI_FLASH_BMC, SPI_FLASH_STATE_RECOVERY_UPDATE_IN_PROGRESS_MASK))
        {
            do_platform_reset = 1;
        }
    }

    // Check if there is any active/recovery update in progress for PCH
    if (check_spi_flash_state(SPI_FLASH_PCH, SPI_FLASH_STATE_ACTIVE_UPDATE_IN_PROGRESS_MASK))
    {
        // Active update was successful. Bump up the SVN.
        write_ufm_svn(fw_active_update_svn_for_pch, UFM_SVN_POLICY_PCH);
        // Active update is complete. Clear the SPI state.
        clear_spi_flash_state(SPI_FLASH_PCH, SPI_FLASH_STATE_ACTIVE_UPDATE_IN_PROGRESS_MASK);

        // Now that active firmware has booted up successful, it can be promoted to the recovery region.
        // Check if there's any recovery update in progress
        if (check_spi_flash_state(SPI_FLASH_PCH, SPI_FLASH_STATE_RECOVERY_UPDATE_IN_PROGRESS_MASK))
        {
            panic_reason = LAST_PANIC_PCH_UPDATE_INTENT;
            do_platform_reset = 1;
        }
    }

    // Nios has set the proper SPI state for both BMC and PCH at this point.
    // If there are two recovery updates, they will be performed in the same T-1 cycle.
    if (do_platform_reset)
    {
        // Recovery update will be performed in T-1
        log_panic(panic_reason);
        perform_platform_reset();
    }
}

#endif /* WHITLEY_INC_T0_UPDATE_H_ */
