// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file t0_watchdog_handler.h
 * @brief Responsible for monitoring timed boot progress and handling timeouts of watchdog timers.
 */

#ifndef WHITLEY_INC_T0_WATCHDOG_HANDLER_H_
#define WHITLEY_INC_T0_WATCHDOG_HANDLER_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "firmware_recovery.h"
#include "gen_gpo_controls.h"
#include "gen_gpi_signals.h"
#include "mailbox_utils.h"
#include "platform_log.h"
#include "spi_ctrl_utils.h"
#include "transition.h"
#include "utils.h"
#include "watchdog_timers.h"

// Forward declarations
static void bmc_watchdog_timer_handler();
static void me_watchdog_timer_handler();
static void acm_bios_watchdog_timer_handler();

/**
 * @brief Responsible for monitoring the state of all the SW watchdog timers.
 *
 * @see bmc_watchdog_timer_handler
 * @see me_watchdog_timer_handler
 * @see acm_bios_watchdog_timer_handler
 */
static void watchdog_routine()
{
    bmc_watchdog_timer_handler();
    me_watchdog_timer_handler();
    acm_bios_watchdog_timer_handler();
}

/**
 * @brief Respond to PAUSE or RESUME checkpoint message.
 *
 * If @p chkpt_cmd matches the PAUSE command, then pause the watchdog timer.
 * If @p chkpt_cmd matches the RESUME command, then resume the watchdog timer.
 *
 * @param chkpt_cmd Checkpoint command read
 * @param timer_addr pointer to the watchdog timer in the timer bank
 */
static void pause_or_resume_wdtimer(alt_u32 chkpt_cmd, alt_u32* timer_addr)
{
    if (chkpt_cmd == MB_CHKPT_PAUSE)
    {
        pause_timer(timer_addr);
    }
    else if (chkpt_cmd == MB_CHKPT_RESUME)
    {
        resume_timer(timer_addr);
    }
}

/**
 * @brief Trigger a WDT recovery on PCH SPI flash, when Nios FW observes a ME/ACM/BIOS watchdog timer timeout
 * or ME/ACM/BIOS internal authentication failure.
 *
 * If the current recovery level is still within 1-3 (inclusive), then proceed to T-1 mode and perform WDT recovery.
 * Otherwise, simply disable the appropriate watchdog timer and let the firmware hang.
 *
 * @param last_recovery_reason The specific recovery reason for the WDT recovery
 * @param panic_reason The specific panic reason for the transition to T-1 mode
 *
 * @see perform_wdt_recovery
 */
static void trigger_pch_wdt_recovery(
        STATUS_LAST_RECOVERY_ENUM last_recovery_reason, STATUS_LAST_PANIC_ENUM panic_reason)
{
    if (get_fw_recovery_level(SPI_FLASH_PCH) & SPI_REGION_PROTECT_MASK_RECOVER_BITS)
    {
        log_wdt_recovery(last_recovery_reason, panic_reason);
        set_spi_flash_state(SPI_FLASH_PCH, SPI_FLASH_STATE_REQUIRE_WDT_RECOVERY_MASK);
        perform_platform_reset();
    }
    else
    {
        wdt_enable_status &= WDT_DISABLE_PCH_TIMERS_MASK;
    }
}

/**
 * @brief Trigger a WDT recovery on BMC SPI flash, when Nios FW observes a BMC watchdog timer timeout.
 *
 * If the current recovery level is still within 1-3 (inclusive), then do a BMC-only reset and perform WDT recovery.
 * Otherwise, simply disable the BMC watchdog timer and let the firmware hang.
 *
 * @param last_recovery_reason The specific recovery reason for the WDT recovery
 * @param panic_reason The specific panic reason for the transition to T-1 mode
 *
 * @see perform_wdt_recovery
 */
static void trigger_bmc_wdt_recovery(
        STATUS_LAST_RECOVERY_ENUM last_recovery_reason, STATUS_LAST_PANIC_ENUM panic_reason)
{
    if (get_fw_recovery_level(SPI_FLASH_BMC) & SPI_REGION_PROTECT_MASK_RECOVER_BITS)
    {
        log_wdt_recovery(last_recovery_reason, panic_reason);
        set_spi_flash_state(SPI_FLASH_BMC, SPI_FLASH_STATE_REQUIRE_WDT_RECOVERY_MASK);
        perform_bmc_only_reset();
    }
    else
    {
        wdt_enable_status &= WDT_DISABLE_BMC_TIMER_MASK;
    }
}

/**
 * @brief Monitor the boot progress for ME firmware with the ME GPIOs.
 *
 * If the ME watchdog timer expires, indicate in the PCH spi flash state global
 * variable that WDT recovery is required for PCH flash. Then perform a platform
 * reset. The WDT recovery will be done in the T-1 stage of that reset.
 *
 * GPIO 1 (FM_ME_PFR_1): 1 means ME Authentication Failed
 * GPIO 2 (FM_ME_PFR_2): 1 means ME Boot Done
 *
 * If ME GPIO 2 is == 1 and ME GPIO 1 is == 0, then ME firmware has completed boot
 * successfully. ME watchdog timer becomes inactive.
 *
 * @see watchdog_timeout_handler
 */
static void me_watchdog_timer_handler()
{
    // Skip if this timer is disabled or inactive.
    if ((wdt_enable_status & WDT_ENABLE_ME_TIMER_MASK) &&
            check_bit(WDT_ME_TIMER_ADDR, U_TIMER_BANK_TIMER_ACTIVE_BIT))
    {
        // If ME timer is enabled
        if (is_wd_timer_expired(WDT_ME_TIMER_ADDR))
        {
            trigger_pch_wdt_recovery(LAST_RECOVERY_ME_LAUNCH_FAIL, LAST_PANIC_ME_WDT_EXPIRED);
        }
        else
        {
            // GPIO 1: ME Authentication Failed
            // GPIO 2: ME Boot Done
            if (check_bit(U_GPI_1_ADDR, GPI_1_FM_ME_PFR_2) &&
                    !check_bit(U_GPI_1_ADDR, GPI_1_FM_ME_PFR_1))
            {
                // When ME firmware booted and authentication pass, stop the ME timer
                wdt_boot_status |= WDT_ME_BOOT_DONE_MASK;

                // Clear the ME timer
                IOWR(WDT_ME_TIMER_ADDR, 0, 0);

                // Clear the fw recovery level upon successful boot of BIOS and ME
                if (wdt_boot_status & WDT_IBB_BOOT_DONE_MASK)
                {
                    reset_fw_recovery_level(SPI_FLASH_PCH);
                }

                // Log boot progress
                log_t0_timed_boot_complete_if_ready(PLATFORM_STATE_T0_ME_BOOTED);
            }
        }
    }
}

/**
 * @brief Monitor the boot progress for BMC firmware with the BMC checkpoint message.
 *
 * If the BMC watchdog timer expires, indicate in the BMC spi flash state global
 * variable that WDT recovery is required for BMC flash. Then perform a BMC only
 * reset. The WDT recovery will be done in the T-1 stage of that reset.
 *
 * BMC boot flow contains two parts: Bootloader and kernel.
 * Currently, the bootloader sends the BLOCK_START checkpoint message and the kernel sends
 * BLOCK_COMPLETE checkpoint message. Hence, only 1 timeout value is used to track BMC
 * boot progress.
 *
 * SMBus filtering is enabled after BMC has completed boot. SMBus filtering rules were
 * stored during T-1 authentication flow.
 *
 * @see watchdog_timeout_handler
 */
static void bmc_watchdog_timer_handler()
{
    // Skip if this timer is disabled or inactive.
    if ((wdt_enable_status & WDT_ENABLE_BMC_TIMER_MASK) &&
            check_bit(WDT_BMC_TIMER_ADDR, U_TIMER_BANK_TIMER_ACTIVE_BIT))
    {
        if (is_wd_timer_expired(WDT_BMC_TIMER_ADDR))
        {
            trigger_bmc_wdt_recovery(LAST_RECOVERY_BMC_LAUNCH_FAIL, LAST_PANIC_BMC_WDT_EXPIRED);
        }
        else
        {
            // Read checkpoint command
            alt_u32 chkpt_cmd = read_from_mailbox(MB_BMC_CHECKPOINT);

            // Clear checkpoint messages after read
            write_to_mailbox(MB_BMC_CHECKPOINT, 0);

            if (chkpt_cmd == MB_CHKPT_START)
            {
                // Start the watchdog timer
                // This would simply restart the watchdog timer if it's already started
                start_timer_ms(WDT_BMC_TIMER_ADDR, WD_BMC_EXPIRE_IN_MS);
            }
            else if (chkpt_cmd == MB_CHKPT_COMPLETE)
            {
                // BMC has completed boot
                wdt_boot_status |= WDT_BMC_BOOT_DONE_MASK;

                // Clear the BMC timer
                IOWR(WDT_BMC_TIMER_ADDR, 0, 0);

                // Clear the fw recovery level upon successful boot
                reset_fw_recovery_level(SPI_FLASH_BMC);

                // Enable SMBus filtering
                set_filter_disable_all(0);

                // Clear IBB detection. It was set during BMC boot up process.
                set_bit(U_GPO_1_ADDR, GPO_1_BMC_SPI_CLEAR_IBB_DETECTED);

                // Log boot progress
                log_t0_timed_boot_complete_if_ready(PLATFORM_STATE_T0_BMC_BOOTED);
            }

            // Process PAUSE or RESUME checkpoint message
            pause_or_resume_wdtimer(chkpt_cmd, WDT_BMC_TIMER_ADDR);
        }
    }
}

/**
 * @brief Monitor the boot progress for ACM and BIOS firmware with the ACM and BIOS checkpoint message.
 *
 * A single watchdog timer is used for tracking both ACM and BIOS boot progress, because of the ACM ->
 * IBB -> OBB multi-level secure boot flow.
 *
 * If the ACM BIOS watchdog timer expires, indicate in the PCH spi flash state global
 * variable that WDT recovery is required for PCH flash. Then perform a BMC platform
 * reset. The WDT recovery will be done in the T-1 stage of that reset.
 *
 * ACM BIOS boot flow contains three parts: ACM, IBB and OBB.
 * For ACM boot progress, Nios firmware monitors the ACM checkpoint message register in the mailbox.
 * Once BOOT DONE/COMPLETE is received, Nios firmware restarts the watchdog timer with the IBB firmware
 * expected timeout value. It also switches to monitor the BIOS checkpoint message register. If BOOT
 * DONE/COMPLETE is received, then Nios firmware restarts the watchdog timer with the OBB firmware
 * expected timeout value. If the final BOOT DONE/COMPLETE is received, the PCH has completed boot and
 * its watchdog timer becomes inactive.
 *
 * @see watchdog_timeout_handler
 */
static void acm_bios_watchdog_timer_handler()
{
    // Skip if this timer is disabled or inactive.
    if ((wdt_enable_status & WDT_ENABLE_ACM_BIOS_TIMER_MASK) &&
            check_bit(WDT_ACM_BIOS_TIMER_ADDR, U_TIMER_BANK_TIMER_ACTIVE_BIT))
    {
        // ACM/BIOS timer is enabled.
        alt_u32 wdtimer_expired = is_wd_timer_expired(WDT_ACM_BIOS_TIMER_ADDR);
        alt_u32 chkpt_cmd = read_from_mailbox(MB_BIOS_CHECKPOINT);

        // Clear checkpoint messages after read
        write_to_mailbox(MB_BIOS_CHECKPOINT, 0);

        // Three-stage boot: ACM -> IBB -> OBB
        if (wdt_boot_status & WDT_IBB_BOOT_DONE_MASK)
        {
            // Both ACM and IBB have booted. Tracking OBB boot progress now
            if (wdtimer_expired)
            {
                trigger_pch_wdt_recovery(LAST_RECOVERY_OBB_LAUNCH_FAIL, LAST_PANIC_OBB_WDT_EXPIRED);
            }
            else if (chkpt_cmd == MB_CHKPT_START)
            {
                // Restart OBB timer (initially started after IBB booted)
                start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, WD_OBB_EXPIRE_IN_MS);
            }
            else if (chkpt_cmd == MB_CHKPT_COMPLETE)
            {
                // BIOS OBB boot has completed
                wdt_boot_status |= WDT_OBB_BOOT_DONE_MASK;
                // Clear the ACM/BIOS timer
                IOWR(WDT_ACM_BIOS_TIMER_ADDR, 0, 0);

                // Clear the fw recovery level upon successful boot of BIOS and ME
                if (wdt_boot_status & WDT_ME_BOOT_DONE_MASK)
                {
                    reset_fw_recovery_level(SPI_FLASH_PCH);
                }

                // Log boot progress
                log_t0_timed_boot_complete_if_ready(PLATFORM_STATE_T0_BIOS_BOOTED);
            }
            else if (chkpt_cmd == MB_CHKPT_AUTH_FAIL)
            {
                trigger_pch_wdt_recovery(LAST_RECOVERY_OBB_LAUNCH_FAIL, LAST_PANIC_ACM_IBB_OBB_AUTH_FAILED);
            }
        }
        else if (wdt_boot_status & WDT_ACM_BOOT_DONE_MASK)
        {
            // ACM has booted. Tracking BIOS IBB boot progress now
            if (wdtimer_expired)
            {
                trigger_pch_wdt_recovery(LAST_RECOVERY_IBB_LAUNCH_FAIL, LAST_PANIC_IBB_WDT_EXPIRED);
            }
            else if (chkpt_cmd == MB_CHKPT_START)
            {
                // Restart the IBB timer (initially started after ACM booted)
                start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, WD_IBB_EXPIRE_IN_MS);
            }
            else if (chkpt_cmd == MB_CHKPT_COMPLETE)
            {
                // BIOS IBB boot has completed
                wdt_boot_status |= WDT_IBB_BOOT_DONE_MASK;
                // Start the BIOS OBB timer.
                start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, WD_OBB_EXPIRE_IN_MS);
            }
            else if (chkpt_cmd == MB_CHKPT_AUTH_FAIL)
            {
                trigger_pch_wdt_recovery(LAST_RECOVERY_IBB_LAUNCH_FAIL, LAST_PANIC_ACM_IBB_OBB_AUTH_FAILED);
            }
        }
        else
        {
            // Start tracking boot progress for ACM (then BIOS)
            chkpt_cmd = read_from_mailbox(MB_ACM_CHECKPOINT);

            // Clear checkpoint messages after read
            write_to_mailbox(MB_ACM_CHECKPOINT, 0);

            if (wdtimer_expired)
            {
                trigger_pch_wdt_recovery(LAST_RECOVERY_ACM_LAUNCH_FAIL, LAST_PANIC_ACM_WDT_EXPIRED);
            }
            else if (chkpt_cmd == MB_CHKPT_START)
            {
                // Restart the ACM timer.
                start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, WD_ACM_DONE_EXPIRE_IN_MS);
            }
            else if (chkpt_cmd == MB_CHKPT_COMPLETE)
            {
                // ACM has completed booting
                wdt_boot_status |= WDT_ACM_BOOT_DONE_MASK;
                // Start the BIOS IBB timer.
                start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, WD_IBB_EXPIRE_IN_MS);

                // Log boot progress
                log_platform_state(PLATFORM_STATE_T0_ACM_BOOTED);
            }
            else if (chkpt_cmd == MB_CHKPT_AUTH_FAIL)
            {
                // When there's BTG (ACM) authentication failure, ACM will pass that information to ME firmware.
                // Wait for ME firmware to clean up and shutdown system.
                // sleep_ms uses ACM_BIOS timer, so there's no conflict with other watchdog timer.
                sleep_ms(WD_ACM_AUTH_FAILURE_WAIT_TIME_IN_MS);

                trigger_pch_wdt_recovery(LAST_RECOVERY_ACM_LAUNCH_FAIL, LAST_PANIC_ACM_IBB_OBB_AUTH_FAILED);
            }
        }

        // Process PAUSE or RESUME checkpoint message
        pause_or_resume_wdtimer(chkpt_cmd, WDT_ACM_BIOS_TIMER_ADDR);
    }
}

#endif /* WHITLEY_INC_T0_WATCHDOG_HANDLER_H_ */
