// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file timer_utils.h
 * @brief Utility functions for using system timer bank.
 */

#ifndef WHITLEY_INC_TIMER_UTILS_H_
#define WHITLEY_INC_TIMER_UTILS_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "utils.h"

// Timer Bank
// A bank of 1ms timers. Each word represents an independent timer.
// The timer value is bits 19:0. The start/stop bit is bit 28.
// There are 3 timers currently available on the timer bank
#define U_TIMER_BANK_TIMER1_ADDR  __IO_CALC_ADDRESS_NATIVE_ALT_U32(U_TIMER_BANK_AVMM_BRIDGE_BASE, 0)
#define U_TIMER_BANK_TIMER2_ADDR  __IO_CALC_ADDRESS_NATIVE_ALT_U32(U_TIMER_BANK_AVMM_BRIDGE_BASE, 1)
#define U_TIMER_BANK_TIMER3_ADDR  __IO_CALC_ADDRESS_NATIVE_ALT_U32(U_TIMER_BANK_AVMM_BRIDGE_BASE, 2)

#define U_TIMER_BANK_TIMER_VALUE_MASK    0x000FFFFF
#define U_TIMER_BANK_TIMER_ACTIVE_MASK   0x10000000
#define U_TIMER_BANK_TIMER_ACTIVE_BIT    28


/**
 * @brief Start a count down of @p ms_time miliseconds with the timer at @p timer_base_addr in 
 * timer bank. 
 * 
 * @param timer_base_addr pointer to a timer register in the timer bank
 * @param ms_time count down value, in terms of miliseconds
 */
static void start_timer_ms(alt_u32* timer_base_addr, alt_u32 ms_time)
{
    IOWR(timer_base_addr, 0, (ms_time & U_TIMER_BANK_TIMER_VALUE_MASK) | U_TIMER_BANK_TIMER_ACTIVE_MASK);
}

/**
 * @brief Check if the given timer has expired.
 *
 * @param timer_base_addr pointer to a timer register in the timer bank
 * @return alt_u32 1 if the given timer has expired; 0, otherwise.
 */
static alt_u32 is_timer_expired(alt_u32* timer_base_addr)
{
    return (IORD(timer_base_addr, 0) & U_TIMER_BANK_TIMER_VALUE_MASK) == 0;
}

/**
 * @brief Pause the Nios processor and sleep for @p ms_time milliseconds.
 * This function should only be used in T-1 mode. Otherwise, it would interfere with the watchdog timers.
 *
 * @param ms_time number of milliseconds to sleep.
 */
static void sleep_ms(alt_u32 ms_time)
{
    start_timer_ms(U_TIMER_BANK_TIMER3_ADDR, ms_time);
    while (!is_timer_expired(U_TIMER_BANK_TIMER3_ADDR)) {}
    IOWR(U_TIMER_BANK_TIMER3_ADDR, 0, 0);
}

/**
 * @brief Pause the given timer
 * @param timer_base_addr pointer to a timer register in the timer bank
 */
static PFR_ALT_INLINE void PFR_ALT_ALWAYS_INLINE pause_timer(alt_u32* timer_base_addr)
{
    clear_bit(timer_base_addr, U_TIMER_BANK_TIMER_ACTIVE_BIT);
}

/**
 * @brief Resume the given timer
 * @param timer_base_addr pointer to a timer register in the timer bank
 */
static PFR_ALT_INLINE void PFR_ALT_ALWAYS_INLINE resume_timer(alt_u32* timer_base_addr)
{
    set_bit(timer_base_addr, U_TIMER_BANK_TIMER_ACTIVE_BIT);
}

#endif /* WHITLEY_INC_TIMER_UTILS_H_ */
