// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file tmin1_routines.h
 * @brief functions that are executed during pre-boot mode (T-1) of operation.
 */

#ifndef WHITLEY_INC_TMIN1_ROUTINES_H_
#define WHITLEY_INC_TMIN1_ROUTINES_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "flash_validation.h"
#include "gen_gpo_controls.h"
#include "gen_gpi_signals.h"
#include "pfm_utils.h"
#include "pit_utils.h"
#include "smbus_relay_utils.h"
#include "spi_ctrl_utils.h"
#include "timer_utils.h"
#include "tmin1_update.h"
#include "watchdog_timers.h"

/**
 * @brief Release SPI control on BMC flash back to BMC and release BMC from reset.
 * If the system is provisioned, start the BMC watchdog timer. 
 *
 * Nios FW releases BMC SPI control by clearing FM_SPI_PFR_BMC_BT_MASTER_SEL mux. Prior to clearing the mux,
 * Nios sends "exit 4-byte addressing mode" command to BMC SPI flash. Nios also notifies the BMC SPI filter
 * that BMC SPI flash is in 3-byte addressing mode. This is only required in platform reset. BMC can boot if
 * its flash is in 3-byte addressing mode, but PCH won't.
 *
 * This function should only be used in platform reset. BMC-only reset should not run this function.
 */
static void tmin1_boot_bmc()
{
    /*
     * Release the BMC SPI control, with BMC flash in 3B addressing mode
     */
    switch_spi_flash(SPI_FLASH_BMC);

    // Exit 4-byte addressing mode before releasing SPI control of BMC flash
    // Write enable command is required prior to sending the enter/exit 4-byte mode commands
    execute_one_byte_spi_cmd(SPI_CMD_WRITE_ENABLE);
    execute_one_byte_spi_cmd(SPI_CMD_EXIT_4B_ADDR_MODE);

    // The BMC SPI flash device should now be in 3-byte addressing mode.
    // Notify the BMC SPI filter of this change.
    set_bit(U_GPO_1_ADDR, GPO_1_BMC_SPI_ADDR_MODE_SET_3B);
    clear_bit(U_GPO_1_ADDR, GPO_1_BMC_SPI_ADDR_MODE_SET_3B);

    // Flip the external mux
    release_spi_ctrl(SPI_FLASH_BMC);

    /*
     * Release BMC from reset and start the BMC watchdog timer
     */
    set_bit(U_GPO_1_ADDR, GPO_1_RST_SRST_BMC_PLD_R_N);

    if (is_ufm_provisioned())
    {
        // Clear any previous BMC boot done status
        wdt_boot_status &= WDT_CLEAR_BMC_BOOT_DONE_MASK;

        // Start the BMC watchdog timer
        start_timer_ms(WDT_BMC_TIMER_ADDR, WD_BMC_EXPIRE_IN_MS);
    }
}

/**
 * @brief Perform all required operations to boot PCH. 
 * 
 * Steps: 
 * 1. Release the PCH SPI flash control back to PCH. The PCH SPI flash is in 4-byte addressing mode.
 * 2. Drive Z to DSWPWROK
 * 3. Wait for at least 100 ms to add some delays between setting DSW_PWROK and SLP_SUS_N. 
 * 4. Drive 0 to SLP_SUS_N
 * 5. Release PCH from reset (by driving 1 to RSMRST#)
 * 6. Start the ME and ACM watchdog timers, if the system is provisioned.
 */
static void tmin1_boot_pch()
{
    // Release SPI flash control to PCH
    release_spi_ctrl(SPI_FLASH_PCH);

    // Clear Deep SX. This will drive Z onto PWRGD_DSW_PWROK_R
    clear_bit(U_GPO_1_ADDR, GPO_1_PWRGD_DSW_PWROK_R);

    // Add some delays between DSW_PWROK and PFR_SLP_SUS
    // The max time for this transition is 100ms. Set a delay of 110ms to be safe.
    // If this delay is not added or is too small, platform would enter
    //   a power loop (constantly shutting down and then power up).
    sleep_ms(110);

#ifndef PLATFORM_WILSON_CITY_FAB1
    // Set PFR_SLP_SUS_N (sleep suspend) low
    clear_bit(U_GPO_1_ADDR, GPO_1_FM_PFR_SLP_SUS_N);
#endif

    // Release PCH from reset (triggers SPS/ME boot)
    set_bit(U_GPO_1_ADDR, GPO_1_RST_RSMRST_PLD_R_N);

    // TODO: Un-comment the following block after ME start driving the GPIOs to report boot progress
    /*
    if (is_ufm_provisioned())
    {
        // Clear any previous ME boot done status
        wdt_boot_status &= WDT_CLEAR_ME_BOOT_DONE_MASK;

        // Start the ME watchdog timer
        start_timer_ms(WDT_ME_TIMER_ADDR, WD_ME_EXPIRE_IN_MS);
    }
    */
}

/**
 * @brief Boot BMC by calling tmin1_boot_bmc(), then boot PCH by calling tmin1_boot_pch().
 * 
 * @see tmin1_boot_bmc()
 * @see tmin1_boot_pch()
 */
static void tmin1_boot_bmc_and_pch()
{
    tmin1_boot_bmc();
    tmin1_boot_pch();
}

/**
 * @brief Perform BMC-specific T-1 operations
 *
 * Process any BMC update requested through the BMC update intent. 
 * Perform authentication and recovery of all critical regions
 * of BMC FW storage. If the active firmware passed authentication, 
 * enable BMC SPI write filtering and store SMBus command filtering rules.
 */
static void perform_tmin1_operations_bmc_only()
{
#ifdef USE_SYSTEM_MOCK
    if (SYSTEM_MOCK::get()->should_exec_code_block(
            SYSTEM_MOCK::CODE_BLOCK_TYPES::SKIP_TMIN1_OPERATIONS))
    {
        return;
    }
#endif
    // Perform WDT recovery if there was a watchdog timeout in the previous T0 mode.
    perform_wdt_recovery(SPI_FLASH_BMC);

    // Check if there's BMC active firmware update request.
    alt_u32 bmc_update_intent = read_from_mailbox(MB_BMC_UPDATE_INTENT);
    if (bmc_update_intent & MB_UPDATE_INTENT_BMC_ACTIVE_MASK)
    {
        // If there's BMC active firmware update intent, that's probably the reason why Nios entered this BMC-only T-1 mode.
        // If there's any other type of update, Nios FW should have triggered platform reset.
        // In case some other update intent sneaks in when Nios has decided to perform BMC only reset, only keep the update
        // intents that are valid in this mode.
        write_to_mailbox(MB_BMC_UPDATE_INTENT, bmc_update_intent & MB_UPDATE_INTENT_VALID_FOR_BMC_ONLY_TMIN1_MASK);
        act_on_bmc_update_intent();
    }

    // Perform authentication and possibly recovery on the BMC flash
    authenticate_and_recover_spi_flash(SPI_FLASH_BMC);
}

/**
 * @brief Perform T-1 operations on both BMC and PCH SPI flashes. 
 *
 * Nios firmware performs these operations in order:
 * 1. Check protect-in-transit feature. If the L1 protection is enabled, Nios firmware
 * attempts to get the PIT password from the RFNVRAM and check it against the provisioned
 * PIT password in UFM. If the L2 protection is enabled, Nios firmware will compute the 
 * firmware hashes for both BMC and PCH flashes. The calculated hashes are compared against
 * the stored hashes in UFM. If either L1 or L2 check failed, Nios firmware enters a lockdown 
 * mode in T-1. 
 * 2. If there was a watchdog timer timeout in the past T0 cycle, Nios firmware performs WDT 
 * recovery for that component. @perform_wdt_recovery for more details on WDT recovery. 
 * 3. Perform any update as indicated in PCH and BMC update intent registers.
 * 4. Perform authentication and recovery of all critical regions of platform FW storage (PCH flash and BMC flash). 
 * If the active firmware is valid, Nios firmware enables the SPI filtering and store SMBus command filtering rules
 * according to the active PFM.
 */
static void perform_tmin1_operations()
{
#ifdef USE_SYSTEM_MOCK
    if (SYSTEM_MOCK::get()->should_exec_code_block(
            SYSTEM_MOCK::CODE_BLOCK_TYPES::SKIP_TMIN1_OPERATIONS))
    {
        return;
    }
#endif

    // If a level of Protect-in-Transit is enabled, perform appropriate protection
    perform_pit_protection();

    // Perform WDT recovery if there was a watchdog timeout in the previous T0 mode.
    perform_wdt_recovery(SPI_FLASH_PCH);
    perform_wdt_recovery(SPI_FLASH_BMC);

    // Process FW/CPLD updates
    process_updates_in_tmin1();

    // Perform authentication and possibly recovery on the BMC flash
    authenticate_and_recover_spi_flash(SPI_FLASH_BMC);

    // Perform authentication and possibly recovery on the PCH flash
    authenticate_and_recover_spi_flash(SPI_FLASH_PCH);
}

#endif /* WHITLEY_INC_TMIN1_ROUTINES_H_ */
