// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file tmin1_update.h
 * @brief In T-1, perform update action as requested in the BMC/PCH update intent.
 */

#ifndef WHITLEY_INC_TMIN1_UPDATE_H_
#define WHITLEY_INC_TMIN1_UPDATE_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "capsule_validation.h"
#include "mailbox_utils.h"
#include "spi_flash_state.h"


/**
 * @brief Read the BMC update intent and perform firmware or CPLD update accordingly.
 * 
 * Nios firmware clears the update intent register to 0, after reading from it.
 * It is expected that the update intent value has been validated in the T0 (in mb_update_intent_handler()).
 * If the update intent is invalid, Nios firmware wouldn't transition to T-1 mode to perform the update.  
 * 
 * The update intent is a bit mask. Hence, multiple updates can be requested together.
 * In the current release, only BMC flash has reserved area for multiple capsule for BMC/PCH/CPLD updates.
 * This function can process all these requests in the same T-1 cycle.
 *
 * This function looks for a match with any firmware update encoding first. BMC firmware update is processed first.
 * If there's any PCH firmware update, Nios Firmware copies the update capsule to PCH flash staging region and
 * then the update is done from there.
 *
 * After processing the firmware updates, this function proceeds to check CPLD update. If there's a CPLD update
 * request, but a PCH/BMC recovery firmware update is in progress, then defer this CPLD update. The CPLD update intent
 * will be written back to BMC update intent register. This is done so that CPLD won't lose track of the
 * recovery firmware update. Update intents and internal global variables will be reset after switching to a 
 * different CPLD image for performing CPLD update. If there's no recovery firmware update in progress, perform the
 * CPLD update.
 * 
 * @see mb_update_intent_handler()
 */
static void act_on_bmc_update_intent()
{
    switch_spi_flash(SPI_FLASH_BMC);
    // Get pointer to the update capsule
    // In PCH flash, the PCH fw update capsule and CPLD update capsule are located at the same place
    // In BMC flash, there's designated locations for BMC fw update capsule, PCH fw update capsule and CPLD update capsule.
    alt_u32* signed_capsule = get_spi_staging_region_ptr(SPI_FLASH_BMC);

    // Read the update intent and then clear it
    alt_u32 update_intent = read_from_mailbox(MB_BMC_UPDATE_INTENT);
    write_to_mailbox(MB_BMC_UPDATE_INTENT, 0);

    // Perform update according to the update intent
    // Nios supports multiple updates from BMC at once. BMC/PCH/CPLD images
    // have dedicated locations on the BMC flash.
    if (update_intent & MB_UPDATE_INTENT_BMC_FW_UPDATE_MASK)
    {
        // This is a BMC firmware update, which can only be triggered from BMC (i.e. BMC update intent register + BMC flash)
        if (check_capsule_before_update(SPI_FLASH_BMC, signed_capsule, 0, update_intent & MB_UPDATE_INTENT_BMC_RECOVERY_MASK))
        {
            perform_firmware_update(SPI_FLASH_BMC, update_intent);
        }
    }

    if (update_intent & MB_UPDATE_INTENT_PCH_FW_UPDATE_MASK)
    {
        // This is an out-of-band PCH firmware update
        // PCH firmware update capsule has a dedicated location in the staging region
        alt_u32* signed_pch_capsule = incr_alt_u32_ptr(signed_capsule, BMC_STAGING_REGION_PCH_UPDATE_CAPSULE_OFFSET);

        if (check_capsule_before_update(SPI_FLASH_BMC, signed_pch_capsule, 0, update_intent & MB_UPDATE_INTENT_PCH_RECOVERY_MASK))
        {
            // Proceed with the firmware update
            // Then copy the capsule over to PCH staging region
            copy_between_flashes(
                get_staging_region_offset(SPI_FLASH_PCH),
                get_staging_region_offset(SPI_FLASH_BMC) + BMC_STAGING_REGION_PCH_UPDATE_CAPSULE_OFFSET,
                SPI_FLASH_PCH,
                SPI_FLASH_BMC,
                get_signed_payload_size(signed_pch_capsule)
             );
            perform_firmware_update(SPI_FLASH_PCH, update_intent);
        }
    }

    // Since CPLD update causes reconfiguration, CPLD update needs to be checked last.
    if (update_intent & MB_UPDATE_INTENT_CPLD_MASK)
    {
        // This is an out-of-band CPLD update
        if (check_spi_flash_state(SPI_FLASH_PCH, SPI_FLASH_STATE_RECOVERY_UPDATE_IN_PROGRESS_MASK) ||
                check_spi_flash_state(SPI_FLASH_BMC, SPI_FLASH_STATE_RECOVERY_UPDATE_IN_PROGRESS_MASK))
        {
            // If there's a recovery update in progress, wait till that recovery update is completed.
            // Write the update intent back to the mailbox register
            write_to_mailbox(MB_BMC_UPDATE_INTENT, MB_UPDATE_INTENT_CPLD_MASK);
        }
        else
        {
            // In BMC flash, there's a designated offset for CPLD update capsule.
            alt_u32* signed_cpld_capsule = incr_alt_u32_ptr(signed_capsule, BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET);

            // Check the update capsule for CPLD update
            if (check_capsule_before_update(SPI_FLASH_BMC, signed_cpld_capsule, 1, 0))
            {
                trigger_cpld_update();
            }
        }
    }
}

/**
 * @brief Read the PCH update intent and perform PCH firmware update.
 *
 * Nios firmware clears the update intent register to 0, after reading from it.
 * It is expected that the update intent value has been validated in the T0 (in mb_update_intent_handler()).
 * If the update intent is invalid, Nios firmware wouldn't transition to T-1 mode to perform the update.
 *
 * Only PCH firmware update is supported from PCH. CPLD update and BMC firmware update are not valid requests.
 *
 * @see mb_update_intent_handler()
 */
static void act_on_pch_update_intent()
{
    // Read the update intent and then clear it
    alt_u32 update_intent = read_from_mailbox(MB_PCH_UPDATE_INTENT);
    write_to_mailbox(MB_PCH_UPDATE_INTENT, 0);

    if (update_intent & MB_UPDATE_INTENT_PCH_FW_UPDATE_MASK)
    {
        switch_spi_flash(SPI_FLASH_PCH);
        alt_u32* signed_capsule = get_spi_staging_region_ptr(SPI_FLASH_PCH);
        if (check_capsule_before_update(SPI_FLASH_PCH, signed_capsule, 0, update_intent & MB_UPDATE_INTENT_PCH_RECOVERY_MASK))
        {
            perform_firmware_update(SPI_FLASH_PCH, update_intent);
        }
    }
}

/**
 * @brief Trigger a recovery firmware update if the global variable spi_flash_state indicates that there's a
 * pending recovery update for the @p spi_flash_type flash.
 *
 * In SPI_FLASH_STATE_MASK_ENUM, Nios firmware has 1 bit to indicate an active update is in progress and 1 bit
 * to indicate a recovery update is in progress. If an active update is completed but a recovery update is
 * still in progress, that tells Nios firmware to trigger a recovery update.
 * 
 * @param spi_flash_type indicate BMC or PCH SPI flash device
 *
 * @see post_update_routine
 */
static void process_pending_recovery_update(SPI_FLASH_TYPE_ENUM spi_flash_type)
{
    if ((!check_spi_flash_state(spi_flash_type, SPI_FLASH_STATE_ACTIVE_UPDATE_IN_PROGRESS_MASK)) &&
            check_spi_flash_state(spi_flash_type, SPI_FLASH_STATE_RECOVERY_UPDATE_IN_PROGRESS_MASK))
    {
        // Once active update is successful, then proceed with recovery update
        perform_firmware_recovery_update(spi_flash_type);
    }
}

/**
 * @brief Process firmware and CPLD updates in T-1 mode.
 * Nios firmware first go through the BMC and PCH update intent registers. Then, it
 * triggers BMC/PCH recovery firmware update if there's any pending.
 *
 * @see act_on_update_intent
 * @see process_pending_recovery_update
 */
static void process_updates_in_tmin1()
{
    // Perform updates as requested in the update intent registers
    act_on_bmc_update_intent();
    act_on_pch_update_intent();

    // Perform pending recovery update if there's any
    process_pending_recovery_update(SPI_FLASH_PCH);
    process_pending_recovery_update(SPI_FLASH_BMC);
}

#endif /* WHITLEY_INC_TMIN1_UPDATE_H_ */
