// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other
// software and tools, and its AMPP partner logic functions, and any output
// files from any of the foregoing (including device programming or simulation
// files), and any associated documentation or information are expressly subject
// to the terms and conditions of the Intel Program License Subscription
// Agreement, Intel FPGA IP License Agreement, or other applicable
// license agreement, including, without limitation, that your use is for the
// sole purpose of programming logic devices manufactured by Intel and sold by
// Intel or its authorized distributors.  Please refer to the applicable
// agreement for further details.

/**
 * @file transition.h
 * @brief Responsible for T0 / T-1 / T0 mode transition.
 */

#ifndef WHITLEY_INC_TRANSITION_H_
#define WHITLEY_INC_TRANSITION_H_

// Always include pfr_sys.h first
#include "pfr_sys.h"

#include "tmin1_routines.h"
#include "hierarchical_pfr.h"

/**
 * @brief Prepare the platform to enter T-1 mode.
 * 
 * Steps:
 * 1. Drive the SLP_SUS_N pin high prior to setting RSMRST# and DSWPWROK.
 * 2. Assert resets on BMC and PCH through SRST# and RSMRST# pin. 
 * 3. Take over the SPI flash devices controls through the SPI control block.
 * 4. Drive DSWPWROK to 0.
 * 
 * If hierarchical PFR is supported, Nios firmware also notifies other CPLDs that
 * it is entering T-1 mode.
 */
static void perform_entry_to_tmin1()
{
#ifdef USE_SYSTEM_MOCK
    SYSTEM_MOCK::get()->incr_t_minus_1_counter();
#endif

    // log that we are entering T-1 mode
    log_platform_state(PLATFORM_STATE_ENTER_TMIN1);

    // Notify other CPLDs if HPFR is supported
    perform_hpfr_entry_to_tmin1();

#ifndef PLATFORM_WILSON_CITY_FAB1
    // PFR should drive the new PFR_SLP_SUS_N pin high, during a T0 to T-1 transition.
    // It should drive that high before setting RSMRST# and DSWPWROK low.
    set_bit(U_GPO_1_ADDR, GPO_1_FM_PFR_SLP_SUS_N);
#endif

    // Assert reset on BMC and PCH
    clear_bit(U_GPO_1_ADDR, GPO_1_RST_SRST_BMC_PLD_R_N);
    clear_bit(U_GPO_1_ADDR, GPO_1_RST_RSMRST_PLD_R_N);

    // Take over controls on the SPI flash devices
    takeover_spi_ctrls();

#ifndef PLATFORM_WILSON_CITY_FAB1
    // Set Deep SX. This will drive 0 on PWRGD_DSW_PWROK_R
    set_bit(U_GPO_1_ADDR, GPO_1_PWRGD_DSW_PWROK_R);
#endif
}

/**
 * @brief Perform the platform to enter T0 mode. 
 *
 * The main tasks here are:
 *   - release controls of SPI flash devices
 *   - release BMC and PCH from resets
 *   - launch watchdog timers if the system is provisioned.
 * 
 * If a SPI flash device is found to have invalid images for active, recovery and staging regions, 
 * then Nios firmware will keep that associated component in reset. 
 * 
 * If only BMC SPI flash has valid image(s), then boot BMC in isolation with watchdog timer turned off. 
 * BMC's behavior is unknown when PCH is in reset. 
 * 
 * When both SPI flash device have invalid images, Nios firmware enters a lockdown mode in T-1. 
 * 
 * If hierarchical PFR is supported, Nios firmware also notifies other CPLDs that
 * it is entering T0 mode.
 */
static void perform_entry_to_t0()
{
    // Notify other CPLDs if HPFR is supported
    perform_hpfr_entry_to_t0();

    alt_u32 bmc_has_valid_image = !check_spi_flash_state(SPI_FLASH_BMC, SPI_FLASH_STATE_ALL_REGIONS_FAILED_AUTH_MASK);
    alt_u32 pch_has_valid_image = !check_spi_flash_state(SPI_FLASH_PCH, SPI_FLASH_STATE_ALL_REGIONS_FAILED_AUTH_MASK);
    if (bmc_has_valid_image && pch_has_valid_image)
    {
        // If both PCH and BMC have valid images, boot both components
        tmin1_boot_bmc_and_pch();
    }
    else if (pch_has_valid_image)
    {
        // Only PCH has valid image(s), boot PCH in isolation.
        tmin1_boot_pch();
    }
    else if (bmc_has_valid_image)
    {
        // Only BMC has valid image(s), boot BMC in isolation.
        // Disable BMC watchdog tiemr because BMC behavior is unknown when PCH is in reset.
        wdt_enable_status &= WDT_DISABLE_BMC_TIMER_MASK;
        tmin1_boot_bmc();
    }
    else
    {
        // If both BMC and PCH flashes failed authentication for all regions, stay in T-1 mode.
        log_platform_state(PLATFORM_STATE_AUTHENTICATION_FAILED_LOCKDOWN);
        never_exit_loop();
    }

    log_platform_state(PLATFORM_STATE_ENTER_T0);
}

/**
 * @brief Transition the platform to T-1 mode, perform T-1 operations 
 * if the PFR system is provisioned, and then transition the platform back to T0 mode.
 */
static void perform_platform_reset()
{
    // Prepare for transition to T-1 mode
    perform_entry_to_tmin1();

    // In unprovisioned state, skip T-1 operations, filter enabling and boot monitoring.
    if (is_ufm_provisioned())
    {
        // Perform T-1 operations
        perform_tmin1_operations();
    }

    // Perform the entry to T0
    perform_entry_to_t0();
}

/********************************************
 *
 * BMC only reset
 *
 ********************************************/

/**
 * @brief This function transitions BMC to a reset state through EXTRST#. 
 *
 * When in T0 mode, Nios firmware may detect some panic events such as BMC 
 * watchdog timer timeout and BMC firmware update. In order to keep the host
 * up while recovering or updating BMC, Nios firmware performs a BMC-only reset. 
 */
static void perform_entry_to_tmin1_bmc_only()
{
#ifdef USE_SYSTEM_MOCK
    SYSTEM_MOCK::get()->incr_t_minus_1_bmc_only_counter();
#endif
    // Assert EXTRST# to reset only the BMC
    clear_bit(U_GPO_1_ADDR, GPO_1_RST_PFR_EXTRST_N);

    // Take over control on the BMC flash device
    takeover_spi_ctrl(SPI_FLASH_BMC);
}

/**
 * @brief This function transitions BMC back to T0 state through EXTRST#. 
 * 
 * After releasing SPI control of BMC flash back to BMC, Nios firmware releases
 * BMC from reset by setting EXTRST#. 
 * 
 * Since BMC-only reset only happens in a provisioned system, BMC watchdog timer 
 * is always ran after releasing the reset.
 */
static void perform_entry_to_t0_bmc_only()
{
    // Release SPI flash control to BMC
    release_spi_ctrl(SPI_FLASH_BMC);

    // Release reset on BMC
    set_bit(U_GPO_1_ADDR, GPO_1_RST_PFR_EXTRST_N);

    // Track the BMC boot progress
    // Clear any previous BMC boot done status
    wdt_boot_status &= WDT_CLEAR_BMC_BOOT_DONE_MASK;

    // Start the BMC watchdog timer
    start_timer_ms(WDT_BMC_TIMER_ADDR, WD_BMC_EXPIRE_IN_MS);

    log_platform_state(PLATFORM_STATE_ENTER_T0);
}

/**
 * @brief Transition the BMC to T-1 mode, perform T-1 operations, 
 * and then transition the platform back to T0 mode.
 * 
 * Since BMC-only reset only happens in a provisioned system, Nios firmware always
 * perform the BMC specific T-1 operations.
 */
static void perform_bmc_only_reset()
{
    // Enter T-1 for BMC
    perform_entry_to_tmin1_bmc_only();

    // Perform T-1 operations on BMC (assuming that Nios is provisioned at this point)
    perform_tmin1_operations_bmc_only();

    // Enter T0 for BMC
    perform_entry_to_t0_bmc_only();
}


#endif /* WHITLEY_INC_TRANSITION_H_ */
