#!/usr/bin/env python

import unittest

from pfr_bitstream import PFRBitstream
from pfm_smbus_rule import SMBusRuleDef
from pfm_spi_region import SPIRegionDef
from convert import Convert


class PFMData(PFRBitstream):
    """
    This is a class that represents Platform Firmware Manifest data structure.
    PFM data structure is described in section 2.3 of PFR CPLD RoT High Level Arch Spec
    """

    # Fixed size portion of the data structure
    _PFM_DATA_FIXED_AREA_SIZE = 32

    # PFM data structure is 128B aligned
    _PFM_DATA_ALIGN_BOUNDARY = 128

    # PFM Padding value
    _PFM_DATA_PADDING = 0xff

    def __init__(self):
        super(PFMData, self).__init__()
        self._pfm_body = list()

    def initialize(self, byte_array=None, size=_PFM_DATA_ALIGN_BOUNDARY):
        super(PFMData, self).initialize(size=size)
        self.set_tag(PFMData.magic())
        self.set_length(size)

    def validate(self):
        # Verify magic number
        if self.tag() != PFMData.magic():
            raise ValueError("The tag is incorrect in this PFM data.")

        # Perform validation on the content of the PFM body
        self.check_pfm_body()

        # Size check
        if self.size() != self.length():
            raise ValueError("The actual size (" + hex(self.size()) + " bytes) of this PFM data does not match the " +
                             "expected size (" + hex(self.length()) + " bytes)at offset 0x00C.")

        return super(PFMData, self).validate()

    def read(self, fp):
        if fp.closed:
            return
        # Read everything until PFM Body
        self.append(byte_array=bytearray(fp.read(PFMData._PFM_DATA_FIXED_AREA_SIZE)))

        # Read PFM Body
        while True:
            def_type_byte = fp.peek(1)
            def_type = Convert().bytearray_to_integer(def_type_byte, offset=0, num_of_bytes=1)

            if def_type == SPIRegionDef.def_type():
                region_def = SPIRegionDef()
                region_def.read(fp=fp)
                region_def.validate()
                self.read_pfm_def(region_def)
            elif def_type == SMBusRuleDef.def_type():
                rule_def = SMBusRuleDef()
                rule_def.read(fp=fp)
                rule_def.validate()
                self.read_pfm_def(rule_def)
            else:
                break

        # Read the padding
        self.append(byte_array=bytearray(fp.read()))

    def extend(self):
        extend_size = PFMData._PFM_DATA_ALIGN_BOUNDARY - self.size() % PFMData._PFM_DATA_ALIGN_BOUNDARY
        self.append(size=extend_size)
        self.set_length(self.size())

    def body_size(self):
        body_size = 0
        for pfm_def in self._pfm_body:
            body_size += pfm_def.size()
        return body_size

    def get_pfm_def(self, index=0):
        return self._pfm_body[index]

    def read_pfm_def(self, pfm_def):
        assert(isinstance(pfm_def, SPIRegionDef) or isinstance(pfm_def, SMBusRuleDef))
        self._pfm_body.append(pfm_def)
        self.append(byte_array=pfm_def.get_raw_byte_array())

    def append_pfm_def(self, pfm_def):
        assert(isinstance(pfm_def, SPIRegionDef) or isinstance(pfm_def, SMBusRuleDef))

        offset = PFMData._PFM_DATA_FIXED_AREA_SIZE + self.body_size()
        pfm_def_size = pfm_def.size()
        if offset + pfm_def_size > self.size():
            self.extend()

        self._pfm_body.append(pfm_def)
        self.set_raw_value(offset=offset, byte_array=pfm_def.get_raw_byte_array())

    @staticmethod
    def magic():
        # Pre-defined magic number for PFM data structure
        return 0x02B3CE1D

    def tag(self):
        return self.get_value(0x0)
    
    def set_tag(self, value):
        self.set_value(value=value, offset=0x0)

    def svn(self):
        return self.get_value(0x4, size=1)

    def set_svn(self, value):
        self.set_value(value=value, offset=0x4, size=1)

    def bkc_version(self):
        return self.get_value(0x5, size=1)

    def set_bkc_version(self, value):
        self.set_value(value=value, offset=0x5, size=1)

    def revision(self):
        return self.get_value(0x6, size=2)

    def set_revision(self, value):
        self.set_value(value=value, offset=0x6, size=2)

    def length(self):
        return self.get_value(0x1C, size=4)

    def set_length(self, value):
        self.set_value(value=value, offset=0x1C, size=4)

    def padding_size(self):
        data_size = PFMData._PFM_DATA_FIXED_AREA_SIZE + self.body_size()
        padding_size = PFMData._PFM_DATA_ALIGN_BOUNDARY - data_size % PFMData._PFM_DATA_ALIGN_BOUNDARY
        padding_size = padding_size % PFMData._PFM_DATA_ALIGN_BOUNDARY
        return padding_size

    def set_padding(self):
        assert(self.length() != 0)
        offset = PFMData._PFM_DATA_FIXED_AREA_SIZE + self.body_size()
        padding_size = self.padding_size()

        if padding_size > 0:
            # Padding should be all 0xFF
            byte_array = bytearray([0xFF] * padding_size)
            self.set_raw_value(offset=offset, byte_array=byte_array)

    def check_pfm_body(self):
        pfm_body_offset = PFMData._PFM_DATA_FIXED_AREA_SIZE
        spi_region_address = list()

        def_type = self.get_value(pfm_body_offset, size=1)
        while True:
            if def_type == SMBusRuleDef.def_type():
                # Validate this SMBus rule definition
                smbus_rule = SMBusRuleDef()
                smbus_rule.initialize(
                    self.get_raw_byte_array()[pfm_body_offset: pfm_body_offset + SMBusRuleDef.def_size()])
                smbus_rule.validate()
                pfm_body_offset += SMBusRuleDef.def_size()
            elif def_type == SPIRegionDef.def_type():
                # Create a temporary SPI region definition to obtain the calculated size
                spi_region_tmp = SPIRegionDef()
                spi_region_tmp.initialize(
                    self.get_raw_byte_array()[pfm_body_offset: pfm_body_offset + SPIRegionDef.def_max_size()])
                spi_region_size = spi_region_tmp.calc_size()

                # Validate this SPIRegionDef object
                spi_region = SPIRegionDef()
                spi_region.initialize(
                    self.get_raw_byte_array()[pfm_body_offset: pfm_body_offset + spi_region_size])
                spi_region.validate()

                # Append the SPI region start address
                spi_region_address.append(spi_region.start_addr())
                # Append the SPI region end address
                spi_region_address.append(spi_region.end_addr())
                # Skip the rest of the SPI region definition
                pfm_body_offset += spi_region.size()
            elif def_type == PFMData._PFM_DATA_PADDING:
                break
            else:
                raise ValueError("Read an unexpected value " + hex(def_type) + " in the PFM, at offset " +
                                 str(pfm_body_offset) + ". "
                                 "This byte is not definition type for either SPI region or SMBus rule definition. "
                                 "This byte is also not padding. Please check your PFM data structure. ")

            if pfm_body_offset == self.size():
                # Done processing PFM data
                break
            def_type = self.get_value(pfm_body_offset, size=1)

        # Ensure that the spi region addresses are in ascending order
        if spi_region_address != sorted(spi_region_address):
            raise ValueError("The addresses in spi region definitions are not in ascending order. "
                             "SPI regions must not overlap and must be arranged in ascending order. ")


class PFMDataTest(unittest.TestCase):

    def test_initialize(self):
        data = PFMData()
        data.initialize()
        data.validate()

    def test_simple_pfm_data(self):
        data = PFMData()
        data.initialize()

        data.set_revision(value=0x0101)
        data.validate()

        self.assertEqual(data.size(), PFMData._PFM_DATA_ALIGN_BOUNDARY)
        self.assertEqual(data.revision(), 0x0101)
        self.assertEqual(data.get_value(offset=PFMData._PFM_DATA_ALIGN_BOUNDARY - 10, size=1), 0xFF)

    def test_complex_pfm_data(self):
        data = PFMData()
        data.initialize()

        for _ in range(4):
            region_def = SPIRegionDef()
            region_def.initialize()
            rule_def = SMBusRuleDef()
            rule_def.initialize()
            rule_def.set_bus_id(1)
            rule_def.set_rule_id(1)
            rule_def.set_device_addr(0x90)

            data.append_pfm_def(region_def)
            data.append_pfm_def(rule_def)

        data.set_padding()
        data.validate()


if __name__ == '__main__':
    unittest.main()
