#!/usr/bin/env python

import unittest

from pfr_bitstream import PFRBitstream


class SMBusRuleDef(PFRBitstream):
    """
    This is a class that represents Platform Firmware Manifest SMBus rule definition.
    """

    # SMBus Rule definition has a fixed size of 40 bytes
    _DEFINITION_SIZE = 40

    # SMBus relay I2C slave device address mapping with Bus ID and Rule ID
    _NUM_RELAYS = 3
    _MAX_I2C_ADDRESSES_PER_RELAY = 11
    _RELAYS_I2C_ADDRESSES = [
        # Relay1 PMBus1
        [0x90, 0xAC, 0xA2, 0xB2, 0xA0, 0xB0, 0x00, 0x00, 0x00, 0x00, 0x00],
        # Relay2 PMBus2
        [0xB4, 0xD4, 0x4A, 0x4C, 0xDC, 0xEC, 0xE0, 0xB0, 0xC4, 0xCC, 0xE4],
        # Relay3 HSBP
        [0x98, 0xA4, 0xD0, 0xD8, 0xE8, 0xE0, 0xA6, 0x36, 0x00, 0x00, 0x00]
    ]

    def __init__(self):
        super(SMBusRuleDef, self).__init__()

    def initialize(self, byte_array=None, size=_DEFINITION_SIZE):
        if byte_array:
            super(SMBusRuleDef, self).initialize(byte_array=byte_array)
        else:
            super(SMBusRuleDef, self).initialize(size=size)
            self.set_def_type()

    def validate(self):
        # definition type: PFM SMBus rule definition
        if self.read_def_type() != SMBusRuleDef.def_type():
            raise ValueError("The definition type is incorrect for this SMBus rule.")

        # Reserved
        if self.get_value(offset=0x1, size=4) != 0xFFFFFFFF:
            raise ValueError("The reserved area is corrupted for this SMBus rule.")

        # Check size
        if self.size() != SMBusRuleDef.def_size():
            raise ValueError("The size of this SMBus rule is incorrect.")

        # Check Bus ID
        if self.bus_id() not in [i for i in range(1, SMBusRuleDef._NUM_RELAYS + 1)]:
            raise ValueError("The bus id (" + hex(self.bus_id()) + ") is invalid. Expected values are 1, 2, or 3. ")

        # Check Rule ID
        if self.rule_id() not in [i for i in range(1, SMBusRuleDef._MAX_I2C_ADDRESSES_PER_RELAY + 1)]:
            raise ValueError("The rule id (" + hex(self.rule_id()) + ") is invalid. "
                             "Expected values are [1..." + str(SMBusRuleDef._MAX_I2C_ADDRESSES_PER_RELAY) + "] ")

        # Check device address and bus ID/Rule ID mapping
        if self.device_addr() != SMBusRuleDef._RELAYS_I2C_ADDRESSES[self.bus_id() - 1][self.rule_id() - 1]:
            raise ValueError("The slave device address (" + hex(self.device_addr()) + ") does not match the expected " +
                             "address for bus id " + hex(self.bus_id()) + " and rule id " + hex(self.rule_id()) + ". "
                             "Please refer to ref/gen_smbus_relay_config.h for the mapping details. ")
        return super(SMBusRuleDef, self).validate()

    @staticmethod
    def def_type():
        return 0x2

    @staticmethod
    def def_size():
        return SMBusRuleDef._DEFINITION_SIZE

    def read_def_type(self):
        return self.get_value(0x0, size=1)

    def set_def_type(self, value=2):
        self.set_value(value=value, offset=0x0, size=1)

    def bus_id(self):
        return self.get_value(0x5, size=1)

    def set_bus_id(self, value):
        self.set_value(value=value, offset=0x5, size=1)

    def rule_id(self):
        return self.get_value(0x6, size=1)

    def set_rule_id(self, value):
        self.set_value(value=value, offset=0x6, size=1)

    def device_addr(self):
        return self.get_value(0x7, size=1)

    def set_device_addr(self, value):
        self.set_value(value=value, offset=0x7, size=1)

    def cmd_whitelist(self):
        return self.get_value(0x8, size=32)

    def set_cmd_whitelist(self, value):
        self.set_value(value=value, offset=0x8, size=32)

    def read(self, fp):
        if fp.closed:
            return
        # Read the SMBus rule definition
        self.append(byte_array=bytearray(fp.read(SMBusRuleDef._DEFINITION_SIZE)))


class SMBusRuleDefTest(unittest.TestCase):

    def test_initialize(self):
        rule_def = SMBusRuleDef()
        rule_def.initialize()

        rule_def.set_bus_id(value=3)
        rule_def.set_rule_id(value=11)
        rule_def.set_device_addr(value=0x00)

        rule_def.validate()

    def test_example_rule_def(self):
        rule_def = SMBusRuleDef()
        rule_def.initialize()
        rule_def.set_bus_id(value=2)
        rule_def.set_rule_id(value=0x3)
        rule_def.set_device_addr(value=0x4A)
        rule_def.set_cmd_whitelist(value=0b100101011)
        rule_def.validate()
        self.assertEqual(rule_def.size(), SMBusRuleDef._DEFINITION_SIZE)
        self.assertEqual(rule_def.bus_id(), 2)
        self.assertEqual(rule_def.rule_id(), 0x3)
        self.assertEqual(rule_def.device_addr(), 0x4A)
        self.assertEqual(rule_def.cmd_whitelist(), 0b100101011)


if __name__ == '__main__':
    unittest.main()
