#!/usr/bin/env python

import unittest

from convert import Convert


class PFRBitstream(object):
    """
    This is a class that represents an arbitrary bit-stream used in PFR collateral.
    """
    def __init__(self):
        PFRBitstream.initialize(self, byte_array=None)

    def initialize(self, byte_array=None, size=0):
        """
        Initialize an PFRBitstream instance so that it can pass validate().
        This function is not meant to be run after read().
        """
        if byte_array is None:
            self.__raw = bytearray([0xFF] * size)
        elif size != 0:
            raise ValueError("size attribute to initialize method if only valid if byte_array not specified")
        else:
            self.__raw = byte_array

    def size(self):
        return len(self.__raw)

    def append(self, byte_array=None, size=0):
        if byte_array is None:
            if size <= 0:
                raise ValueError('size attribute to append method is <= 0')
            else:
                self.__raw += bytearray([0]*size)
        elif size != 0:
                raise ValueError('size attribute to initialize method if only valid if byte_array not specified')
        else:
            self.__raw += byte_array

    def validate(self):
        return True

    def update(self):
        pass

    def get_raw_byte_array(self):
        return self.__raw

    def get_raw_value(self, offset, size, endianness='little'):
        assert (offset + size <= self.size())
        raw_byte_array = self.get_raw_byte_array()

        if endianness == 'big':
            return raw_byte_array[offset:offset + size][::-1]
        else:
            return raw_byte_array[offset:offset + size]

    def get_value(self, offset, size=4, endianness='little'):
        return Convert().bytearray_to_integer(self.get_raw_value(offset=offset, size=size, endianness=endianness))

    def set_raw_value(self, byte_array, offset, endianness='little'):
        sz = len(byte_array)
        assert (offset + sz <= self.size())

        if endianness == 'big':
            self.__raw[offset:offset + sz] = byte_array[0:sz][::-1]
        else:
            self.__raw[offset:offset + sz] = byte_array[0:sz]

    def set_value(self, value, offset, size=4, endianness='little'):
        ba = Convert().integer_to_bytes(n=value, length=size)
        assert (len(ba) == size)
        self.set_raw_value(byte_array=ba, offset=offset, endianness=endianness)

    def write(self, fp):
        self.validate()
        fp.write(self.get_raw_byte_array())

    def read(self, fp):
        if fp.closed:
            return
        self.append(byte_array=bytearray(fp.read()))


if __name__ == '__main__':
    unittest.main()
