#ifndef SYSTEM_SPI_FLASH_MOCK_H
#define SYSTEM_SPI_FLASH_MOCK_H

// Standard headers
#include <memory>
#include <string>
#include <unordered_map>
#include <fstream>

// Mock headers
#include "alt_types_mock.h"
#include "memory_mock.h"
#include "unordered_map_memory_mock.h"

// PFR system
#include "pfr_sys.h"
#include "spi_common.h"

// 128MB (can fit either of PCH or BMC image)
#define FLASH_SIZE 0x8000000

class SPI_FLASH_MOCK
{
public:
    static SPI_FLASH_MOCK* get();

    void reset();
    void reset(SPI_FLASH_TYPE_ENUM spi_flash_type);

    void load(SPI_FLASH_TYPE_ENUM spi_flash_type, const std::string& file_path,
            int file_size, int load_offset);

    // Expose the x86 addresses of these flash memories
    //   Return the pointer to the selected flash
    alt_u32* get_flash_ptr(SPI_FLASH_TYPE_ENUM spi_flash_type)
    {
        if (spi_flash_type == SPI_FLASH_PCH)
        {
            return m_pch_flash_mem;
        }
        return m_bmc_flash_mem;
    }

    void erase_4kb(SPI_FLASH_TYPE_ENUM spi_flash_type, alt_u32 addr)
    {
        alt_u32* spi_flash_ptr = m_pch_flash_mem;
        if (spi_flash_type == SPI_FLASH_BMC)
        {
            spi_flash_ptr = m_bmc_flash_mem;
        }

        spi_flash_ptr += addr >> 2;

        // Erase the 4KB page at the target address
        for (alt_u32 i = 0; i < (0x1000 >> 2); i++)
        {
            spi_flash_ptr[i] = 0xffffffff;
        }
    }


private:
    // Singleton inst
    static SPI_FLASH_MOCK* s_inst;

    SPI_FLASH_MOCK();
    ~SPI_FLASH_MOCK();

    // Memory area for the flash memory
    alt_u32 *m_pch_flash_mem = nullptr;
    alt_u32 *m_bmc_flash_mem = nullptr;
};

#endif /* SYSTEM_SPI_FLASH_MOCK_H_ */
