#ifndef UNITTEST_SYSTEM_UT_NIOS_WRAPPER_H_
#define UNITTEST_SYSTEM_UT_NIOS_WRAPPER_H_

// Always include the BSP mock first
#include "bsp_mock.h"

// Include the PFR headers
#include "authentication.h"
#include "capsule_validation.h"
#include "cpld_update.h"
#include "cpld_reconfig.h"
#include "cpld_recovery.h"
#include "crypto.h"
#include "decompression.h"
#include "firmware_recovery.h"
#include "firmware_update.h"
#include "flash_validation.h"
#include "gen_gpi_signals.h"
#include "gen_gpo_controls.h"
#include "gen_smbus_relay_config.h"
#include "global_state.h"
#include "keychain.h"
#include "keychain_utils.h"
#include "mailbox_enums.h"
#include "mailbox_utils.h"
#include "pbc.h"
#include "pbc_utils.h"
#include "pfm.h"
#include "pfm_utils.h"
#include "pfm_validation.h"
#include "pfr_main.h"
#include "pfr_pointers.h"
#include "pfr_sys.h"
#include "platform_log.h"
#include "rfnvram_utils.h"
#include "smbus_relay_utils.h"
#include "spi_common.h"
#include "spi_ctrl_utils.h"
#include "spi_flash_state.h"
#include "spi_rw_utils.h"
#include "status_enums.h"
#include "t0_provisioning.h"
#include "t0_routines.h"
#include "t0_update.h"
#include "t0_watchdog_handler.h"
#include "timer_utils.h"
#include "tmin1_routines.h"
#include "transition.h"
#include "ufm.h"
#include "ufm_rw_utils.h"
#include "ufm_utils.h"
#include "utils.h"
#include "watchdog_timers.h"


static void ut_allow_nios_to_start()
{
    // Have the check on the Nios ready bit return true after 1 check
    SYSTEM_MOCK::get()->register_read_write_callback(
            [](SYSTEM_MOCK::READ_OR_WRITE read_or_write, void* addr, alt_u32 data) {
        if (read_or_write == SYSTEM_MOCK::READ_OR_WRITE::READ && addr == U_GPI_1_ADDR)
        {
            // De-assert both PCH and BMC resets (0b11)
            // Keep FORCE_RECOVERY signal inactive (0b1000000)
            // Keep PLTRST# inactive (0b10000)
            SYSTEM_MOCK::get()->set_mem_word(U_GPI_1_ADDR, alt_u32(0x53), true);

            if (read_from_mailbox(MB_PLATFORM_STATE) == PLATFORM_STATE_ENTER_T0)
            {
                // When Nios firmware just reaches T0, simulate the hw PLTRST# toggle
                SYSTEM_MOCK::get()->set_mem_word(U_GPI_1_ADDR, alt_u32(0x43), true);
            }
        }
    });
}

static void ut_set_gpi_nios_start_and_me_boot_done()
{
    SYSTEM_MOCK::get()->register_read_write_callback(
            [](SYSTEM_MOCK::READ_OR_WRITE read_or_write, void* addr, alt_u32 data) {
        if (read_or_write == SYSTEM_MOCK::READ_OR_WRITE::READ && addr == U_GPI_1_ADDR)
        {
            // Assert both PCH and BMC resets (0b11)
            // Signals that ME firmware has booted (0b1000)
            // Keep FORCE_RECOVERY signal inactive (0b1000000)
            // Keep PLTRST# inactive (0b10000)
            SYSTEM_MOCK::get()->set_mem_word(U_GPI_1_ADDR, alt_u32(0x5b), true);

            if (read_from_mailbox(MB_PLATFORM_STATE) == PLATFORM_STATE_ENTER_T0)
            {
                // When Nios firmware just reaches T0, simulate the hw PLTRST# toggle
                SYSTEM_MOCK::get()->set_mem_word(U_GPI_1_ADDR, alt_u32(0x4b), true);
            }
        }
    });
}

static void ut_send_bmc_reset_detected_gpi_once_upon_boot_complete()
{
    SYSTEM_MOCK::get()->register_read_write_callback(
            [](SYSTEM_MOCK::READ_OR_WRITE read_or_write, void* addr, alt_u32 data) {
        if (read_or_write == SYSTEM_MOCK::READ_OR_WRITE::READ && addr == U_GPI_1_ADDR)
        {
            if ((read_from_mailbox(MB_PLATFORM_STATE) == PLATFORM_STATE_T0_BOOT_COMPLETE) &&
                    (read_from_mailbox(MB_PANIC_EVENT_COUNT) == 0))
            {
                // De-assert both PCH and BMC resets (0b11)
                // Keep FORCE_RECOVERY signal inactive (0b1000000)
                // Keep PLTRST# inactive (0b10000)
                // BMC reset detected (0b100000)
                SYSTEM_MOCK::get()->set_mem_word(U_GPI_1_ADDR, alt_u32(0x73), true);
            }
        }
    });
}

/**
 * @brief Return the value in the global state register
 *
 * @return alt_u32 global state
 */
static alt_u32 ut_get_global_state()
{
    return IORD_32DIRECT(U_GLOBAL_STATE_REG_ADDR, 0);
}

static void ut_send_block_complete_chkpt_msg()
{
    // Signals that BMC/ACM/BIOS have all booted after one check
    SYSTEM_MOCK::get()->register_read_write_callback(
            [](SYSTEM_MOCK::READ_OR_WRITE read_or_write, void* addr, alt_u32 data) {
        alt_u32* bmc_ckpt_addr = U_MAILBOX_AVMM_BRIDGE_ADDR + MB_BMC_CHECKPOINT;
        if (addr == bmc_ckpt_addr)
        {
            SYSTEM_MOCK::get()->set_mem_word(bmc_ckpt_addr, MB_CHKPT_COMPLETE, true);
        }

        alt_u32* acm_ckpt_addr = U_MAILBOX_AVMM_BRIDGE_ADDR + MB_ACM_CHECKPOINT;
        if (addr == acm_ckpt_addr)
        {
            SYSTEM_MOCK::get()->set_mem_word(acm_ckpt_addr, MB_CHKPT_COMPLETE, true);
        }

        alt_u32* bios_ckpt_addr = U_MAILBOX_AVMM_BRIDGE_ADDR + MB_BIOS_CHECKPOINT;
        if (addr == bios_ckpt_addr)
        {
            SYSTEM_MOCK::get()->set_mem_word(bios_ckpt_addr, MB_CHKPT_COMPLETE, true);
        }
    });
}

static void ut_send_in_update_intent(MB_REGFILE_OFFSET_ENUM update_intent_offset, alt_u32 update_intent_value)
{
    SYSTEM_MOCK::get()->register_read_write_callback(
            [update_intent_offset, update_intent_value](SYSTEM_MOCK::READ_OR_WRITE read_or_write, void* addr, alt_u32 data) {
        if (read_or_write == SYSTEM_MOCK::READ_OR_WRITE::READ)
        {
            alt_u32* update_intent_addr = U_MAILBOX_AVMM_BRIDGE_ADDR + update_intent_offset;
            if (addr == update_intent_addr)
            {
                if (read_from_mailbox(MB_PLATFORM_STATE) == PLATFORM_STATE_T0_BOOT_COMPLETE)
                {
                    SYSTEM_MOCK::get()->set_mem_word(update_intent_addr, update_intent_value, true);
                }
            }
        }
    });
}

static void ut_send_in_update_intent_once_upon_entry_to_t0(MB_REGFILE_OFFSET_ENUM update_intent_offset, alt_u32 update_intent_value)
{
    SYSTEM_MOCK::get()->register_read_write_callback(
            [update_intent_offset, update_intent_value](SYSTEM_MOCK::READ_OR_WRITE read_or_write, void* addr, alt_u32 data) {
        if (read_or_write == SYSTEM_MOCK::READ_OR_WRITE::READ)
        {
            alt_u32* update_intent_addr = U_MAILBOX_AVMM_BRIDGE_ADDR + update_intent_offset;
            if (addr == update_intent_addr)
            {
                if ((read_from_mailbox(MB_PLATFORM_STATE) == PLATFORM_STATE_ENTER_T0) && (read_from_mailbox(MB_PANIC_EVENT_COUNT) == 0))
                {
                    SYSTEM_MOCK::get()->set_mem_word(update_intent_addr, update_intent_value, true);
                }
            }
        }
    });
}

static void ut_send_in_update_intent_once_upon_boot_complete(MB_REGFILE_OFFSET_ENUM update_intent_offset, alt_u32 update_intent_value)
{
    SYSTEM_MOCK::get()->register_read_write_callback(
            [update_intent_offset, update_intent_value](SYSTEM_MOCK::READ_OR_WRITE read_or_write, void* addr, alt_u32 data) {
        if (read_or_write == SYSTEM_MOCK::READ_OR_WRITE::READ)
        {
            alt_u32* update_intent_addr = U_MAILBOX_AVMM_BRIDGE_ADDR + update_intent_offset;
            if (addr == update_intent_addr)
            {
                if ((read_from_mailbox(MB_PLATFORM_STATE) == PLATFORM_STATE_T0_BOOT_COMPLETE) && (read_from_mailbox(MB_PANIC_EVENT_COUNT) == 0))
                {
                    SYSTEM_MOCK::get()->set_mem_word(update_intent_addr, update_intent_value, true);
                }
            }
        }
    });
}

static alt_u32 ut_check_ufm_prov_status(MB_UFM_PROV_STATUS_MASK_ENUM status_mask)
{
    return (IORD(U_MAILBOX_AVMM_BRIDGE_ADDR, MB_PROVISION_STATUS) & status_mask) == status_mask;
}

static void ut_send_in_ufm_command(MB_UFM_PROV_CMD_ENUM ufm_prov_cmd)
{
    alt_u32* mb_ufm_prov_cmd_addr = U_MAILBOX_AVMM_BRIDGE_ADDR + MB_PROVISION_CMD;
    alt_u32* mb_ufm_cmd_trigger_addr = U_MAILBOX_AVMM_BRIDGE_ADDR + MB_UFM_CMD_TRIGGER;

    SYSTEM_MOCK::get()->set_mem_word(mb_ufm_prov_cmd_addr, ufm_prov_cmd, true);
    SYSTEM_MOCK::get()->set_mem_word(mb_ufm_cmd_trigger_addr, MB_UFM_CMD_EXECUTE_MASK, true);
}

static void ut_wait_for_ufm_prov_cmd_done()
{
    while ((IORD(U_MAILBOX_AVMM_BRIDGE_ADDR, MB_PROVISION_STATUS) & MB_UFM_PROV_CMD_DONE_MASK) == 0) {}
}


static void ut_disable_watchdog_timers()
{
    wdt_enable_status = 0;
}

static void ut_reset_watchdog_timers()
{
    // TODO: Default should be 0. Setting ME to be done by default for now.
    // Change this back to 0 once ME starts sending checkpoint messages.
    wdt_boot_status = 0b00010;
    wdt_enable_status = WDT_ENABLE_ALL_TIMERS_MASK;
}

static void ut_reset_failed_update_attempts()
{
    num_failed_pch_update_attempts = 0;
    num_failed_bmc_update_attempts = 0;
    num_failed_cpld_update_attempts = 0;
}

static void ut_reset_fw_recovery_levels()
{
    reset_fw_recovery_level(SPI_FLASH_PCH);
    reset_fw_recovery_level(SPI_FLASH_BMC);
}

static void ut_reset_nios_fw()
{
    ut_reset_watchdog_timers();
    ut_reset_failed_update_attempts();
    ut_reset_fw_recovery_levels();
}

#endif /* UNITTEST_SYSTEM_UT_NIOS_WRAPPER_H_ */
