#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"
#include "testdata_info.h"

class PFRAntiRollbackWithSVNTest : public testing::Test
{
public:
    virtual void SetUp()
    {
        SYSTEM_MOCK* sys = SYSTEM_MOCK::get();
        // Reset system mocks and SPI flash
        sys->reset();
        sys->reset_spi_flash_mock();

        // Perform provisioning
        SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);

        // Reset Nios firmware
        ut_reset_nios_fw();

        // Load the entire image to flash
        SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, FULL_PFR_IMAGE_BMC_FILE, FULL_PFR_IMAGE_BMC_FILE_SIZE);
        SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_PCH, FULL_PFR_IMAGE_PCH_FILE, FULL_PFR_IMAGE_PCH_FILE_SIZE);
    }

    virtual void TearDown() {}
};

/**
 * @brief This test sends in an PCH active firmware update request with a capsule that has invalid SVN in PFM.
 * The PFM has 0xFF as SVN in the update capsule.
 */
TEST_F(PFRAntiRollbackWithSVNTest, test_active_fw_update_with_invalid_svn)
{
    /*
     * Test Preparation
     */
    // Load the update capsule
    SYSTEM_MOCK::get()->load_to_flash(
            SPI_FLASH_PCH,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN_255_FILE,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN_255_FILE_SIZE,
            get_ufm_pfr_data()->pch_staging_region);

    /*
     * Flow Preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 20 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);

    // Send the active update intent
    ut_send_in_update_intent_once_upon_entry_to_t0(MB_PCH_UPDATE_INTENT, MB_UPDATE_INTENT_PCH_ACTIVE_MASK);

    /*
     * Run Nios FW
     */
    // Run PFR Main. Always run with the timeout
    ASSERT_DURATION_LE(40, pfr_main());

    // Nios should transition from T0 to T-1 exactly once for firmware update
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_PCH_UPDATE_INTENT);

    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), alt_u32(0));

    // Expecting error
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(MAJOR_ERROR_UPDATE_FROM_PCH_FAILED));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(MINOR_ERROR_FW_UPDATE_INVALID_SVN));
}

/**
 * @brief This test sends in an PCH recovery firmware update request with a capsule that has invalid SVN in PFM.
 * The PFM has 0xFF as SVN in the update capsule.
 */
TEST_F(PFRAntiRollbackWithSVNTest, test_recovery_fw_update_with_invalid_svn)
{
    /*
     * Test Preparation
     */
    // Load the update capsule
    SYSTEM_MOCK::get()->load_to_flash(
            SPI_FLASH_PCH,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN_255_FILE,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN_255_FILE_SIZE,
            get_ufm_pfr_data()->pch_staging_region);

    /*
     * Flow Preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 20 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);

    // Send the active update intent
    ut_send_in_update_intent_once_upon_entry_to_t0(MB_PCH_UPDATE_INTENT, MB_UPDATE_INTENT_PCH_RECOVERY_MASK);

    /*
     * Run Nios FW
     */
    // Run PFR Main. Always run with the timeout
    ASSERT_DURATION_LE(40, pfr_main());

    // Nios should transition from T0 to T-1 exactly once for firmware update
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_PCH_UPDATE_INTENT);

    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), alt_u32(0));

    // Expecting error
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(MAJOR_ERROR_UPDATE_FROM_PCH_FAILED));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(MINOR_ERROR_FW_UPDATE_INVALID_SVN));
}

/**
 * @brief This test sends in an PCH active firmware update request with a capsule that has banned SVN in PFM.
 * The PFM has 0x7 as SVN in the update capsule.
 * In this test, the update intent is sent from "BMC" (i.e. sent to the BMC update intent register). This is
 * for exercising more FW path.
 */
TEST_F(PFRAntiRollbackWithSVNTest, test_active_fw_update_with_banned_svn)
{
    /*
     * Test Preparation
     */
    // Load the update capsule
    SYSTEM_MOCK::get()->load_to_flash(
            SPI_FLASH_BMC,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN7_FILE,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN7_FILE_SIZE,
            get_ufm_pfr_data()->bmc_staging_region + BMC_STAGING_REGION_PCH_UPDATE_CAPSULE_OFFSET);

    /*
     * Flow Preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 20 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);

    // Send the active update intent
    ut_send_in_update_intent_once_upon_entry_to_t0(MB_BMC_UPDATE_INTENT, MB_UPDATE_INTENT_PCH_ACTIVE_MASK);

    // Ban all SVNs up to 8 (excluding 8)
    write_ufm_svn(8, UFM_SVN_POLICY_PCH);

    /*
     * Run Nios FW
     */
    // Run PFR Main. Always run with the timeout
    ASSERT_DURATION_LE(40, pfr_main());

    // Nios should transition from T0 to T-1 exactly once for firmware update
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_BMC_UPDATE_INTENT);

    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), alt_u32(0));

    // Expecting error
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(MAJOR_ERROR_UPDATE_FROM_BMC_FAILED));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(MINOR_ERROR_FW_UPDATE_INVALID_SVN));
}

/**
 * @brief This test sends in an PCH recovery firmware update request with a capsule that has banned SVN in PFM.
 * The PFM has 0x7 as SVN in the update capsule.
 * In this test, the update intent is sent from "BMC" (i.e. sent to the BMC update intent register). This is
 * for exercising more FW path.
 */
TEST_F(PFRAntiRollbackWithSVNTest, test_recovery_fw_update_with_banned_svn)
{
    /*
     * Test Preparation
     */
    // Load the update capsule
    SYSTEM_MOCK::get()->load_to_flash(
            SPI_FLASH_BMC,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN7_FILE,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN7_FILE_SIZE,
            get_ufm_pfr_data()->bmc_staging_region + BMC_STAGING_REGION_PCH_UPDATE_CAPSULE_OFFSET);

    /*
     * Flow Preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 20 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);

    // Send the active update intent
    ut_send_in_update_intent_once_upon_entry_to_t0(MB_BMC_UPDATE_INTENT, MB_UPDATE_INTENT_PCH_RECOVERY_MASK);

    // Ban all SVNs up to 8 (excluding 8)
    write_ufm_svn(8, UFM_SVN_POLICY_PCH);

    /*
     * Run Nios FW
     */
    // Run PFR Main. Always run with the timeout
    ASSERT_DURATION_LE(40, pfr_main());

    // Nios should transition from T0 to T-1 exactly once for firmware update
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_BMC_UPDATE_INTENT);

    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), alt_u32(0));

    // Expecting error
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(MAJOR_ERROR_UPDATE_FROM_BMC_FAILED));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(MINOR_ERROR_FW_UPDATE_INVALID_SVN));
}

/**
 * @brief This test attempts to bump the PCH SVN with an active firmware update, which is not allowed.
 * If user wishes to bump the SVN, a recovery update must be issued.
 */
TEST_F(PFRAntiRollbackWithSVNTest, test_bump_svn_with_active_fw_update)
{
    /*
     * Test Preparation
     */
    // Load the update capsule
    SYSTEM_MOCK::get()->load_to_flash(
            SPI_FLASH_PCH,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN9_FILE,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN9_FILE_SIZE,
            get_ufm_pfr_data()->pch_staging_region);

    /*
     * Flow preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 20 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);

    // Send the active update intent
    ut_send_in_update_intent_once_upon_entry_to_t0(MB_PCH_UPDATE_INTENT, MB_UPDATE_INTENT_PCH_ACTIVE_MASK);

    /*
     * Run Nios FW
     */
    // Run PFR Main. Always run with the timeout
    ASSERT_DURATION_LE(80, pfr_main());

    // Expect timed boot to be completed
    EXPECT_EQ(read_from_mailbox(MB_PLATFORM_STATE), alt_u32(PLATFORM_STATE_T0_BOOT_COMPLETE));
    // Have entered T-1 three times: Boot up + going T-1 for active update
    EXPECT_EQ(SYSTEM_MOCK::get()->get_t_minus_1_counter(), alt_u32(2));

    // Nios should transition from T0 to T-1 exactly once for active update flow
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_PCH_UPDATE_INTENT);

    // Expecting no error
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(MAJOR_ERROR_UPDATE_FROM_PCH_FAILED));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(MINOR_ERROR_FW_UPDATE_INVALID_SVN));
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), alt_u32(0));

}

/**
 * @brief This test bumps the SVN with a firmware update and issues another update request with a capsule
 * that has lower SVN.
 */
TEST_F(PFRAntiRollbackWithSVNTest, test_pch_active_update_with_lower_svn)
{
    /*
     * Test Preparation
     */
    // Load the update capsule
    SYSTEM_MOCK::get()->load_to_flash(
            SPI_FLASH_PCH,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN9_FILE,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN9_FILE_SIZE,
            get_ufm_pfr_data()->pch_staging_region);

    /*
     * Flow preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 20 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);

    // Send the recovery update intent
    // Important: In order to bump the SVN, a recovery update must be issued.
    ut_send_in_update_intent_once_upon_entry_to_t0(MB_PCH_UPDATE_INTENT, MB_UPDATE_INTENT_PCH_RECOVERY_MASK);

    /*
     * Step 1: Perform recovery firmware update with capsule that has SVN 0x9
     */
    // Run PFR Main. Always run with the timeout
    ASSERT_DURATION_LE(80, pfr_main());

    // Expect timed boot to be completed
    EXPECT_EQ(read_from_mailbox(MB_PLATFORM_STATE), alt_u32(PLATFORM_STATE_T0_BOOT_COMPLETE));
    // Have entered T-1 three times: Boot up + going T-1 for active update + going T-1 for recovery update
    EXPECT_EQ(SYSTEM_MOCK::get()->get_t_minus_1_counter(), alt_u32(3));

    // Nios should transition from T0 to T-1 twice for recovery update flow
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(2));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_PCH_UPDATE_INTENT);

    // Expecting no error
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), alt_u32(0));

    /*
     * Step 2: Attempt another active firmware update with capsule that has SVN 0x7
     */
    // Load the update capsule
    SYSTEM_MOCK::get()->load_to_flash(
            SPI_FLASH_PCH,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN7_FILE,
            SIGNED_CAPSULE_PCH_PFM_WITH_SVN7_FILE_SIZE,
            get_ufm_pfr_data()->pch_staging_region);

    // Send the update intent
    write_to_mailbox(MB_PCH_UPDATE_INTENT, MB_UPDATE_INTENT_PCH_ACTIVE_MASK);

    // Process update intent in T0
    mb_update_intent_handler();

    // Nios should transition from T0 to T-1 once for active update flow
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_PCH_UPDATE_INTENT);

    // Expecting errors since the minimal allowed SVN is now 0x9.
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(MAJOR_ERROR_UPDATE_FROM_PCH_FAILED));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(MINOR_ERROR_FW_UPDATE_INVALID_SVN));
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), alt_u32(0));

    // Check the PCH SVN policy
    EXPECT_FALSE(is_svn_valid(UFM_SVN_POLICY_PCH, 0x8));
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_PCH, 0x9));
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_PCH, 0xA));

    // Make sure BMC/CPLD SVN policies are unaffected
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_BMC, 0x0));
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_BMC, 0x1));
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_CPLD, 0x0));
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_CPLD, 0x1));
}
