#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"

class AuthenticationNegativeTest : public testing::Test
{
public:
    virtual void SetUp()
    {
        SYSTEM_MOCK::get()->reset();

        // Provision the system (e.g. root key hash)
        SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);
    }

    virtual void TearDown() {}
};


TEST_F(AuthenticationNegativeTest, test_authenticate_signed_can_cert_with_corrupted_payload)
{
    // Load an example key cancellation certificate to memory
    alt_u8 signed_cert[KEY_CAN_CERT_FILE_SIZE];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(KEY_CAN_CERT_PCH_PFM_KEY2, (alt_u32*) signed_cert);

    signed_cert[SIGNATURE_SIZE] = 0xFF;
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) signed_cert));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_signed_can_cert_with_bad_key_id)
{
    // Load an example key cancellation certificate to memory
    alt_u8 signed_cert[KEY_CAN_CERT_FILE_SIZE];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(KEY_CAN_CERT_PCH_PFM_KEY255, (alt_u32*) signed_cert);

    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) signed_cert));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_signed_can_cert_with_bad_pc_length)
{
    // Load an example key cancellation certificate to memory
    alt_u8 signed_cert[KEY_CAN_CERT_BAD_PCH_LEGNTH_FILE_SIZE];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(KEY_CAN_CERT_BAD_PCH_LEGNTH, (alt_u32*) signed_cert);

    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) signed_cert));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_signed_can_cert_with_modified_reserved_area)
{
    // Load an example key cancellation certificate to memory
    alt_u8 signed_cert[KEY_CAN_CERT_FILE_SIZE];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(KEY_CAN_CERT_MODIFIED_RESERVED_AREA, (alt_u32*) signed_cert);

    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) signed_cert));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_binary_with_bad_magic_number)
{
    alt_u8 binary[128] = {0x19, 0xee, 0xca, 0xb6,};
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) binary));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_binary_with_0_as_pc_length)
{
    alt_u8 binary[128] = {0x19, 0xfd, 0xea, 0xb6, 0x0, 0x0, 0x0, 0x0};
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) binary));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_fake_cpld_capsule_with_enormous_pc_length)
{
    // magic (4 bytes), PC length (4 bytes), PC type (4 bytes)
    alt_u8 binary[128] = {0x19, 0xfd, 0xea, 0xb6, 0xfe, 0xff, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0};
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) binary));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_fake_pch_pfm_with_enormous_pc_length)
{
    // magic (4 bytes), PC length (4 bytes), PC type (4 bytes)
    alt_u8 binary[128] = {0x19, 0xfd, 0xea, 0xb6, 0xfe, 0xff, 0xff, 0xff, 0x01, 0x0, 0x0, 0x0};
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) binary));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_fake_pch_capsule_with_enormous_pc_length)
{
    // magic (4 bytes), PC length (4 bytes), PC type (4 bytes)
    alt_u8 binary[128] = {0x19, 0xfd, 0xea, 0xb6, 0xfe, 0xff, 0xff, 0xff, 0x02, 0x0, 0x0, 0x0};
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) binary));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_fake_bmc_pfm_with_enormous_pc_length)
{
    // magic (4 bytes), PC length (4 bytes), PC type (4 bytes)
    alt_u8 binary[128] = {0x19, 0xfd, 0xea, 0xb6, 0xfe, 0xff, 0xff, 0xff, 0x03, 0x0, 0x0, 0x0};
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) binary));
}

TEST_F(AuthenticationNegativeTest, test_authenticate_fake_bmc_capsule_with_enormous_pc_length)
{
    // magic (4 bytes), PC length (4 bytes), PC type (4 bytes)
    alt_u8 binary[128] = {0x19, 0xfd, 0xea, 0xb6, 0xfe, 0xff, 0xff, 0xff, 0x04, 0x0, 0x0, 0x0};
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) binary));
}

/**
 * @brief Authenticate a PCH firmware update capsule that has 10 as the PC Type value. That is invalid. Expecting
 * to see an authentication failure.
 */
TEST_F(AuthenticationNegativeTest, test_authenticate_pch_fw_capsule_with_invalid_pc_type)
{
    alt_u32* pch_update_capsule = new alt_u32[10771072 / 4];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat("testdata/bad_firmware_capsule/signed_capsule_pch_with_pc_type_10.bin", pch_update_capsule);

    // PC Type 10 is invalid
    EXPECT_EQ(get_kch_pc_type((KCH_BLOCK0*) pch_update_capsule), alt_u32(10));
    EXPECT_FALSE(is_block0_valid((KCH_BLOCK0*) pch_update_capsule, incr_alt_u32_ptr(pch_update_capsule, SIGNATURE_SIZE)));
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) pch_update_capsule));

    delete[] pch_update_capsule;
}

/**
 * @brief Authenticate a PCH firmware update capsule for which the CSK Key has been cancelled.
 */
TEST_F(AuthenticationNegativeTest, test_authenticate_pch_fw_capsule_with_cancelled_key)
{
    alt_u32* pch_update_capsule = new alt_u32[SIGNED_CAPSULE_PCH_WITH_CSK_ID10_FILE_SIZE / 4];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(SIGNED_CAPSULE_PCH_WITH_CSK_ID10_FILE, pch_update_capsule);

    // Cancel CSK Key #10 for signing PCH update capsule
    cancel_key(KCH_PC_PFR_PCH_UPDATE_CAPSULE, 10);

    // Authentication should fail
    EXPECT_FALSE(is_signature_valid((KCH_SIGNATURE*) pch_update_capsule));

    delete[] pch_update_capsule;
}
