#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"

class CapsuleValidationTest : public testing::Test
{
public:
    alt_u32* m_flash_x86_ptr = nullptr;

    // For simplicity, use PCH flash for all tests.
    SPI_FLASH_TYPE_ENUM m_spi_flash_in_use = SPI_FLASH_PCH;

    virtual void SetUp()
    {
        SYSTEM_MOCK* sys = SYSTEM_MOCK::get();
        sys->reset();

        sys->reset_spi_flash(m_spi_flash_in_use);
        m_flash_x86_ptr = sys->get_x86_ptr_to_spi_flash(m_spi_flash_in_use);
        switch_spi_flash(m_spi_flash_in_use);

        // Perform provisioning
        SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);
    }

    virtual void TearDown() {}
};

TEST_F(CapsuleValidationTest, test_sanity)
{
    // Just need the first two signature blocks
    SYSTEM_MOCK::get()->load_to_flash(m_spi_flash_in_use, SIGNED_CAPSULE_BMC_FILE, SIGNATURE_SIZE * 2);

    // Check the Block 0 Magic number
    EXPECT_EQ(*m_flash_x86_ptr, 0xB6EAFD19);

    alt_u8* m_bmc_flash_x86_u8_ptr = (alt_u8*) m_flash_x86_ptr;
    EXPECT_EQ(*m_bmc_flash_x86_u8_ptr, (alt_u8) 0x19);
    EXPECT_EQ(*(m_bmc_flash_x86_u8_ptr + 1), (alt_u8) 0xfd);
    EXPECT_EQ(*(m_bmc_flash_x86_u8_ptr + 2), (alt_u8) 0xea);
    EXPECT_EQ(*(m_bmc_flash_x86_u8_ptr + 3), (alt_u8) 0xb6);

    m_bmc_flash_x86_u8_ptr += 0x400;
    EXPECT_EQ(*m_bmc_flash_x86_u8_ptr, (alt_u8) 0x19);
    EXPECT_EQ(*(m_bmc_flash_x86_u8_ptr + 1), (alt_u8) 0xfd);
    EXPECT_EQ(*(m_bmc_flash_x86_u8_ptr + 2), (alt_u8) 0xea);
    EXPECT_EQ(*(m_bmc_flash_x86_u8_ptr + 3), (alt_u8) 0xb6);
}

TEST_F(CapsuleValidationTest, test_is_pbc_valid)
{
    SYSTEM_MOCK::get()->load_to_flash(m_spi_flash_in_use, SIGNED_CAPSULE_BMC_FILE, SIGNED_CAPSULE_BMC_FILE_SIZE);

    PBC_HEADER* pbc = get_pbc_ptr_from_signed_capsule(m_flash_x86_ptr);
    EXPECT_TRUE(is_pbc_valid(pbc));
}

TEST_F(CapsuleValidationTest, test_authenticate_signed_capsule_bmc)
{
    SYSTEM_MOCK::get()->load_to_flash(m_spi_flash_in_use, SIGNED_CAPSULE_BMC_FILE, SIGNED_CAPSULE_BMC_FILE_SIZE);
    EXPECT_TRUE(are_signatures_in_capsule_valid(m_flash_x86_ptr));
}

TEST_F(CapsuleValidationTest, test_is_capsule_valid_with_bmc)
{
    SYSTEM_MOCK::get()->load_to_flash(m_spi_flash_in_use, SIGNED_CAPSULE_BMC_FILE, SIGNED_CAPSULE_BMC_FILE_SIZE);
    EXPECT_TRUE(is_capsule_valid(m_flash_x86_ptr));
}

TEST_F(CapsuleValidationTest, test_is_capsule_valid_with_pch)
{
    SYSTEM_MOCK::get()->load_to_flash(m_spi_flash_in_use, SIGNED_CAPSULE_PCH_FILE, SIGNED_CAPSULE_PCH_FILE_SIZE);
    EXPECT_TRUE(is_capsule_valid(m_flash_x86_ptr));
}
