#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the PFR headers
// Always include the BSP mock then pfr_sys.h first
#include "bsp_mock.h"
#include "pfr_sys.h"
#include "cpld_update.h"
#include "cpld_recovery.h"
#include "ufm_utils.h"
#include "recovery_main.h"


// Other mock system header
#include "testdata_info.h"

// Other PFR headers
//#include "pfr_main.h"
#include "ut_nios_wrapper.h"

class CPLDRecoveryFlowTest : public testing::Test
{
public:
    alt_u32* m_flash_x86_ptr = nullptr;

    virtual void SetUp()
    {
        SYSTEM_MOCK* sys = SYSTEM_MOCK::get();
        sys->reset();
        sys->reset_spi_flash_mock();

        // Perform provisioning
        SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);

        // Reset fw update failed attempt
        reset_fw_update_failed_attempts(SPI_FLASH_PCH);
        reset_fw_update_failed_attempts(SPI_FLASH_BMC);
    }

    virtual void TearDown() {}
};

/*
 * This test runs the cpld recovery flow when only the factory backup is valid
 */
TEST_F(CPLDRecoveryFlowTest, test_cpld_recovery_from_factory)
{
    /*
     * Flow Preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();

    // Load the factory default image to BMC flash
    switch_spi_flash(SPI_FLASH_BMC);
    SYSTEM_MOCK::get()->set_mem_word((void*) (U_DUAL_CONFIG_BASE + (4 << 2)), 0);
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET);

    // Load a recovery image to CFM0
    alt_u32 *cfm0_image = new alt_u32[CFM0_RECOVERY_IMAGE_FILE_SIZE/4];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(CFM0_RECOVERY_IMAGE_FILE, cfm0_image);

    alt_u32* cfm0_ptr = get_ufm_ptr_with_offset(UFM_CPLD_RECOVERY_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < (CFM0_RECOVERY_IMAGE_FILE_SIZE/4); i++)
    {
        cfm0_ptr[i] = cfm0_image[i];
    }

    /*
     * Execute the flow
     */
    ASSERT_DURATION_LE(15, recovery_main());

    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x3, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));

    /*
     * Verify the results
     */
    // Check to see if CFM1 got updated properly
    alt_u32* bmc_cpld_update_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	    	        + get_ufm_pfr_data()->bmc_staging_region/4
    	    	        + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET/4;
    alt_u32* cpld_update_bitstream = ((CPLD_UPDATE_PC*)(bmc_cpld_update_backup_ptr + SIGNATURE_SIZE/4))->cpld_bitstream;
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < (UFM_CPLD_ACTIVE_IMAGE_LENGTH / 4); i++)
    {
        ASSERT_EQ(cfm1_ptr[i], cpld_update_bitstream[i]);
    }

    // Check to see if CFM0 is untouched
    for (alt_u32 i = 0; i < (UFM_CPLD_RECOVERY_IMAGE_LENGTH/4); i++)
    {
        ASSERT_EQ(cfm0_ptr[i], cfm0_image[i]);
    }

    /*
     * Clean up
     */
    delete[] cfm0_image;
}

/*
 * This test runs the cpld recovery flow when the factory backup and the backup in slot A is valid
 */
TEST_F(CPLDRecoveryFlowTest, test_cpld_recovery_slot_a_backup_valid)
{
    switch_spi_flash(SPI_FLASH_BMC);
    ut_set_gpi_nios_start_and_me_boot_done();
    SYSTEM_MOCK::get()->set_mem_word((void*) (U_DUAL_CONFIG_BASE + (4 << 2)), 0);
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET);

    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET + 4);

    alt_u32* bmc_cpld_update_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	    	        + get_ufm_pfr_data()->bmc_staging_region/4
    	    	        + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET/4;

    bmc_cpld_update_backup_ptr[0] = 1;

    ASSERT_DURATION_LE(15, recovery_main());

    // Check to see if we switched the configuration back to active
    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x3, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));

    // Check to see if we invalidated the slot A backup
    EXPECT_EQ((alt_u32) 0xffffffff, bmc_cpld_update_backup_ptr[0]);

    // Check to see if CFM1 got updated properly
    alt_u32* bmc_cpld_update_factory_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	    	        + get_ufm_pfr_data()->bmc_staging_region/4
    	    	        + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET/4;
    alt_u32 pc_size = ((KCH_BLOCK0*) bmc_cpld_update_factory_backup_ptr)->pc_length;
    alt_u32* cpld_update_bitstream = ((CPLD_UPDATE_PC*)(bmc_cpld_update_factory_backup_ptr + SIGNATURE_SIZE/4))->cpld_bitstream;
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < pc_size/4; i++)
    {
        ASSERT_EQ(cfm1_ptr[i], cpld_update_bitstream[i]);
    }
}

/*
 * This test runs the cpld recovery flow when the factory backup and the backup in slot B is valid
 */
TEST_F(CPLDRecoveryFlowTest, test_cpld_recovery_slot_b_backup_valid)
{
    switch_spi_flash(SPI_FLASH_BMC);
    ut_set_gpi_nios_start_and_me_boot_done();
    SYSTEM_MOCK::get()->set_mem_word((void*) (U_DUAL_CONFIG_BASE + (4 << 2)), 0);
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET);

    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET + 4);

    alt_u32* bmc_cpld_update_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	            + get_ufm_pfr_data()->bmc_staging_region/4
    	            + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET/4;

    bmc_cpld_update_backup_ptr[0] = 1;


    ASSERT_DURATION_LE(15, recovery_main());

    // Check to see if we switched the configuration back to active
    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x3, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));

    // Check to see if we invalidated the slot A backup
    EXPECT_EQ((alt_u32) 0xffffffff, bmc_cpld_update_backup_ptr[0]);

    // Check to see if CFM1 got updated properly
    alt_u32* bmc_cpld_update_factory_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	    	        + get_ufm_pfr_data()->bmc_staging_region/4
    	    	        + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET/4;
    alt_u32 pc_size = ((KCH_BLOCK0*) bmc_cpld_update_factory_backup_ptr)->pc_length;
    alt_u32* cpld_update_bitstream = ((CPLD_UPDATE_PC*)(bmc_cpld_update_factory_backup_ptr + SIGNATURE_SIZE/4))->cpld_bitstream;
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < pc_size/4; i++)
    {
        ASSERT_EQ(cfm1_ptr[i], cpld_update_bitstream[i]);
    }
}

/*
 * This test runs the cpld recovery flow when all 3 backups are valid
 */
TEST_F(CPLDRecoveryFlowTest, test_cpld_recovery_both_slots_valid)
{
    switch_spi_flash(SPI_FLASH_BMC);
    ut_set_gpi_nios_start_and_me_boot_done();
    SYSTEM_MOCK::get()->set_mem_word((void*) (U_DUAL_CONFIG_BASE + (4 << 2)), 0);
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET);

    // Create backup slots, slot a is currently the newest backup
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET + 4);

    alt_u32* bmc_cpld_slot_b_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	            + get_ufm_pfr_data()->bmc_staging_region/4
    	            + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET/4;

    bmc_cpld_slot_b_backup_ptr[0] = 1;

    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET + 4);

    alt_u32* bmc_cpld_slot_a_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	            + get_ufm_pfr_data()->bmc_staging_region/4
    	            + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET/4;

    bmc_cpld_slot_a_backup_ptr[0] = 2;


    ASSERT_DURATION_LE(15, recovery_main());

    // Check to see if we switched the configuration back to active
    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x3, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));

    // Check to see if we invalidated the slot A backup
    EXPECT_EQ((alt_u32) 0xffffffff, bmc_cpld_slot_a_backup_ptr[0]);

    // Check to see if CFM1 got updated properly

    // Skip over internal version number
    bmc_cpld_slot_b_backup_ptr++;

    alt_u32 pc_size = ((KCH_BLOCK0*) bmc_cpld_slot_b_backup_ptr)->pc_length;
    alt_u32* cpld_update_bitstream = ((CPLD_UPDATE_PC*)(bmc_cpld_slot_b_backup_ptr + SIGNATURE_SIZE/4))->cpld_bitstream;
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < pc_size/4; i++)
    {
        ASSERT_EQ(cfm1_ptr[i], cpld_update_bitstream[i]);
    }
}

/*
 * This test runs the cpld recovery flow when all backups are valid, but the target backup slot has been corrupted
 * fails authentication due to corruption
 */
TEST_F(CPLDRecoveryFlowTest, test_cpld_recovery_backup_slot_corrupted)
{
    switch_spi_flash(SPI_FLASH_BMC);
    ut_set_gpi_nios_start_and_me_boot_done();
    SYSTEM_MOCK::get()->set_mem_word((void*) (U_DUAL_CONFIG_BASE + (4 << 2)), 0);
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET);

    // Create backup slots, slot a is currently the newest backup
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET + 4);

    alt_u32* bmc_cpld_slot_b_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	            + get_ufm_pfr_data()->bmc_staging_region/4
    	            + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET/4;

    bmc_cpld_slot_b_backup_ptr[0] = 1;
    // corrupt the backup
    bmc_cpld_slot_b_backup_ptr[100] = 0x55555555;

    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET + 4);

    alt_u32* bmc_cpld_slot_a_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	            + get_ufm_pfr_data()->bmc_staging_region/4
    	            + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET/4;

    bmc_cpld_slot_a_backup_ptr[0] = 2;

    ASSERT_DURATION_LE(15, recovery_main());

    // Check to see if we switched the configuration back to active
    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x3, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));

    // Check to see if we invalidated both slots
    EXPECT_EQ((alt_u32) 0xffffffff, bmc_cpld_slot_a_backup_ptr[0]);
    EXPECT_EQ((alt_u32) 0xffffffff, bmc_cpld_slot_b_backup_ptr[0]);

    // Check to see if CFM1 got updated properly

    // Skip over internal version number
    alt_u32* bmc_cpld_update_factory_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	    	        + get_ufm_pfr_data()->bmc_staging_region/4
    	    	        + BMC_STAGING_REGION_CPLD_FACTORY_DEFAULT_IMAGE_OFFSET/4;

    alt_u32 pc_size = ((KCH_BLOCK0*) bmc_cpld_update_factory_backup_ptr)->pc_length;
    alt_u32* cpld_update_bitstream = ((CPLD_UPDATE_PC*)(bmc_cpld_update_factory_backup_ptr + SIGNATURE_SIZE/4))->cpld_bitstream;
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < pc_size/4; i++)
    {
        ASSERT_EQ(cfm1_ptr[i], cpld_update_bitstream[i]);
    }
}
