#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the PFR headers
// Always include the BSP mock then pfr_sys.h first
#include "bsp_mock.h"
#include "pfr_sys.h"
#include "cpld_update.h"
#include "ufm_utils.h"
#include "recovery_main.h"


// Other mock system header
#include "testdata_info.h"

// Other PFR headers
//#include "pfr_main.h"
#include "ut_nios_wrapper.h"

class CPLDUpdateFlowTest : public testing::Test
{
public:
    alt_u32* m_flash_x86_ptr = nullptr;

    virtual void SetUp()
    {
        SYSTEM_MOCK* sys = SYSTEM_MOCK::get();
        sys->reset();
        sys->reset_spi_flash_mock();

        // Perform provisioning
        SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);

        // Reset Nios firmware
        ut_reset_nios_fw();
    }

    virtual void TearDown() {}
};

/*
 * This test runs the portion cpld update flow that happens
 * before switching to the recovery system.
 * when the update capsule is in the bmc flash
 */
TEST_F(CPLDUpdateFlowTest, test_trigger_cpld_update_from_bmc)
{
    ASSERT_DURATION_LE(1, trigger_cpld_update());
    EXPECT_EQ(alt_u32(0x1), IORD(U_DUAL_CONFIG_BASE, 0));
    EXPECT_EQ(alt_u32(0x1), IORD(U_DUAL_CONFIG_BASE, 1));
}

/*
 * This test requests 12 multiple CPLD updates through BMC update intent.
 * Since there's no CPLD update capsule present in the flash, authentication should fail.
 */
TEST_F(CPLDUpdateFlowTest, test_multiple_failed_cpld_updates_from_bmc)
{
    /*
     * Flow preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 10 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);
    // Skip authentication in T-1
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::SKIP_FLASH_AUTHENTICATION);

    ut_send_in_update_intent(MB_BMC_UPDATE_INTENT, MB_UPDATE_INTENT_CPLD_MASK);
    // Run the flow
    ASSERT_DURATION_LE(1, pfr_main());

    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(MAX_CPLD_UPDATE_FAIL_ATTEMPTS));
}

/*
 * This test runs the portion cpld update flow that happens
 * before switching to the recovery system.
 * This test makes sure that we don't switch cfm when the authentication fails
 */
TEST_F(CPLDUpdateFlowTest, test_cpld_update_from_bmc_failed_authentication)
{
    write_to_mailbox(MB_BMC_UPDATE_INTENT, MB_UPDATE_INTENT_CPLD_MASK);
    ASSERT_DURATION_LE(1, act_on_bmc_update_intent());

    // Check that no CFM switch has occurred
    EXPECT_EQ(alt_u32(0), IORD(U_DUAL_CONFIG_BASE, 0));
    EXPECT_EQ(alt_u32(0), IORD(U_DUAL_CONFIG_BASE, 1));

    // Check the major/minor error
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(MAJOR_ERROR_UPDATE_FROM_BMC_FAILED));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(MINOR_ERROR_CPLD_UPDATE_AUTH_FAILED));
}

/*
 * This test runs the portion cpld update flow that happens
 * after switching to the recovery system.
 */
TEST_F(CPLDUpdateFlowTest, test_cpld_update_post_reconfig)
{
    switch_spi_flash(SPI_FLASH_BMC);
    alt_u32 cpld_slota_image_offset_in_flash = get_ufm_pfr_data()->bmc_staging_region + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET;
    alt_u32 cpld_slotb_image_offset_in_flash = get_ufm_pfr_data()->bmc_staging_region + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET;

    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET);

    ASSERT_DURATION_LE(16, perform_update_post_reconfig(cpld_slota_image_offset_in_flash, cpld_slotb_image_offset_in_flash));

    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x3, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));

    alt_u32* bmc_cpld_update_capsule_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    			        + get_ufm_pfr_data()->bmc_staging_region/4
    			        + BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET/4;

    alt_u32* bmc_cpld_update_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	    	        + get_ufm_pfr_data()->bmc_staging_region/4
    	    	        + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET/4;

    // Check to see if the update got copied to the backup slot properly
    EXPECT_EQ((alt_u32) 0x0, bmc_cpld_update_backup_ptr[0]);
    for (int i = 1; i <SIGNED_CAPSULE_CPLD_FILE_SIZE/4; i++)
    {
        ASSERT_EQ(bmc_cpld_update_capsule_ptr[i], bmc_cpld_update_backup_ptr[i + 1]);
    }

    // Check to see if the SVN got updated
    EXPECT_EQ((alt_u32) 1, get_ufm_svn(UFM_SVN_POLICY_CPLD));

    // Check to see if CFM1 got updated properly
    alt_u32 pc_size = ((KCH_BLOCK0*) bmc_cpld_update_capsule_ptr)->pc_length;
    alt_u32* cpld_update_bitstream = ((CPLD_UPDATE_PC*)(bmc_cpld_update_capsule_ptr + SIGNATURE_SIZE/4))->cpld_bitstream;
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < pc_size/4; i++)
    {
        ASSERT_EQ(cfm1_ptr[i], cpld_update_bitstream[i]);
    }
}

/*
 * This test runs the portion cpld update flow that happens
 * after switching to the recovery system. When authentication fails
 */
TEST_F(CPLDUpdateFlowTest, test_cpld_update_post_reconfig_failed_auth)
{
    switch_spi_flash(SPI_FLASH_BMC);
    alt_u32 cpld_slota_image_offset_in_flash = get_ufm_pfr_data()->bmc_recovery_region + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET;
    alt_u32 cpld_slotb_image_offset_in_flash = get_ufm_pfr_data()->bmc_recovery_region  + BMC_STAGING_REGION_CPLD_SLOTB_IMAGE_OFFSET;

    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET);

    alt_u32* bmc_cpld_update_capsule_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    			        + get_ufm_pfr_data()->bmc_staging_region/4
    			        + BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET/4;

    alt_u32* bmc_cpld_update_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	    	        + get_ufm_pfr_data()->bmc_recovery_region/4
    	    	        + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET/4;

    // Corrupt the update capsule
    bmc_cpld_update_capsule_ptr[0] = 0xffffffff;


    ASSERT_DURATION_LE(16, perform_update_post_reconfig(cpld_slota_image_offset_in_flash, cpld_slotb_image_offset_in_flash));

    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x3, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));

    // Check to see if the update slot remains blank
    ASSERT_EQ((alt_u32) 0xFFFFFFFF, bmc_cpld_update_backup_ptr[0]);
    for (int i = 1; i <SIGNED_CAPSULE_CPLD_FILE_SIZE/4; i++)
    {
        ASSERT_EQ((alt_u32) 0xFFFFFFFF, bmc_cpld_update_backup_ptr[i + 1]);
    }

    // Check to see if the SVN did not get updated
    EXPECT_EQ((alt_u32) 0, get_ufm_svn(UFM_SVN_POLICY_CPLD));

    // Check to see if CFM1 remains black
    alt_u32 pc_size = ((KCH_BLOCK0*) bmc_cpld_update_capsule_ptr)->pc_length;
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = 1; i < pc_size/4; i++)
    {
        ASSERT_EQ((alt_u32) 0xFFFFFFFF, cfm1_ptr[i]);
    }
}

/*
 * This test tests to see if the main code handles a cpld update intent form the bmc
 */
TEST_F(CPLDUpdateFlowTest, test_cpld_update_bmc_intent)
{
    switch_spi_flash(SPI_FLASH_BMC);
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET);

    /*
     * Flow preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 10 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);
    // Skip authentication in T-1
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::SKIP_FLASH_AUTHENTICATION);

    ut_send_in_update_intent_once_upon_boot_complete(MB_BMC_UPDATE_INTENT, MB_UPDATE_INTENT_CPLD_MASK);
    // Run the flow
    ASSERT_DURATION_LE(16, pfr_main());

    //ASSERT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));
}

/*
 * This test tests to see if the main code handles a cpld update intent form the bmc
 * when the update fails authentication
 */
TEST_F(CPLDUpdateFlowTest, test_cpld_update_bmc_intent_negative_test)
{

    switch_spi_flash(SPI_FLASH_BMC);

    /*
     * Flow preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 10 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);
    // Skip authentication in T-1
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::SKIP_FLASH_AUTHENTICATION);

    ut_send_in_update_intent_once_upon_boot_complete(MB_BMC_UPDATE_INTENT, MB_UPDATE_INTENT_CPLD_MASK);
    // Run the flow
    ASSERT_DURATION_LE(16, pfr_main());

    //ASSERT_EQ((alt_u32) 0x0, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x0, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));
}

/**
 * @brief This test attempts to send CPLD update intent to PCH update intent register.
 * CPLD should not go into T-1 mode since this in-band CPLD update is not supported.
 */
TEST_F(CPLDUpdateFlowTest, test_cpld_update_pch_intent_negative_test)
{
    switch_spi_flash(SPI_FLASH_PCH);

    /*
     * Flow preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    ut_send_block_complete_chkpt_msg();

    // Exit after 10 iterations in the T0 loop
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::T0_OPERATIONS_END_AFTER_20_ITERS);
    // Skip authentication in T-1
    SYSTEM_MOCK::get()->insert_code_block(SYSTEM_MOCK::CODE_BLOCK_TYPES::SKIP_FLASH_AUTHENTICATION);

    ut_send_in_update_intent_once_upon_boot_complete(MB_PCH_UPDATE_INTENT, MB_UPDATE_INTENT_CPLD_MASK);
    // Run the flow
    ASSERT_DURATION_LE(1, pfr_main());

    // Expect Nios fw only entered T-1 mode once during boot up
    EXPECT_EQ(SYSTEM_MOCK::get()->get_t_minus_1_counter(), alt_u32(1));
    // Expect no BMC-only reset happened
    EXPECT_EQ(SYSTEM_MOCK::get()->get_t_minus_1_bmc_only_counter(), alt_u32(0));

    // Expect update failure
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(MAJOR_ERROR_UPDATE_FROM_PCH_FAILED));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(MINOR_ERROR_INVALID_UPDATE_INTENT));

    // Expect no panic or recovery event
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), alt_u32(0));
}

/*
 * This test tests the full cpld update flow starting from entry into recovery_main
 */
TEST_F(CPLDUpdateFlowTest, test_cpld_update_post_reconfig_full_flow)
{
    /*
     * Flow preparation
     */
    ut_set_gpi_nios_start_and_me_boot_done();
    SYSTEM_MOCK::get()->set_mem_word((void*) (U_DUAL_CONFIG_BASE + (4 << 2)), RECONFIG_CPLD_UPDATE << 13);

    switch_spi_flash(SPI_FLASH_BMC);
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_CPLD_FILE,
            SIGNED_CAPSULE_CPLD_FILE_SIZE, get_ufm_pfr_data()->bmc_staging_region
            + BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET);
    // recovery_main() looks for the breadcrumb to be 0 to determine if we are in recovery from a power cycle or a reconfig
    alt_u32* bread_crumb_ptr = get_ufm_ptr_with_offset(CFM1_BREAD_CRUMB);
    *bread_crumb_ptr = 0x0;

    // Load a recovery image to CFM0
    alt_u32 *cfm0_image = new alt_u32[CFM0_RECOVERY_IMAGE_FILE_SIZE/4];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(CFM0_RECOVERY_IMAGE_FILE, cfm0_image);

    alt_u32* cfm0_ptr = get_ufm_ptr_with_offset(UFM_CPLD_RECOVERY_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < (CFM0_RECOVERY_IMAGE_FILE_SIZE/4); i++)
    {
        cfm0_ptr[i] = cfm0_image[i];
    }

    /*
     * Execute flow
     */
    ASSERT_DURATION_LE(16, recovery_main());

    EXPECT_EQ((alt_u32) 0x1, SYSTEM_MOCK::get()->get_mem_word((void*) U_DUAL_CONFIG_BASE));
    EXPECT_EQ((alt_u32) 0x3, SYSTEM_MOCK::get()->get_mem_word((void*) (U_DUAL_CONFIG_BASE  + 4)));

    alt_u32* bmc_cpld_update_capsule_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    			        + get_ufm_pfr_data()->bmc_staging_region/4
    			        + BMC_STAGING_REGION_CPLD_UPDATE_CAPSULE_OFFSET/4;

    alt_u32* bmc_cpld_update_backup_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC)
    	    	        + get_ufm_pfr_data()->bmc_staging_region/4
    	    	        + BMC_STAGING_REGION_CPLD_SLOTA_IMAGE_OFFSET/4;

    // Check to see if the update got copied to the backup slot properly
    EXPECT_EQ((alt_u32) 0x0, bmc_cpld_update_backup_ptr[0]);
    for (int i = 1; i < (SIGNED_CAPSULE_CPLD_FILE_SIZE/4); i++)
    {
        ASSERT_EQ(bmc_cpld_update_capsule_ptr[i], bmc_cpld_update_backup_ptr[i + 1]);
    }

    // Check to see if the SVN got updated
    EXPECT_EQ((alt_u32) 1, get_ufm_svn(UFM_SVN_POLICY_CPLD));

    // Check to see if CFM1 got updated properly
    alt_u32* cpld_update_bitstream = ((CPLD_UPDATE_PC*)(bmc_cpld_update_capsule_ptr + SIGNATURE_SIZE/4))->cpld_bitstream;
    alt_u32* cfm1_ptr = get_ufm_ptr_with_offset(UFM_CPLD_ACTIVE_IMAGE_OFFSET);
    for (alt_u32 i = 0; i < (UFM_CPLD_ACTIVE_IMAGE_LENGTH/4); i++)
    {
        ASSERT_EQ(cfm1_ptr[i], cpld_update_bitstream[i]);
    }

    // Check to see if CFM0 is untouched
    for (alt_u32 i = 0; i < (UFM_CPLD_RECOVERY_IMAGE_LENGTH/4); i++)
    {
        ASSERT_EQ(cfm0_ptr[i], cfm0_image[i]);
    }

    /*
     * Clean up
     */
    delete[] cfm0_image;
}
