#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"
#include "testdata_info.h"


class RecoveryFlowTest : public testing::Test
{
public:
    virtual void SetUp()
    {
        SYSTEM_MOCK* sys = SYSTEM_MOCK::get();
        // Reset system mocks and SPI flash
        sys->reset();
        sys->reset_spi_flash_mock();

        // Perform provisioning
        SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);

        // Reset Nios firmware
        ut_reset_nios_fw();
    }

    virtual void TearDown() {}
};

TEST_F(RecoveryFlowTest, test_static_recovery_on_pch)
{
    /*
     * Preparation
     */
    switch_spi_flash(SPI_FLASH_PCH);
    alt_u32* flash_x86_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_PCH);

    // Load the entire image to flash
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_PCH, FULL_PFR_IMAGE_PCH_FILE, FULL_PFR_IMAGE_PCH_FILE_SIZE);

    // Load the entire image locally for comparison purpose
    alt_u32 *full_image = new alt_u32[FULL_PFR_IMAGE_PCH_FILE_SIZE/4];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(FULL_PFR_IMAGE_PCH_FILE, full_image);

    /*
     * Erase read-only SPI regions and verify the erase
     */
    // Erase the target SPI region first so that we can show a recovery is performed
    for (alt_u32 region_i = 0; region_i < PCH_NUM_STATIC_REGIONS; region_i++)
    {
    	alt_u32 start_addr = testdata_pch_static_regions_start_addr[region_i];
    	alt_u32 end_addr = testdata_pch_static_regions_end_addr[region_i] - start_addr;
        erase_spi_region(start_addr, end_addr);
    }

    // For sanity purpose, verify the erase on some of SPI regions
    for (alt_u32 region_i = 0; region_i < PCH_NUM_STATIC_REGIONS; region_i++)
    {
        for (alt_u32 word_i = testdata_pch_static_regions_start_addr[region_i] >> 2;
                word_i < testdata_pch_static_regions_end_addr[region_i] >> 2; word_i++)
        {
            ASSERT_EQ(alt_u32(0xffffffff), flash_x86_ptr[word_i]);
        }
    }

    /*
     * Erase active PFM and verify the erase
     */
    alt_u32 pch_active_pfm_start = get_ufm_pfr_data()->pch_active_pfm;
    alt_u32 pch_active_pfm_end = pch_active_pfm_start + get_signed_payload_size(get_spi_active_pfm_ptr(SPI_FLASH_PCH));
    erase_spi_region(pch_active_pfm_start, get_signed_payload_size(get_spi_active_pfm_ptr(SPI_FLASH_PCH)));

    for (alt_u32 word_i = pch_active_pfm_start >> 2; word_i < pch_active_pfm_end >> 2; word_i++)
    {
        ASSERT_EQ(alt_u32(0xffffffff), flash_x86_ptr[word_i]);
    }

    /*
     * Perform the recovery
     */
    alt_u32* recovery_region_ptr = get_spi_recovery_region_ptr(SPI_FLASH_PCH);
    decompress_capsule(recovery_region_ptr, SPI_FLASH_PCH, DECOMPRESSION_STATIC_REGIONS_MASK);

    /*
     * Verify recovered data
     */
    for (alt_u32 region_i = 0; region_i < PCH_NUM_STATIC_REGIONS; region_i++)
    {
        for (alt_u32 word_i = testdata_pch_static_regions_start_addr[region_i] >> 2;
                word_i < testdata_pch_static_regions_end_addr[region_i] >> 2; word_i++)
        {
            ASSERT_EQ(full_image[word_i], flash_x86_ptr[word_i]);
        }
    }

    // Verify updated PFM
    for (alt_u32 word_i = pch_active_pfm_start >> 2; word_i < pch_active_pfm_end >> 2; word_i++)
    {
        ASSERT_EQ(full_image[word_i], flash_x86_ptr[word_i]);
    }

    /*
     * Clean ups
     */
    delete[] full_image;
}

TEST_F(RecoveryFlowTest, test_static_recovery_on_bmc)
{
    /*
     * Preparation
     */
    switch_spi_flash(SPI_FLASH_BMC);
    alt_u32* flash_x86_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC);

    // Load the entire image to flash
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, FULL_PFR_IMAGE_BMC_FILE, FULL_PFR_IMAGE_BMC_FILE_SIZE);

    // Load the entire image locally for comparison purpose
    alt_u32 *full_image = new alt_u32[FULL_PFR_IMAGE_BMC_FILE_SIZE/4];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(FULL_PFR_IMAGE_BMC_FILE, full_image);

    /*
     * Erase read-only SPI regions and verify the erase
     */
    for (alt_u32 region_i = 0; region_i < BMC_NUM_STATIC_REGIONS; region_i++)
    {
    	alt_u32 start_addr = testdata_bmc_static_regions_start_addr[region_i];
    	alt_u32 end_addr = testdata_bmc_static_regions_end_addr[region_i] - start_addr;
        erase_spi_region(start_addr, end_addr);
    }

    // For sanity purpose, verify the erase on some of SPI regions
    for (alt_u32 region_i = 0; region_i < BMC_NUM_STATIC_REGIONS; region_i++)
    {
        for (alt_u32 word_i = testdata_bmc_static_regions_start_addr[region_i] >> 2;
                word_i < testdata_bmc_static_regions_end_addr[region_i] >> 2; word_i++)
        {
            ASSERT_EQ(alt_u32(0xffffffff), flash_x86_ptr[word_i]);
        }
    }

    /*
     * Erase active PFM and verify the erase
     */
    alt_u32 bmc_active_pfm_start = get_ufm_pfr_data()->bmc_active_pfm;
    alt_u32 bmc_active_pfm_end = bmc_active_pfm_start + get_signed_payload_size(get_spi_active_pfm_ptr(SPI_FLASH_BMC));
    erase_spi_region(bmc_active_pfm_start, get_signed_payload_size(get_spi_active_pfm_ptr(SPI_FLASH_BMC)));

    for (alt_u32 word_i = bmc_active_pfm_start >> 2; word_i < bmc_active_pfm_end >> 2; word_i++)
    {
        ASSERT_EQ(alt_u32(0xffffffff), flash_x86_ptr[word_i]);
    }

    /*
     * Perform the recovery
     */
    alt_u32* recovery_region_ptr = get_spi_recovery_region_ptr(SPI_FLASH_BMC);
    decompress_capsule(recovery_region_ptr, SPI_FLASH_BMC, DECOMPRESSION_STATIC_REGIONS_MASK);

    /*
     * Verify recovered data
     */
    for (alt_u32 region_i = 0; region_i < BMC_NUM_STATIC_REGIONS; region_i++)
    {
        for (alt_u32 word_i = testdata_bmc_static_regions_start_addr[region_i] >> 2;
                word_i < testdata_bmc_static_regions_end_addr[region_i] >> 2; word_i++)
        {
            ASSERT_EQ(full_image[word_i], flash_x86_ptr[word_i]);
        }
    }

    // Verify updated PFM
    for (alt_u32 word_i = bmc_active_pfm_start >> 2; word_i < bmc_active_pfm_end >> 2; word_i++)
    {
        ASSERT_EQ(full_image[word_i], flash_x86_ptr[word_i]);
    }

    /*
     * Clean ups
     */
    delete[] full_image;
}

/*
 *
 */
TEST_F(RecoveryFlowTest, test_static_recovery_on_corrupted_pfm_bmc)
{
    /*
     * Preparation
     */
    switch_spi_flash(SPI_FLASH_BMC);
    // Load the entire image to flash
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, FULL_PFR_IMAGE_BMC_FILE, FULL_PFR_IMAGE_BMC_FILE_SIZE);

    // Load the entire image locally for comparison purpose
    alt_u32 *full_image = new alt_u32[FULL_PFR_IMAGE_BMC_FILE_SIZE/4];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(FULL_PFR_IMAGE_BMC_FILE, full_image);

    /*
     * Corrupt the signed active PFM
     */
    alt_u32* pfm_header = (alt_u32*) get_active_pfm(SPI_FLASH_BMC);
    *pfm_header = 0xdeadbeef;

    pfm_header = (pfm_header - 1);
    *pfm_header = 0xdeadbeef;

    /*
     * Perform the recovery
     */
    alt_u32* recovery_region_ptr = get_spi_recovery_region_ptr(SPI_FLASH_BMC);
    decompress_capsule(recovery_region_ptr, SPI_FLASH_BMC, DECOMPRESSION_STATIC_REGIONS_MASK);

    /*
     * Verify recovered data
     */
    alt_u32* signed_capsule_pfm = incr_alt_u32_ptr(get_spi_recovery_region_ptr(SPI_FLASH_BMC), SIGNATURE_SIZE);
    alt_u32* signed_active_pfm = get_spi_active_pfm_ptr(SPI_FLASH_BMC);
    alt_u32 nbytes = SIGNATURE_SIZE + ((KCH_BLOCK0*) signed_capsule_pfm)->pc_length;

    for (alt_u32 word_i = 0; word_i < nbytes / 4; word_i++)
    {
        alt_u32 expected_word = full_image[(get_ufm_pfr_data()->bmc_active_pfm/4) + word_i];
        ASSERT_EQ(expected_word, signed_active_pfm[word_i]);
    }

    /*
     * Clean ups
     */
    delete[] full_image;
}

TEST_F(RecoveryFlowTest, test_wdt_recovery_on_bmc)
{
    /*
     * Preparation
     */
    switch_spi_flash(SPI_FLASH_BMC);
    alt_u32* flash_x86_ptr = SYSTEM_MOCK::get()->get_x86_ptr_to_spi_flash(SPI_FLASH_BMC);

    // Load the entire image to flash
    SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, FULL_PFR_IMAGE_BMC_FILE, FULL_PFR_IMAGE_BMC_FILE_SIZE);

    // Load the entire image locally for comparison purpose
    alt_u32 *full_image = new alt_u32[FULL_PFR_IMAGE_BMC_FILE_SIZE/4];
    SYSTEM_MOCK::get()->init_x86_mem_from_dat(FULL_PFR_IMAGE_BMC_FILE, full_image);

    /*
     * Erase read-only SPI regions and verify the erase
     */
    for (alt_u32 region_i = 0; region_i < BMC_NUM_STATIC_REGIONS; region_i++)
    {
    	alt_u32 start_addr = testdata_bmc_static_regions_start_addr[region_i];
    	alt_u32 end_addr = testdata_bmc_static_regions_end_addr[region_i] - start_addr;
        erase_spi_region(start_addr, end_addr);
    }
    for (alt_u32 region_i = 0; region_i < BMC_NUM_DYNAMIC_REGIONS; region_i++)
    {
    	alt_u32 start_addr = testdata_bmc_dynamic_regions_start_addr[region_i];
    	alt_u32 end_addr = testdata_bmc_dynamic_regions_end_addr[region_i] - start_addr;
        erase_spi_region(start_addr, end_addr);
    }

    // For sanity purpose, verify the erase on some of SPI regions
    for (alt_u32 region_i = 0; region_i < BMC_NUM_STATIC_REGIONS; region_i++)
    {
        for (alt_u32 word_i = testdata_bmc_static_regions_start_addr[region_i] >> 2;
                word_i < testdata_bmc_static_regions_end_addr[region_i] >> 2; word_i++)
        {
            ASSERT_EQ(alt_u32(0xffffffff), flash_x86_ptr[word_i]);
        }
    }
    for (alt_u32 region_i = 0; region_i < BMC_NUM_DYNAMIC_REGIONS; region_i++)
    {
        for (alt_u32 word_i = testdata_bmc_dynamic_regions_start_addr[region_i] >> 2;
                word_i < testdata_bmc_dynamic_regions_end_addr[region_i] >> 2; word_i++)
        {
            ASSERT_EQ(alt_u32(0xffffffff), flash_x86_ptr[word_i]);
        }
    }

    /*
     * Perform the recovery
     */
    set_spi_flash_state(SPI_FLASH_BMC, SPI_FLASH_STATE_REQUIRE_WDT_RECOVERY_MASK);
    perform_wdt_recovery(SPI_FLASH_BMC);

    /*
     * Verify recovered data
     */
    for (alt_u32 region_i = 0; region_i < BMC_NUM_STATIC_REGIONS; region_i++)
    {
        for (alt_u32 word_i = testdata_bmc_static_regions_start_addr[region_i] >> 2;
                word_i < testdata_bmc_static_regions_end_addr[region_i] >> 2; word_i++)
        {
            ASSERT_EQ(full_image[word_i], flash_x86_ptr[word_i]);
        }
    }
    for (alt_u32 region_i = 0; region_i < BMC_NUM_DYNAMIC_REGIONS; region_i++)
    {
        for (alt_u32 word_i = testdata_bmc_dynamic_regions_start_addr[region_i] >> 2;
                word_i < testdata_bmc_dynamic_regions_end_addr[region_i] >> 2; word_i++)
        {
            ASSERT_EQ(full_image[word_i], flash_x86_ptr[word_i]);
        }
    }

    /*
     * Clean ups
     */
    delete[] full_image;
}
