#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"

class RFNVRamUtilsTest : public testing::Test
{
public:
    alt_u32* m_rfnvram_memory = nullptr;

    virtual void SetUp() {
        SYSTEM_MOCK::get()->reset();
        m_rfnvram_memory = U_RFNVRAM_SMBUS_MASTER_ADDR;
    }

    virtual void TearDown() {  }
};

TEST_F(RFNVRamUtilsTest, test_basic_rw)
{
    // write to addr 0
    IOWR(m_rfnvram_memory, 0, 0x2DC);
    IOWR(m_rfnvram_memory, 0, 0x00);
    IOWR(m_rfnvram_memory, 0, 0x00);
    IOWR(m_rfnvram_memory, 0, 0xAB);
    IOWR(m_rfnvram_memory, 0, 0x1DC);
    // read from addr 0
    IOWR(m_rfnvram_memory, 0, 0x2DC);
    IOWR(m_rfnvram_memory, 0, 0x00);
    IOWR(m_rfnvram_memory, 0, 0x00);
    IOWR(m_rfnvram_memory, 0, 0x2DD);
    IOWR(m_rfnvram_memory, 0, 0x00);
    IOWR(m_rfnvram_memory, 0, 0x100);

    while (IORD_32DIRECT(RFNVRAM_RX_FIFO_BYTES_LEFT, 0) != 2) {}

    EXPECT_EQ(IORD(m_rfnvram_memory, 1), (alt_u32) 0xAB);
    EXPECT_EQ(IORD(m_rfnvram_memory, 1), (alt_u32) 0xDC);
}

TEST_F(RFNVRamUtilsTest, test_read_pit_password)
{
    alt_u8 example_pit_password[RFNVRAM_PIT_PASSWORD_LENGTH] = {0x1b, 0x2a, 0xef, 0xa8, 0x11, 0x5c, 0x80, 0x37};
    alt_u8 read_pit_password[RFNVRAM_PIT_PASSWORD_LENGTH] = {};

    // Write PIT password
    IOWR(m_rfnvram_memory, 0, 0x2DC);
    IOWR(m_rfnvram_memory, 0, RFNVRAM_PIT_PASSWORD_MSB_OFFSET);
    IOWR(m_rfnvram_memory, 0, RFNVRAM_PIT_PASSWORD_LSB_OFFSET);
    for (alt_u32 byte_i = 0; byte_i < RFNVRAM_PIT_PASSWORD_LENGTH - 1; byte_i++)
    {
        IOWR(m_rfnvram_memory, 0, example_pit_password[byte_i]);
    }
    IOWR(m_rfnvram_memory, 0, 0x100 | example_pit_password[RFNVRAM_PIT_PASSWORD_LENGTH - 1]);

    // Read PIT password
    read_pit_password_from_rfnvram(read_pit_password);

    // Check results
    for (alt_u32 byte_i = 0; byte_i < RFNVRAM_PIT_PASSWORD_LENGTH; byte_i++)
    {
        EXPECT_EQ(example_pit_password[byte_i], read_pit_password[byte_i]);
    }
}

TEST_F(RFNVRamUtilsTest, test_write_then_read_pit_password)
{
    alt_u8 example_pit_password[RFNVRAM_PIT_PASSWORD_LENGTH] = {0x71, 0xb9, 0xef, 0xa8, 0x11, 0x54, 0x80, 0x33};
    alt_u8 read_pit_password[RFNVRAM_PIT_PASSWORD_LENGTH] = {};

    // Save PIT password to UFM
    alt_u8* ufm_pit_password = (alt_u8*) get_ufm_pfr_data()->pit_password;
    for (alt_u32 byte_i = 0; byte_i < RFNVRAM_PIT_PASSWORD_LENGTH; byte_i++)
    {
        ufm_pit_password[byte_i] = example_pit_password[byte_i];
    }

    // Write PIT password from UFM to RFNVRAM
    write_ufm_pit_password_to_rfnvram();

    // Read PIT password
    read_pit_password_from_rfnvram(read_pit_password);

    // Check results
    for (alt_u32 byte_i = 0; byte_i < RFNVRAM_PIT_PASSWORD_LENGTH; byte_i++)
    {
        EXPECT_EQ(example_pit_password[byte_i], read_pit_password[byte_i]);
    }
}
