#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"


class SPIFilterTest : public testing::Test
{
public:
    virtual void SetUp()
    {
        SYSTEM_MOCK::get()->reset();

        // Perform provisioning
        SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);
    }

    virtual void TearDown() {}
};

/*
 * Test SPI filtering with some hand-made SPI region definitions.
 */
TEST_F(SPIFilterTest, test_spi_filtering_with_custom_pfm_case1)
{
    switch_spi_flash(SPI_FLASH_PCH);

    // Define a custom PFM for PCH
    //   SPI Region 1: 0x0000 - 0x5C000 is RO
    //   SPI Region 2: 0x5C000 - 0x118000 is RW
    //   SPI Region 3: 0x118000 - 0x31C000 is RO
    //   SPI Region 4: 0x31C000 - 0x4000000 is RW
    alt_u8 pfm_data_bin[256] = {
      // PFM header
      0x1d, 0xce, 0xb3, 0x02, 0x03, 0x01, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0x00, 0x01, 0x00, 0x00,
      // PFM body
      // SPI Region 1
      0x01, 0x01, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      // SPI Region 2
      0x01, 0x03, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0xc0, 0x05, 0x00,
      0x00, 0x80, 0x11, 0x00,
      // Some SMBus rule
      0x02, 0xff, 0xff, 0xff,
      0xff, 0x01, 0x05, 0xa0, 0x2b, 0xe6, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      // SPI Region 3
      0x01, 0x01, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x80, 0x11, 0x00,
      0x00, 0xc0, 0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      // Some SMBus rule
      0x02, 0xff, 0xff, 0xff, 0xff, 0x03, 0x05, 0xdc, 0x2b, 0xe6, 0x0a, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00,
      // SPI Region 4
      0x01, 0x03, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0xc0, 0x31, 0x00,
      0x00, 0x00, 0x00, 0x04,
      // Padding
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff
    };

    // Save the PFM in the SPI flash
    alt_u32_memcpy((alt_u32*) get_active_pfm(SPI_FLASH_PCH), (alt_u32*) pfm_data_bin, 256);

    // Apply SPI write protection based on the PFM
    apply_spi_write_protection_and_smbus_rules(SPI_FLASH_PCH);

    // SPI regions 1 & 2
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 0), alt_u32(0xFF800000));
    // SPI region 2
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 1), alt_u32(0xFFFFFFFF));
    // SPI region 2 & 3
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 2), alt_u32(0x0000003F));
    // SPI region 3
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 3), alt_u32(0x00000000));
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 4), alt_u32(0x00000000));
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 5), alt_u32(0x00000000));
    // SPI region 3 & 4
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 6), alt_u32(0xFFFFFF80));

    // SPI region 4
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 7), alt_u32(0xFFFFFFFF));
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, 8), alt_u32(0xFFFFFFFF));
    // Expected maximum size of PCH flash is PCH_SPI_FLASH_SIZE
    alt_u32 last_word_pos_for_pch_flash_in_we_mem = PCH_SPI_FLASH_SIZE >> (14 + 5);
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, last_word_pos_for_pch_flash_in_we_mem - 2), alt_u32(0xFFFFFFFF));
    EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, last_word_pos_for_pch_flash_in_we_mem - 1), alt_u32(0xFFFFFFFF));

    // Ensure that BMC rules are untouched
    for (alt_u32 word_i = 0; word_i < U_SPI_FILTER_BMC_WE_AVMM_BRIDGE_SPAN / 4; word_i++)
    {
        EXPECT_EQ(IORD(U_SPI_FILTER_BMC_WE_AVMM_BRIDGE_BASE, word_i), alt_u32(0));
    }
}

/*
 * Test SPI filtering with some hand-made SPI region definitions.
 */
TEST_F(SPIFilterTest, test_spi_filtering_with_custom_pfm_case2)
{
    switch_spi_flash(SPI_FLASH_BMC);

    // Define a custom PFM for BMC
    //   SPI Region 1: 0x0000 - 0x10000 is RW
    //   SPI Region 2: 0x10000 - 0x7B000 is RO
    //   SPI Region 3: 0x7B000 - 0x8000000 is RW
    alt_u8 pfm_data_bin[256] = {
      // PFM header
      0x1d, 0xce, 0xb3, 0x02, 0x03, 0x01, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0x00, 0x01, 0x00, 0x00,
      // PFM body
      // SPI Region 1
      0x01, 0x03, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x01, 0x00,
      // SPI Region 2
      0x01, 0x01, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x01, 0x00,
      0x00, 0xB0, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      // Some SMBus rule
      0x02, 0xff, 0xff, 0xff,
      0xff, 0x01, 0x05, 0xa0, 0x2b, 0xe6, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      // SPI Region 3
      0x01, 0x03, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0xB0, 0x07, 0x00,
      0x00, 0x00, 0x00, 0x08,
      // Some SMBus rule
      0x02, 0xff, 0xff, 0xff, 0xff, 0x03, 0x05, 0xdc, 0x2b, 0xe6, 0x0a, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00,
      // Padding
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff,
    };

    // Save the PFM in the SPI flash
    alt_u32_memcpy((alt_u32*) get_active_pfm(SPI_FLASH_BMC), (alt_u32*) pfm_data_bin, 256);

    // Apply SPI write protection based on the PFM
    apply_spi_write_protection_and_smbus_rules(SPI_FLASH_BMC);

    // SPI regions 1/2/3
    EXPECT_EQ(IORD(U_SPI_FILTER_BMC_WE_AVMM_BRIDGE_BASE, 0), alt_u32(0xC000000F));
    // SPI region 3
    EXPECT_EQ(IORD(U_SPI_FILTER_BMC_WE_AVMM_BRIDGE_BASE, 1), alt_u32(0xFFFFFFFF));
    EXPECT_EQ(IORD(U_SPI_FILTER_BMC_WE_AVMM_BRIDGE_BASE, 2), alt_u32(0xFFFFFFFF));
    // Expected maximum size of BMC flash is BMC_SPI_FLASH_SIZE
    alt_u32 last_word_pos_for_bmc_flash_in_we_mem = BMC_SPI_FLASH_SIZE >> (14 + 5);
    EXPECT_EQ(IORD(U_SPI_FILTER_BMC_WE_AVMM_BRIDGE_BASE, last_word_pos_for_bmc_flash_in_we_mem - 2), alt_u32(0xFFFFFFFF));
    EXPECT_EQ(IORD(U_SPI_FILTER_BMC_WE_AVMM_BRIDGE_BASE, last_word_pos_for_bmc_flash_in_we_mem - 1), alt_u32(0xFFFFFFFF));

    // Ensure that PCH rules are untouched
    for (alt_u32 word_i = 0; word_i < U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_SPAN / 4; word_i++)
    {
        EXPECT_EQ(IORD(U_SPI_FILTER_PCH_WE_AVMM_BRIDGE_BASE, word_i), alt_u32(0));
    }
}
