#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"


class SPIFlashRWTest : public testing::Test
{
public:
    virtual void SetUp()
    {
        SYSTEM_MOCK::get()->reset();
        SYSTEM_MOCK::get()->load_to_flash(SPI_FLASH_BMC, SIGNED_CAPSULE_BMC_FILE, SIGNED_CAPSULE_BMC_FILE_SIZE);
        switch_spi_flash(SPI_FLASH_BMC);
    }

    virtual void TearDown() {}
};

TEST_F(SPIFlashRWTest, test_basic_read)
{
    alt_u32* bmc_flash_ptr = get_spi_flash_ptr();
    EXPECT_EQ(*bmc_flash_ptr, (alt_u32) BLOCK0_MAGIC);
}

TEST_F(SPIFlashRWTest, test_basic_write)
{
    alt_u32* bmc_flash_ptr = get_spi_flash_ptr();
    *bmc_flash_ptr = 0x12345678;
    EXPECT_EQ(*bmc_flash_ptr, (alt_u32) 0x12345678);
}

TEST_F(SPIFlashRWTest, test_erase_4kb_in_spi_flash)
{
    // Get pointer to the SPI flash
    alt_u32* bmc_flash_ptr = get_spi_flash_ptr();

    // Fill the first three pages with 0xdeadbeef and verify it
    for (alt_u32 i = 0; i < (0x3000 >> 2); i++)
    {
        bmc_flash_ptr[i] = 0xdeadbeef;
    }
    for (alt_u32 i = 0; i < (0x3000 >> 2); i++)
    {
        EXPECT_EQ(bmc_flash_ptr[i], alt_u32(0xdeadbeef));
    }

    // Erase the second page
    erase_4kb_in_spi_flash(0x1000);

    // The first page should remain untouched
    for (alt_u32 i = 0; i < (0x1000 >> 2); i++)
    {
        EXPECT_EQ(bmc_flash_ptr[i], alt_u32(0xdeadbeef));
    }
    // The second page should now be empty (filled with 0xffffffff)
    for (alt_u32 i = (0x1000 >> 2); i < (0x2000 >> 2); i++)
    {
        EXPECT_EQ(bmc_flash_ptr[i], alt_u32(0xffffffff));
    }
    // The third page should remain untouched
    for (alt_u32 i = (0x2000 >> 2); i < (0x3000 >> 2); i++)
    {
        EXPECT_EQ(bmc_flash_ptr[i], alt_u32(0xdeadbeef));
    }
}


TEST_F(SPIFlashRWTest, test_erase_spi_region)
{
    // Get pointer to the SPI flash
    switch_spi_flash(SPI_FLASH_PCH);
    alt_u32* pch_flash_ptr = get_spi_flash_ptr();
    pch_flash_ptr[0] = 0x12345678;
    pch_flash_ptr[1] = 0xdeadbeef;
    // 4KB boundary
    pch_flash_ptr[1023] = 0x33336666;
    pch_flash_ptr[1024] = BLOCK0_MAGIC;

    // Erase the first page
    erase_spi_region(0, 0x1000);

    // The first page should be cleared now
    for (alt_u32 i = 0; i < (0x1000 >> 2); i++)
    {
        EXPECT_EQ(pch_flash_ptr[i], 0xFFFFFFFF);
    }

    EXPECT_EQ(pch_flash_ptr[1024], alt_u32(BLOCK0_MAGIC));
}

TEST_F(SPIFlashRWTest, test_memcpy_signed_payload)
{
    // A valid capsule is available from the begining of the BMC flash
    alt_u32* bmc_flash_ptr = get_spi_flash_ptr();
    EXPECT_EQ(*bmc_flash_ptr, alt_u32(BLOCK0_MAGIC));

    // Copy this capsule to another SPI region
    bmc_flash_ptr[(0x2a00000 - 4) >> 2] = 0xdeadbeef;
    memcpy_signed_payload(0x2a00000, bmc_flash_ptr);

    // Ensure the whole capsule has been copied over.
    for (alt_u32 word_i = 0; word_i < (SIGNED_CAPSULE_BMC_FILE_SIZE >> 2); word_i++)
    {
        EXPECT_EQ(bmc_flash_ptr[word_i], bmc_flash_ptr[word_i + (0x2a00000 >> 2)]);
    }

    // Make sure the word before the target SPI region are not affected
    // The word after the region may be erased, if it's part of the page that capsule would occupy.
    EXPECT_EQ(bmc_flash_ptr[(0x2a00000 - 4) >> 2], 0xdeadbeef);
}
