// Include the GTest headers
#include "gtest_headers.h"

#include <math.h>
#include <chrono>
#include <thread>
#include <stdlib.h>

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"

class PFRTimerUtilsTest : public testing::Test
{
public:
    virtual void SetUp() { SYSTEM_MOCK::get()->reset(); }

    virtual void TearDown() {}
};

TEST_F(PFRTimerUtilsTest, test_sanity)
{
    EXPECT_EQ(U_TIMER_BANK_TIMER_ACTIVE_MASK, 1 << U_TIMER_BANK_TIMER_ACTIVE_BIT);
}

TEST_F(PFRTimerUtilsTest, test_check_timer_expired)
{
    alt_u32 count_ms = 5;
    start_timer_ms(U_TIMER_BANK_TIMER1_ADDR, count_ms);
    EXPECT_FALSE(is_timer_expired(U_TIMER_BANK_TIMER1_ADDR));

    // Sleep for 3 more ms. Timer should be expired.
    std::this_thread::sleep_for(std::chrono::milliseconds(5));
    EXPECT_TRUE(is_timer_expired(U_TIMER_BANK_TIMER1_ADDR));
}

TEST_F(PFRTimerUtilsTest, test_pause_timer)
{
    // Set timer to expire in 2 ms
    start_timer_ms(U_TIMER_BANK_TIMER2_ADDR, 2);
    pause_timer(U_TIMER_BANK_TIMER2_ADDR);

    // Sleep for 4 ms.
    std::this_thread::sleep_for(std::chrono::milliseconds(4));
    // Since we paused the timer, it should not have expired
    EXPECT_FALSE(is_timer_expired(U_TIMER_BANK_TIMER2_ADDR));

    resume_timer(U_TIMER_BANK_TIMER2_ADDR);
    // Sleep for 2 more ms.
    std::this_thread::sleep_for(std::chrono::milliseconds(2));
    // Since we resumed the timer, it should be expired by now
    EXPECT_TRUE(is_timer_expired(U_TIMER_BANK_TIMER2_ADDR));
}

TEST_F(PFRTimerUtilsTest, DISABLED_test_sleep_ms)
{
    auto clock_start = std::chrono::steady_clock::now();

    // Sleep for 10ms. Check
    sleep_ms(10);

    auto clock_end = std::chrono::steady_clock::now();
    auto duration_us = std::chrono::duration_cast<std::chrono::microseconds>(clock_end - clock_start);

    // Expect the time difference is within 15ms (10ms for the sleep, 5ms error).
    EXPECT_LE((alt_u32) duration_us.count(), alt_u32(15000));
}

TEST_F(PFRTimerUtilsTest, test_restart_timer)
{
    start_timer_ms(U_TIMER_BANK_TIMER3_ADDR, 3);

    std::this_thread::sleep_for(std::chrono::milliseconds(4));
    EXPECT_TRUE(is_timer_expired(U_TIMER_BANK_TIMER3_ADDR));

    // Restart the timer
    start_timer_ms(U_TIMER_BANK_TIMER3_ADDR, 6);
    // Timer should be active
    EXPECT_FALSE(is_timer_expired(U_TIMER_BANK_TIMER3_ADDR));
    EXPECT_GT(IORD(U_TIMER_BANK_TIMER3_ADDR, 0) & U_TIMER_BANK_TIMER_VALUE_MASK, (alt_u32) 0);
    EXPECT_TRUE(IORD(U_TIMER_BANK_TIMER3_ADDR, 0) & U_TIMER_BANK_TIMER_ACTIVE_MASK);
    EXPECT_TRUE(check_bit(U_TIMER_BANK_TIMER3_ADDR, U_TIMER_BANK_TIMER_ACTIVE_BIT));
}

/*
 * Check the accuracy of the timer.
 */
TEST_F(PFRTimerUtilsTest, DISABLED_test_timer_accuracy)
{
    alt_u32 countdown_ms = 60;

    start_timer_ms(U_TIMER_BANK_TIMER1_ADDR, countdown_ms);
    std::this_thread::sleep_for(std::chrono::milliseconds(50));

    // Allows the timer to be off by at most 5ms
    EXPECT_GE(IORD(U_TIMER_BANK_TIMER1_ADDR, 0) & U_TIMER_BANK_TIMER_VALUE_MASK, alt_u32(5));
    EXPECT_LE(IORD(U_TIMER_BANK_TIMER1_ADDR, 0) & U_TIMER_BANK_TIMER_VALUE_MASK, alt_u32(15));
}

/*
 * Check the accuracy of the timer.
 * Please examine the time this test took visually in the ./main output.
 */
TEST_F(PFRTimerUtilsTest, test_timer_accuracy_visual_check)
{
    start_timer_ms(U_TIMER_BANK_TIMER1_ADDR, 60);
    while(!is_timer_expired(U_TIMER_BANK_TIMER1_ADDR)) {}
}
