#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the PFR headers
// Always include the BSP mock then pfr_sys.h first
#include "bsp_mock.h"
#include "pfr_sys.h"

// Other PFR headers
#include "ufm.h"
#include "ufm_utils.h"

class UFMUtilsTest : public testing::Test
{
public:
    virtual void SetUp()
    {
        SYSTEM_MOCK::get()->reset();
        SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);
    }
    virtual void TearDown() {}
};

TEST_F(UFMUtilsTest, test_svn_read)
{
    get_ufm_pfr_data()->svn_policy_cpld[0] = 0xfffffffc;
    get_ufm_pfr_data()->svn_policy_cpld[1] = 0xffffffff;

    EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_CPLD), (alt_u32) 2);
}

TEST_F(UFMUtilsTest, test_svn_write)
{
    get_ufm_pfr_data()->svn_policy_pch[0] = 0xfffffffc;
    get_ufm_pfr_data()->svn_policy_pch[1] = 0xffffffff;

    write_ufm_svn(3, UFM_SVN_POLICY_PCH);

    EXPECT_EQ(get_ufm_pfr_data()->svn_policy_pch[0], (alt_u32) 0xfffffff8);
    EXPECT_EQ(get_ufm_pfr_data()->svn_policy_pch[1], (alt_u32) 0xffffffff);
}

TEST_F(UFMUtilsTest, test_svn_read_write)
{
    get_ufm_pfr_data()->svn_policy_bmc[0] = 0xffffffff;
    get_ufm_pfr_data()->svn_policy_bmc[1] = 0xffffffff;
    for (alt_u32 i = 0; i < 65; i++)
    {
        write_ufm_svn(i, UFM_SVN_POLICY_BMC);
        EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_BMC), (alt_u32) i);
    }
}

TEST_F(UFMUtilsTest, test_read_svn_policies)
{
    // Erase provisioning
    SYSTEM_MOCK::get()->erase_ufm_data(0);

    // Before provisioning UFM data, the SVN policies are the default values (i.e. 0)
    EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_CPLD), (alt_u32) 0);
    EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_PCH), (alt_u32) 0);
    EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_BMC), (alt_u32) 0);

    // Provision UFM
    SYSTEM_MOCK::get()->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);

    // Check expected SVN policies after provisioning
    EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_CPLD), (alt_u32) 0);
    EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_PCH), (alt_u32) 0);
    EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_BMC), (alt_u32) 0);
}

TEST_F(UFMUtilsTest, test_max_svn_policies)
{
    get_ufm_pfr_data()->svn_policy_bmc[0] = 0;
    get_ufm_pfr_data()->svn_policy_bmc[1] = 0;

    // Check expected SVN policies after provisioning
    EXPECT_EQ(get_ufm_svn(UFM_SVN_POLICY_BMC), (alt_u32) UFM_MAX_SVN);
}

TEST_F(UFMUtilsTest, test_is_svn_valid)
{
    write_ufm_svn(3, UFM_SVN_POLICY_PCH);
    write_ufm_svn(9, UFM_SVN_POLICY_BMC);

    EXPECT_FALSE(is_svn_valid(UFM_SVN_POLICY_PCH, 1));
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_PCH, 3));
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_PCH, 32));

    EXPECT_FALSE(is_svn_valid(UFM_SVN_POLICY_BMC, 7));
    EXPECT_TRUE(is_svn_valid(UFM_SVN_POLICY_BMC, 9));
    EXPECT_FALSE(is_svn_valid(UFM_SVN_POLICY_BMC, 128));
}
