#include <iostream>

// Include the GTest headers
#include "gtest_headers.h"

// Include the SYSTEM MOCK and PFR headers
#include "ut_nios_wrapper.h"
#include "testdata_info.h"


class WDTRecoveryFlowTest : public testing::Test
{
public:
    virtual void SetUp()
    {
        SYSTEM_MOCK* sys = SYSTEM_MOCK::get();
        // Reset system mocks and SPI flash
        sys->reset();
        sys->reset_spi_flash_mock();

        // Perform provisioning
        sys->provision_ufm_data(UFM_PFR_DATA_EXAMPLE_KEY_FILE);

        // Load the entire image to flash
        sys->load_to_flash(SPI_FLASH_BMC, FULL_PFR_IMAGE_BMC_FILE, FULL_PFR_IMAGE_BMC_FILE_SIZE);
        sys->load_to_flash(SPI_FLASH_PCH, FULL_PFR_IMAGE_PCH_FILE, FULL_PFR_IMAGE_PCH_FILE_SIZE);

        // Reset Nios firmware
        ut_reset_nios_fw();
    }

    virtual void TearDown() {}
};

TEST_F(WDTRecoveryFlowTest, test_bmc_wdt_timeout_recovery)
{
    // Flow preparation
    ut_set_gpi_nios_start_and_me_boot_done();
    while (!check_ready_for_nios_start());

    // BMC timer should be enabled
    EXPECT_TRUE(wdt_enable_status & WDT_ENABLE_BMC_TIMER_MASK);

    /*
     * First timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_BMC_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    // Expect it to catch the timeout and perform recovery
    // After getting platform back to T0, this function should be done
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PLATFORM_STATE), PLATFORM_STATE_ENTER_T0);
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_BMC_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_BMC_LAUNCH_FAIL);

    // Don't expect any error for now
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(0));

    // Recovery level should now be at 2
    EXPECT_EQ(get_fw_recovery_level(SPI_FLASH_BMC), alt_u32(SPI_REGION_PROTECT_MASK_RECOVER_ON_SECOND_RECOVERY));

    /*
     * Second timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_BMC_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(2));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_BMC_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(2));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_BMC_LAUNCH_FAIL);

    // Recovery level should now be at 3
    EXPECT_EQ(get_fw_recovery_level(SPI_FLASH_BMC), alt_u32(SPI_REGION_PROTECT_MASK_RECOVER_ON_THIRD_RECOVERY));

    /*
     * Third timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_BMC_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_BMC_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_BMC_LAUNCH_FAIL);

    // Recovery level should now exceed the maximum level (3)
    EXPECT_FALSE(get_fw_recovery_level(SPI_FLASH_BMC) & SPI_REGION_PROTECT_MASK_RECOVER_BITS);

    /*
     * Fourth timeout
     * Nios should just disable the BMC timer and let BMC hang
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_BMC_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // There should not be any more panic/recovery event
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(3));

    // Recovery level should now exceed the maximum level (3)
    EXPECT_FALSE(get_fw_recovery_level(SPI_FLASH_BMC) & SPI_REGION_PROTECT_MASK_RECOVER_BITS);

    // BMC timer should be disabled
    EXPECT_FALSE(wdt_enable_status & WDT_ENABLE_BMC_TIMER_MASK);
}

TEST_F(WDTRecoveryFlowTest, test_acm_wdt_timeout_recovery)
{
    // Flow preparation
    ut_set_gpi_nios_start_and_me_boot_done();
    while (!check_ready_for_nios_start());

    // BMC timer should be enabled
    EXPECT_TRUE(wdt_enable_status & WDT_ENABLE_ACM_BIOS_TIMER_MASK);

    /*
     * First timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    // Expect it to catch the timeout and perform recovery
    // After getting platform back to T0, this function should be done
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PLATFORM_STATE), PLATFORM_STATE_ENTER_T0);
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_ACM_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_ACM_LAUNCH_FAIL);

    // Don't expect any error for now
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(0));

    // Recovery level should now be at 2
    EXPECT_EQ(get_fw_recovery_level(SPI_FLASH_PCH), alt_u32(SPI_REGION_PROTECT_MASK_RECOVER_ON_SECOND_RECOVERY));

    /*
     * Second timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(2));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_ACM_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(2));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_ACM_LAUNCH_FAIL);

    // Recovery level should now be at 3
    EXPECT_EQ(get_fw_recovery_level(SPI_FLASH_PCH), alt_u32(SPI_REGION_PROTECT_MASK_RECOVER_ON_THIRD_RECOVERY));

    /*
     * Third timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_ACM_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_ACM_LAUNCH_FAIL);

    // Recovery level should now exceed the maximum level (3)
    EXPECT_FALSE(get_fw_recovery_level(SPI_FLASH_PCH) & SPI_REGION_PROTECT_MASK_RECOVER_BITS);

    /*
     * Fourth timeout
     * Nios should just disable the ACM/BIOS timer and let PCH hang
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_ACM_BIOS_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // There should not be any more panic/recovery event
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(3));

    // Recovery level should now exceed the maximum level (3)
    EXPECT_FALSE(get_fw_recovery_level(SPI_FLASH_PCH) & SPI_REGION_PROTECT_MASK_RECOVER_BITS);

    // BMC timer should be disabled
    EXPECT_FALSE(wdt_enable_status & WDT_ENABLE_ACM_BIOS_TIMER_MASK);
}

TEST_F(WDTRecoveryFlowTest, test_me_wdt_timeout_recovery)
{
    // Flow preparation
    ut_set_gpi_nios_start_and_me_boot_done();
    while (!check_ready_for_nios_start());

    // BMC timer should be enabled
    EXPECT_TRUE(wdt_enable_status & WDT_ENABLE_ME_TIMER_MASK);

    /*
     * First timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_ME_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    // Expect it to catch the timeout and perform recovery
    // After getting platform back to T0, this function should be done
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PLATFORM_STATE), PLATFORM_STATE_ENTER_T0);
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_ME_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(1));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_ME_LAUNCH_FAIL);

    // Don't expect any error for now
    EXPECT_EQ(read_from_mailbox(MB_MAJOR_ERROR_CODE), alt_u32(0));
    EXPECT_EQ(read_from_mailbox(MB_MINOR_ERROR_CODE), alt_u32(0));

    // Recovery level should now be at 2
    EXPECT_EQ(get_fw_recovery_level(SPI_FLASH_PCH), alt_u32(SPI_REGION_PROTECT_MASK_RECOVER_ON_SECOND_RECOVERY));

    /*
     * Second timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_ME_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(2));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_ME_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(2));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_ME_LAUNCH_FAIL);

    // Recovery level should now be at 3
    EXPECT_EQ(get_fw_recovery_level(SPI_FLASH_PCH), alt_u32(SPI_REGION_PROTECT_MASK_RECOVER_ON_THIRD_RECOVERY));

    /*
     * Third timeout
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_ME_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // Check the platform state and other status
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_LAST_PANIC_REASON), LAST_PANIC_ME_WDT_EXPIRED);
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_LAST_RECOVERY_REASON), LAST_RECOVERY_ME_LAUNCH_FAIL);

    // Recovery level should now exceed the maximum level (3)
    EXPECT_FALSE(get_fw_recovery_level(SPI_FLASH_PCH) & SPI_REGION_PROTECT_MASK_RECOVER_BITS);

    /*
     * Fourth timeout
     * Nios should just disable the ACM/BIOS timer and let PCH hang
     */
    // Start the timer with 0ms on the countdown, i.e. it's expired already
    start_timer_ms(WDT_ME_TIMER_ADDR, 0);
    // Run the T0 watchdog timers' handlers
    watchdog_routine();

    // There should not be any more panic/recovery event
    EXPECT_EQ(read_from_mailbox(MB_PANIC_EVENT_COUNT), alt_u32(3));
    EXPECT_EQ(read_from_mailbox(MB_RECOVERY_COUNT), alt_u32(3));

    // Recovery level should now exceed the maximum level (3)
    EXPECT_FALSE(get_fw_recovery_level(SPI_FLASH_PCH) & SPI_REGION_PROTECT_MASK_RECOVER_BITS);

    // BMC timer should be disabled
    EXPECT_FALSE(wdt_enable_status & WDT_ENABLE_ME_TIMER_MASK);
}
