// (C) 2019 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.

module Mem_Seq
(
	input			iClk,//%Clock input
	input			iRst_n,//%Reset enable on high
	input			i1mSCE, //% 1 mS Clock Enable
	input			iMemPwrEn,//% Memory power secuence enable
	input			iGoOutFltSt,//% Go out fault state.
	input			iEnableTimeOut, //% Enable the Timeout
	input			PWRGD_P3V3,
	input			PWRGD_VDDQ_ABCD,//% Memory VR PWRGD VDDQ
	input			PWRGD_VDDQ_EFGH,//% Memory VR PWRGD VDDQ
	input			FM_SLPS4_N,	//% SLPS4 to support S3


	output reg		FM_MEM_VPP_EN,//% Memory VR VPP Enable 
	output reg		oMemPwrgd,//% PWRGD of all Memory VR's
	output reg		oMemPwrFltVDDQ_ABCD,//% Fault VDDQ
	output reg		oMemPwrFltVDDQ_EFHG,//% Fault VDDQ
	output reg		oMemPwrFlt,//% Fault Memory VR's
	output    [2:0]	oDbgMemSt//% State of Memory Secuence.

);

//////////////////////////////////////////////////////////////////////////////////
// Parameters
//////////////////////////////////////////////////////////////////////////////////
parameter  ST_MFAULT        = 3'd1;
parameter  ST_MOFF          = 3'd2;
parameter  ST_VDD           = 3'd3;
parameter  ST_MDONE         = 3'd4;
parameter  ST_VDD_OFF       = 3'd5;

parameter  LOW =1'b0;
parameter  HIGH=1'b1;
parameter  T_500US_2M  =  10'd1000;

//////////////////////////////////////////////////////////////////////////////////
// Internal Signals
//////////////////////////////////////////////////////////////////////////////////
reg [2:0]   rCurrSt;
reg [2:0]   rCurrSt_ff1;
reg [2:0]   rNextSt;

reg rMemFlt;
reg rPwrgd_Vddq_ABCD_Ff1;
reg rPWRGD_VDDQ_EFGH_Ff1;
wire wCntRst_n,wTimeOut_500us;
wire wPwrgd_VDDQ_dly_up2_5ms,wDly_500us, wPWRGD_P3V3_2ms;

//////////////////////////////////////////////////////////////////////////////////
// Continuous assignments
//////////////////////////////////////////////////////////////////////////////////
assign  oDbgMemSt = rCurrSt;
assign  wCntRst_n = (rCurrSt == rCurrSt_ff1);
assign  wTimeOut_500us = wDly_500us & wCntRst_n & iEnableTimeOut;

//////////////////////////////////////////////////////////////////////////////////
// FSM CurrentState generation
//////////////////////////////////////////////////////////////////////////////////
			
always @ ( posedge iClk) 
begin 
	if (  !iRst_n  )   
		begin
			rCurrSt      <= ST_MOFF;
			rCurrSt_ff1  <= ST_MOFF;			
		end 
	else 
		begin
            rCurrSt      <= rNextSt;
			rCurrSt_ff1  <= rCurrSt; 								
		end
end  

//////////////////////////////////////////////////////////////////////////////////
// State Machine logic
//////////////////////////////////////////////////////////////////////////////////
always @( * )
begin
	case( rCurrSt )			
		ST_MFAULT: //1 
			if( iGoOutFltSt )								rNextSt = ST_MOFF;
			else											rNextSt = ST_MFAULT;          

		ST_MOFF:  //2
			if     ( rMemFlt )								rNextSt = ST_MFAULT;
			else if(  iMemPwrEn )							rNextSt = ST_VDD;   
		else												rNextSt = ST_MOFF;				

		ST_VDD: //3
			if( !iMemPwrEn )								rNextSt = ST_VDD_OFF;      
			else if(  wPwrgd_VDDQ_dly_up2_5ms)				rNextSt = ST_MDONE;                        
			else											rNextSt = ST_VDD;
			
		ST_MDONE: //4 
			if( !iMemPwrEn || rMemFlt)						rNextSt = ST_VDD_OFF;
			else											rNextSt = ST_MDONE;	
			
		ST_VDD_OFF: //5
			if(~wPwrgd_VDDQ_dly_up2_5ms  ||
					wTimeOut_500us)							rNextSt = ST_MOFF;					
			else											rNextSt = ST_VDD_OFF;
				
		default:											rNextSt = ST_MOFF;  			
	endcase  
end

//////////////////////////////////////////////////////////////////////////////////
// Output State Machine 
//////////////////////////////////////////////////////////////////////////////////
always @( posedge iClk) 
begin 
	if (  !iRst_n ) 
	begin		
		FM_MEM_VPP_EN        <= LOW;

		oMemPwrgd            <= LOW;
		oMemPwrFlt	    	 <= LOW;
		rMemFlt			     <= LOW;

		rPwrgd_Vddq_ABCD_Ff1 <= LOW;
		rPWRGD_VDDQ_EFGH_Ff1 <= LOW;

		oMemPwrFltVDDQ_ABCD  <= LOW;
		oMemPwrFltVDDQ_EFHG  <= LOW;
	end
	else 
	begin
		FM_MEM_VPP_EN        <= ((rCurrSt >= ST_VDD) && (rCurrSt < ST_VDD_OFF) && (wPWRGD_P3V3_2ms || FM_SLPS4_N)) ? HIGH : LOW;
		
		oMemPwrgd            <= ((rCurrSt >= ST_MDONE) && (rCurrSt <= ST_VDD_OFF)) ? HIGH : LOW;	

		oMemPwrFlt           <= (rCurrSt == ST_MFAULT);

		rPwrgd_Vddq_ABCD_Ff1 <= PWRGD_VDDQ_ABCD;
		rPWRGD_VDDQ_EFGH_Ff1 <= PWRGD_VDDQ_EFGH;

		oMemPwrFltVDDQ_ABCD  <= (FM_MEM_VPP_EN  && !PWRGD_VDDQ_ABCD && rPwrgd_Vddq_ABCD_Ff1) ? HIGH: oMemPwrFltVDDQ_ABCD;
		oMemPwrFltVDDQ_EFHG	 <= (FM_MEM_VPP_EN  && !PWRGD_VDDQ_EFGH && rPWRGD_VDDQ_EFGH_Ff1) ? HIGH: oMemPwrFltVDDQ_EFHG;
		rMemFlt              <= (iMemPwrEn && (oMemPwrFltVDDQ_ABCD || oMemPwrFltVDDQ_EFHG)) ? HIGH: rMemFlt;
	end
end 
		



//////////////////////////////////////////////////////////////////////
// Instances
//////////////////////////////////////////////////////////////////////
//% 2.5ms Delay for power UP  VDDQ
//
SignalValidationDelay#
(
    .VALUE                  ( 1'b1 ),
    .TOTAL_BITS             ( 2'd3 ),
    .POL                    ( 1'b1 )
)mVDD_DLY         
(           
    .iClk                   ( iClk ),
    .iRst                   ( ~iRst_n ),
    .iCE                    ( i1mSCE ),
    .ivMaxCnt               ( 3'd3 ),        //3ms delay
    .iStart                 ( PWRGD_VDDQ_ABCD && PWRGD_VDDQ_EFGH),
    .oDone                  ( wPwrgd_VDDQ_dly_up2_5ms )
);

SignalValidationDelay#
(
    .VALUE                  ( 1'b1 ),
    .TOTAL_BITS             ( 2'd3 ),
    .POL                    ( 1'b1 )
)mP3V3_DLY         
(           
    .iClk                   ( iClk ),
    .iRst                   ( ~iRst_n ),
    .iCE                    ( i1mSCE ),
    .ivMaxCnt               ( 3'd2 ),        //2ms delay
    .iStart                 ( PWRGD_P3V3 ),
    .oDone                  ( wPWRGD_P3V3_2ms )
);

counter2 #( .MAX_COUNT(T_500US_2M) ) ForceOff_500usDly  
( .iClk         	( iClk           ), 
.iRst_n       	( iRst_n         ),

.iCntRst_n  	(  wCntRst_n  ),
.iCntEn      	(  wCntRst_n  ),
.iLoad        	(!(wCntRst_n) ),
.iSetCnt    	( T_500US_2M  [9: 0]   ),  // set the time out time as 500us

.oDone        	( wDly_500us   ),
.oCntr         	(             )
);

endmodule
