/** @file
  This file is PeiCpuPolicy library.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 2014 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/
#include "PeiCpuPolicyLibrary.h"
#include <Library/PostCodeLib.h>

VOID
LoadCpuPreMemDefaultPreMem(
  IN   VOID         *ConfigBlockPointer
  )
{
#ifdef SGX_SUPPORT
  CPU_SECURITY_PREMEM_CONFIG    *CpuSecurityPreMemConfig;

  CpuSecurityPreMemConfig = ConfigBlockPointer;

  CpuSecurityPreMemConfig->EnableSgx =  CPU_FEATURE_DISABLE;
  CpuSecurityPreMemConfig->PrmrrSize = 0;
#endif

}

VOID
LoadCpuPreMemConfigDefault(
  IN   VOID         *ConfigBlockPointer
  )
{

  CPU_PREMEM_CONFIG    *CpuSecurityPreMemConfig;

  CpuSecurityPreMemConfig = ConfigBlockPointer;
  CpuSecurityPreMemConfig->ProcessorTraceMemSize = 0xFF;

}

static IP_BLOCK_ENTRY  mGlkCpuIpBlocksPreMem [] = {
  {&gCpuSecurityPreMemConfigGuid, sizeof (CPU_SECURITY_PREMEM_CONFIG), CPU_SECURITY_CONFIG_PREMEM_REVISION,  LoadCpuPreMemDefaultPreMem},
  {&gCpuPreMemConfigGuid,         sizeof (CPU_PREMEM_CONFIG),          CPU_PREMEM_POLICY_REVISION,           LoadCpuPreMemConfigDefault}
};

/**
  Get CPU PREMEM config block table total size.

  @retval  Size of SC PREMEM config block table
**/
UINT32
EFIAPI
CpuGetPreMemConfigBlockTotalSize (
  VOID
  )
{
  UINT32            TotalBlockCount;
  UINT32            TotalBlockSize;
  UINT32            ConfigBlockHdrSize;
  UINT32            BlockCount;

  TotalBlockCount = sizeof(mGlkCpuIpBlocksPreMem) / sizeof (IP_BLOCK_ENTRY);
  TotalBlockSize = 0;
  for (BlockCount = 0 ; BlockCount < TotalBlockCount; BlockCount++) {
    TotalBlockSize += (UINT32)mGlkCpuIpBlocksPreMem[BlockCount].Size;
    DEBUG ((DEBUG_INFO, "TotalBlockSize after adding Block[0x%x]= 0x%x\n", BlockCount, TotalBlockSize));
  }

  ConfigBlockHdrSize = sizeof (CONFIG_BLOCK_TABLE_HEADER) + (TotalBlockCount * 4);

  //
  // Because CreateConfigBlockTable has the padding for each config block,
  // we need extra size, which is TotalBlockCount * 3, to create the table
  //
  return ConfigBlockHdrSize + TotalBlockSize + (TotalBlockCount * 3);
}

/**
  CreateCpuConfigBlocks creates the default setting of PEI Cpu Policy.
  It allocates and zero out buffer, and fills in the Intel default settings.

  @param[out] SiCpuPolicyPpi            The pointer to get PEI Cpu Policy PPI instance

  @retval EFI_SUCCESS                   The policy default is initialized.
  @retval EFI_OUT_OF_RESOURCES          Insufficient resources to create buffer
**/
EFI_STATUS
EFIAPI
CreateCpuConfigBlocksPreMem (
  OUT  SI_CPU_POLICY_PPI      **SiCpuPolicyPpi
  )
{
  UINT16                    TotalBlockCount;
  UINT16                    BlockCount;
  VOID                      *ConfigBlockPointer;
  CONFIG_BLOCK              ConfigBlockBuf;
  EFI_STATUS                Status;
  SI_CPU_POLICY_PPI         *SiCpuPolicy;
  UINT32                    RequiredSize;

  DEBUG ((DEBUG_INFO, "CPU PreMem CreateConfigBlocks\n"));

  SiCpuPolicy = NULL;
  TotalBlockCount = sizeof(mGlkCpuIpBlocksPreMem) / sizeof (IP_BLOCK_ENTRY);
  DEBUG ((DEBUG_INFO, "TotalBlockCount = 0x%x\n", TotalBlockCount));

  RequiredSize = CpuGetPreMemConfigBlockTotalSize ();

  Status = CreateConfigBlockTable ((UINT16)RequiredSize, (VOID *)&SiCpuPolicy);
  ASSERT_EFI_ERROR (Status);

  //
  // Initialize Policy Revision
  //
  SiCpuPolicy->ConfigBlockTableHeader.Header.Revision = SI_CPU_POLICY_PPI_REVISION;
  //
  // Initialize ConfigBlockPointer to NULL
  //
  ConfigBlockPointer = NULL;
  //
  // Loop to identify each config block from mGlkCpuIpBlocksPreMem[] Table and add each of them
  //
  for (BlockCount = 0 ; BlockCount < TotalBlockCount; BlockCount++) {
    CopyMem (&(ConfigBlockBuf.Header.GuidHob.Name),  mGlkCpuIpBlocksPreMem[BlockCount].Guid, sizeof (EFI_GUID));
    ConfigBlockBuf.Header.GuidHob.Header.HobLength = mGlkCpuIpBlocksPreMem[BlockCount].Size;
    ConfigBlockBuf.Header.Revision                 = mGlkCpuIpBlocksPreMem[BlockCount].Revision;
    ConfigBlockPointer                             = (VOID *)&ConfigBlockBuf;
    Status = AddConfigBlock ((VOID *)SiCpuPolicy, (VOID *)&ConfigBlockPointer);
    ASSERT_EFI_ERROR (Status);
    mGlkCpuIpBlocksPreMem[BlockCount].LoadDefault (ConfigBlockPointer);
  }

  //
  // Assignment for returning CpuInitPolicy config block base address
  //

  *SiCpuPolicyPpi = SiCpuPolicy;
  PostCode (0xC2F);
  return EFI_SUCCESS;
}

/**
  CpuInstallPolicyPpi installs SiCpuPolicyPpi.
  While installed, RC assumes the Policy is ready and finalized. So please update and override
  any setting before calling this function.

  @param[in] SiCpuPolicyPpi             The pointer to PEI Cpu Policy PPI instance

  @retval EFI_SUCCESS                   The policy is installed.
  @retval EFI_OUT_OF_RESOURCES          Insufficient resources to create buffer
**/
EFI_STATUS
EFIAPI
CpuInstallPolicyPpiPreMem (
  IN  SI_CPU_POLICY_PPI   *SiCpuPolicy
  )
{
  EFI_STATUS                  Status;
  EFI_PEI_PPI_DESCRIPTOR      *SiCpuPolicyPpiDesc;

  SiCpuPolicyPpiDesc = (EFI_PEI_PPI_DESCRIPTOR *) AllocateZeroPool (sizeof (EFI_PEI_PPI_DESCRIPTOR));
  if (SiCpuPolicyPpiDesc == NULL) {
    ASSERT (SiCpuPolicyPpiDesc != NULL);
    return EFI_OUT_OF_RESOURCES;
  }

  SiCpuPolicyPpiDesc->Flags = EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST;
  SiCpuPolicyPpiDesc->Guid  = &gSiCpuPolicyPpiGuid;
  SiCpuPolicyPpiDesc->Ppi   = SiCpuPolicy;

  //
  // Print whole SI_CPU_POLICY_PPI and serial out in PreMem.
  //
  CpuPrintPolicyPreMemPpi (SiCpuPolicy);

  //
  // Install PEI Cpu Policy PPI
  //
  Status = PeiServicesInstallPpi (SiCpuPolicyPpiDesc);
  ASSERT_EFI_ERROR (Status);
  return Status;
}
