/** @file
  Memory Configuration.

@copyright
 Copyright (c) 2005 - 2016 Intel Corporation. All rights reserved
 This software and associated documentation (if any) is furnished
 under a license and may only be used or copied in accordance
 with the terms of the license. Except as permitted by the
 license, no part of this software or documentation may be
 reproduced, stored in a retrieval system, or transmitted in any
 form or by any means without the express written consent of
 Intel Corporation.
 This file contains an 'Intel Peripheral Driver' and is uniquely
 identified as "Intel Reference Module" and is licensed for Intel
 CPUs and chipsets under the terms of your license agreement with
 Intel or your vendor. This file may be modified by the user, subject
 to additional terms of the license agreement.

@par Specification Reference:
**/

#include "ConfigMem.h"
#include "ConfigMemData.h"
#include "IchRegs.h"

#include "OemHooks.h"
#include "McFunc.h"
#include "DetectDimms.h"
#include "Mailbox.h"
#include "MmrcHooks.h"
#include "Mailbox.h"
#include "MmrcLibraries.h"
#include "MmrcProjectData.h"
#include "MrcFunc.h"
#if EFI_MEMORY_INIT
#include <Library/SteppingLib.h>
#endif


/**
  Initialize Punit setting

  @param[in,out]  MrcData

  @retval  None
**/
MMRC_STATUS
InitPunitSetting (
  IN  OUT   MMRC_DATA   *MrcData
)
{
  return MMRC_SUCCESS;
}

#ifdef __GNUC__
#pragma GCC push_options
#pragma GCC optimize("O0")
#endif
/**
  LockBarAddress

  @param[in]  MrcData

  @retval  None
**/
VOID LockBarAddress
  (MMRC_DATA         *MrcData )
{
#ifndef SIM
    MmrcExecuteTask (MrcData, LockAunitSliceChannel, NO_PRINT, 0);
#if defined __GNUC__  // GCC compiler
  __asm__ __volatile__ (
  "\n\t pushl  %eax"
  "\n\t pushl  %edx"
  "\n\t pushl  %ecx"
  "\n\t movl   $0x00, %eax"
  "\n\t movl   $0x00, %edx"
  "\n\t movl   $0x122, %ecx"
  "\n\t rdmsr"
  "\n\t orl    $0x80000000, %edx"
  "\n\t wrmsr"
  "\n\t popl   %ecx"
  "\n\t popl   %edx"
  "\n\t popl   %eax"
  );
#else //MSFT compiler
  _asm {
    push eax
    push edx
    push ecx
    mov eax, 00h
    mov edx, 00h
    mov ecx, 0122h
    rdmsr
    or  edx, 80000000h
    wrmsr
    pop ecx
    pop edx
    pop eax
  }
#endif
#endif //SIM
}
#ifdef __GNUC__
#pragma GCC pop_options
#endif

/**
  Returns TRUE if the bootmode if not a restore path and the MRC fast boot data
  has been loaded.

  @param[in]  MrcData  Host structure for all MRC global data.

  @retval  TRUE
  @retval  FALSE
**/
BOOLEAN
MrcDataStructDirty(
  IN  MMRC_DATA   *MrcData
)
{
  //
  // MrcData->MrcParamsValidFlag will be set to 1
  // if the MRC NV data was previously loaded
  //
  if (MrcData->MrcParamsValidFlag) {
    if (!InResorePath(MrcData)){
    //
    // If the MRC NV data was previously loaded and the boot mode is not a restore path
    // then return TRUE since the structure is "dirty" with data from the previous boot.
    //
      return TRUE;
    }
  }
  return FALSE;
}

/**
  Configure the memory thru several training steps

  @param[in,out]  MrcData  Host structure for all MRC global data.

  @retval  Success
  @retval  Failure
**/
MMRC_STATUS
ConfigureMemory (
  IN  OUT   MMRC_DATA   *MrcData
)
{
  MMRC_STATUS        Status;
  UINT32             TempRegData;

#if MINIBIOS
  UINT32             Address;
  UINT32             Data;
  UINT32             i;
#endif
  TempRegData = 0;


  Status = MMRC_SUCCESS;

  ///
  /// Some routines require the MrcData host structure but we don't pass the structure
  /// to them for several reasons:
  ///   1) To save code space - 1 less parameter per function call on routines which
  ///      are frequently called.
  ///   2) Macros such as MrcDeadLoop() are able to be used from any routine. If a
  ///      routine does not have MrcData as a parameter, it can't pass it to MrcDeadLoop().
  ///
  SaveMrcHostStructureAddress (MrcData);


  ///
  /// Initialize other components such as UART, PUNIT, HPET, etc.
  ///
  ExternalSiliconInit (MrcData);

  ///
  ///
  /// Populate all input parameters that MMRC needs to execute.
  ///
  GetInputParameters (MrcData);
  ///
  /// Convert input parameters to actual values which will be written
  /// into registers. For example, if TCL of 11 needs a value of 6
  /// written to the DUNIT, this routine will change TCL to 6.
  ///
  //ConvertInputParameters (MrcData);

  PrintMemoryConfig (MrcData);

  //
  // FALSE = Start memory initialization
  //
#if !SIM && !JTAG
  RecordMrcStatus(MrcData, FALSE);
#endif

  ///
  /// Single entry point to MMRC to execute the entire flow.
  ///
  if (MrcDataStructDirty(MrcData)) {
    //
    // If this is an S5 boot and the MRC Data structure is loaded with data from a
    // previous boot, we need to re-initialize the structure to make sure there's
    // no stale data being passed in to the MMRC. This is a fallback in case we've
    // already loaded the FB data but need to do perform an S5 boot.
    // Situations where this path is excercised:
    // - SODIMM configuration is changed from first boot
    // - Previous boot failed and DramInit flag was not cleared
    //
    return MMRC_DATA_DIRTY;
  }
  Status = MmrcEntry (MrcData, EntireMmrc, NULL);

  if (Status != MMRC_SUCCESS) {
    return Status;
  }
  FillOutputStructure (MrcData);
  //
  // If we're in the MiniBIOS, MRC has completed. Now need to deadloop and optionally
  // display all delays and dump all registers.
  //

  //
  // Disables the High Precision Event Timer
  //
  McEnableHpet (MrcData);

#ifdef BXTPMRC
  SetSliceChannelHash (MrcData);
  LockBarAddress (MrcData);
#endif

#if !SIM && !JTAG
  RecordMrcStatus(MrcData, TRUE);
#endif

  return Status;
}

/**
  Mark MRC as started so that if a reset occurs during the middle of MRC,
  we know the memory subsystem is partially initialized. In this case, we
  need to force a powergood reset to get the DDRIO back into a reset state
  where the registers are at default values.

  @param[in,out]  MrcData      Host structure for all MRC global data.
  @param[in]      Finish

  @retval  Success
  @retval  Failure
**/
VOID
RecordMrcStatus (
  IN  OUT   MMRC_DATA   *MrcData,
  IN  BOOLEAN           Finish
  )
{
  UINT32 PmcMrcStartedIndication ;
  //
  // Read PMC GCR PMCON1 register and clear dram init bit when complete. Set dram init bit to indicate MRC is not complete/in progress.
  //
  PmcMrcStartedIndication  = Mmio32Read(MrcData->PmcBar + PMC_GCR_BASE_ADDRESS + PMC_GCR_GEN_PMCON1);
  if (Finish) {
    Mmio32Write (MrcData->PmcBar + PMC_GCR_BASE_ADDRESS + PMC_GCR_GEN_PMCON1, (PmcMrcStartedIndication  & ~PMC_GCR_GEN_PMCON1_DRAM_INIT_BIT_STS));
  } else {
    Mmio32Write (MrcData->PmcBar + PMC_GCR_BASE_ADDRESS + PMC_GCR_GEN_PMCON1, (PmcMrcStartedIndication  | PMC_GCR_GEN_PMCON1_DRAM_INIT_BIT_STS));
  }
}
/**
  Routine to initialize other pieces of silicon which may be required
  for MRC to successfully initialize memory. High precision event timer,
  PUNIT mailbox, etc.

  @param[in,out]  MrcData       Host structure for all MRC global data.

  @retval  Success
  @retval  Failure
**/
MMRC_STATUS
ExternalSiliconInit (
  IN  OUT   MMRC_DATA   *MrcData
)
{
#ifndef SIM
#ifdef MINIBIOS
#if DEBUG_MSG
 // Init16550Uart ();
#endif
#endif
  ///
  /// Get platform setup including all BAR address
  ///
  if (GetPlatformSettings (MrcData) == MMRC_FAILURE) {
    MrcDeadLoop ();
  }
  ///
  /// Enables the High Precision Event Timer
  ///
  McEnableHpet (MrcData);
  ///
  /// Initialize Punit Setting
  ///
  InitPunitSetting(MrcData);
#endif // SIM

  return MMRC_SUCCESS;
}

//
// DramDensity[]
// Used to convert the index value of MrcData->Channel[Channel].D_Size[0] into
// a decimal value in units of gigabits (Gb)
//
UINT8 DramDensity[MaxMemoryDeviceDensity] = {
  DRAM_DENSITY_4Gb,  // 0 = 4Gb
  DRAM_DENSITY_6Gb,  // 1 = 6Gb
  DRAM_DENSITY_8Gb,  // 2 = 8Gb
  DRAM_DENSITY_12Gb, // 3 = 12Gb
  DRAM_DENSITY_16Gb  // 4 = 16Gb
};

/**
  Calculates and returns the rank size in megabytes for a given channel

  @param[in]  MrcData       Host structure for all data related to MMRC
  @param[in]  Channel       Target channel for rank size calculation

  @retval  RankSize         Rank size in MB for a given channel
**/
UINT16 CalculateRankSize(
  IN  MMRC_DATA *MrcData,
  IN  UINT8     Channel
  )
{
  UINT8  DQBusWidth;
  // 0 = x8; 1 = x16; 2 = x32
  UINT8  DeviceWidth;
  //  0000 =  4 Gb 0001 = 6 Gb  0010 = 8 Gb  0011 = 12 Gb
  //  0100 = 16 Gb
  UINT16 DramCapacity;
  UINT16 RankSize;
  UINT8  DramDensityInGb;
  BXT_SERIES BxtSeries;

  BxtSeries = MrcData->BxtSeries;

  DQBusWidth   = 8;
  DeviceWidth  = MrcData->Channel[Channel].D_DataWidth[0];
  DramCapacity = MrcData->Channel[Channel].D_Size[0];

  if (MrcData->Channel[Channel].DramType == TypeLpDdr3) {
    DQBusWidth = 32;
    //
    // The LPDDR3 Density is reported per rank. The total device width
    // is always x32 in this case
    //
    DeviceWidth = 2;
  } else if (MrcData->Channel[Channel].DramType == TypeLpDdr4) {
    if (BxtSeries == Bxt1 ) {
      DQBusWidth = 16;
    } else {
      DQBusWidth = 32; //  32bit DQ bus (DRAMdevice_PerRank * device width)
    }
    //
    // The LPDDR4 Density is reported per rank. The total device width
    // is always x32 in this case
    //
    DeviceWidth = 2;
  } else {
    DQBusWidth = 64;
  }

  if (MrcData->Channel[Channel].D_SizeActual[0] != 0) {
    DramDensityInGb = MrcData->Channel[Channel].D_SizeActual[0];
  } else {
    DramDensityInGb = DramDensity[DramCapacity];
  }

  RankSize = (DramDensityInGb * 1024) / 8 *
        (DQBusWidth) /
        (8 << DeviceWidth );

  //MmrcDebugPrint((MMRC_DBG_MIN, "RankSize = %d, DQBusWidth = %d, DeviceWidth = %d, DramDensity[%d] = %d | ", RankSize, DQBusWidth, DeviceWidth, DramCapacity, DramDensity[DramCapacity]));
  return RankSize;
}

/**
  Sets the DRAM density fields and automatically applies software workarounds
  for hardware unsupported densities.

  @param[in,out]  MrcData       Host structure for all data related to MMRC
  @param[in]      Channel       The channel to preform the update on
  @param[in]      DramDensity   The requested density for the DRAM
  @param[in]      DramWidth     The reported width of the DRAM
  @param[in]      RankEnable    A bit mask indicating what ranks are enabled

  @retval  Success
  @retval  Failure
**/
MMRC_STATUS
SetDramDensity (
  IN  OUT   MMRC_DATA                *MrcData,
  IN        UINT8                    Channel,
  IN        SMIP_DRAM_DEVICE_DENSITY DramDensity,
  IN        SMIP_DRAM_DEVICE_WIDTH   DramWidth,
  IN        UINT8                    RankEnable
  )
{
  if ( MrcData->Channel[Channel].DramType == TypeLpDdr3 &&
       DramDensity == SMIP_DramDensity8Gb &&
       DramWidth == SMIP_DramWidthx16 &&
       RankEnable == (BIT0|BIT1) &&
       MrcData->DramPolicyData.Package == BGA) {
    //
    // Special case for 16Gb QDP (4 x 4Gb x16) Devices
    // Treat these devices as 32Gb DRAMs with 16Gb sized ranks
    //
    MrcData->Channel[Channel].D_Size[0] = DeviceDensity16Gb;
    MrcData->Channel[Channel].D_SizeActual[0] = DRAM_DENSITY_8Gb;
    return MMRC_SUCCESS;
  }

  //
  // Supported DrpPolicy.DramDensity inputs:
  // 0 = 4Gb
  // 1 = 6Gb
  // 2 = 8Gb
  // 3 = 12Gb
  // 4 = 16Gb
  // 5 = 2Gb
  //
  if (DramDensity == SMIP_DramDensity2Gb) {
    MrcData->Channel[Channel].D_Size[0] = DeviceDensity4Gb;
    MrcData->Channel[Channel].D_SizeActual[0] = DRAM_DENSITY_2Gb;
  }
  else if (DramDensity == SMIP_DramDensity6Gb) {
    MrcData->Channel[Channel].D_Size[0] = DeviceDensity8Gb;
    MrcData->Channel[Channel].D_SizeActual[0] = DRAM_DENSITY_6Gb;
  }
  else {
    MrcData->Channel[Channel].D_Size[0] = (DramDensity & 0x7);
  }
  return MMRC_SUCCESS;
}

/**
  Fills in the complete parameter list for the Modular MRC.
  This includes all the Dynamic entries that are listed in
  the Input Spreadsheet, but also the PFCT values.
  Want to

  @param[in,out]  MrcData       Host structure for all data related to MMRC

  @retval  Success
  @retval  Failure
**/
MMRC_STATUS
  GatherSocConfigs (
  IN  OUT   MMRC_DATA            *MrcData
  )
{
  UINT8               Channel;
  DRP_DRAM_POLICY     DrpPolicy;
  DRAM_POLICY         DramPolicyData;
  UINT8               i;
  UINT8               Tcl;

  MmrcMemset (&DrpPolicy, 0, sizeof (DRP_DRAM_POLICY));
  MmrcMemset (&DramPolicyData, 0, sizeof (DRAM_POLICY));

  //
  // Get SMIP setting and convert it to MRC Data
  //
  if (MrcData->DramPolicyData.MemoryDown != MD_SODIMM) {
    DramPolicyData.Profile  = MrcData->DramPolicyData.Profile;  // Add after smip available
    DramPolicyData.ChDrp[0] = MrcData->DramPolicyData.ChDrp[0]; 
    DramPolicyData.ChDrp[1] = MrcData->DramPolicyData.ChDrp[1]; 
    DramPolicyData.ChDrp[2] = MrcData->DramPolicyData.ChDrp[2];      
    DramPolicyData.ChDrp[3] = MrcData->DramPolicyData.ChDrp[3];  
  } else {
    // Set DDR3L profile for SODIMM

    Tcl = (MrcData->Channel[0].Enabled) ? MrcData->Channel[0].Tcl : MrcData->Channel[1].Tcl;

    switch (MrcData->CurrentFrequency) {
    case MMRC_1600:
      DramPolicyData.Profile = DDR3_1600_8_8_8;
      i = 9;
      break;
    case MMRC_1866:
      DramPolicyData.Profile = DDR3_1866_10_10_10;
      i = 11;
      break;
    case MMRC_1333:
    default:
      DramPolicyData.Profile = DDR3_1333_7_7_7;
      i = 8;
      break;
    }

    while (i++ <= Tcl)  {
      DramPolicyData.Profile += 1;
    }
    MmrcDebugPrint ((MMRC_DBG_MIN, "profile is %d freq is %d\n", DramPolicyData.Profile, MrcData->CurrentFrequency ));
  }
  MrcData->TimingParamsConfig = DramPolicyData.Profile;
  MrcData->TimingParamsConfig = DramPolicyData.Profile;
#ifdef BXT_PERIODIC_ENABLED
    MrcData->DynamicVars[0][BXT_PERIODIC_ENABLED]=1;
    MrcData->DynamicVars[1][BXT_PERIODIC_ENABLED]=1;
    MrcData->DynamicVars[2][BXT_PERIODIC_ENABLED]=1;
    MrcData->DynamicVars[3][BXT_PERIODIC_ENABLED]=1;
#endif

  for (Channel = 0; Channel < MAX_CHANNELS; Channel++) {
    if (MrcData->DramPolicyData.MemoryDown != MD_SODIMM) {
      Mmrcmemcpy(&DrpPolicy, &(DramPolicyData.ChDrp[Channel]), sizeof(DRP_DRAM_POLICY));

      if (DrpPolicy.RankEnable != 0) {
        MrcData->Channel[Channel].Enabled = TRUE;
        MrcData->Channel[Channel].DimmPresent[0] = 1;
        MrcData->Channel[Channel].RankEnabled[0] = (BOOLEAN) DrpPolicy.RankEnable & 0x1;
        MrcData->Channel[Channel].RankEnabled[1] = (BOOLEAN)((DrpPolicy.RankEnable >> 1) & 0x1);
        MrcData->Channel[Channel].D_BusWidth[0] = 1;
        MrcData->Channel[Channel].D_DataWidth[0] = (UINT8) (DrpPolicy.DeviceWidth & 0x3);// 1;
        MrcData->Channel[Channel].DramType = (UINT8) ConfigToDdrType[MrcData->MrcMemConfig];
        MrcData->Channel[Channel].ASR_Supported = MrcData->DramPolicyData.DDR3LASR;
        SetDramDensity(
          MrcData,
          Channel,
          DrpPolicy.DramDensity,
          DrpPolicy.DeviceWidth,
          DrpPolicy.RankEnable
          );
      }
    }

    if ((MrcData->Channel[Channel].RankEnabled[0] || MrcData->Channel[Channel].RankEnabled[1]) == 0){
      MrcData->Channel[Channel].Enabled = FALSE;
    }

    if (MrcData->Channel[Channel].Enabled) {
      if (MrcData->Channel[Channel].RankEnabled[0] && MrcData->Channel[Channel].RankEnabled[1]) {
        MrcData->Channel[Channel].D_Ranks[0] = 2;
        MmrcDebugPrint((MMRC_DBG_MIN, "Ch[%d] 2 ranks\n", Channel));
      } else {
        MrcData->Channel[Channel].D_Ranks[0] = 1;
        MmrcDebugPrint((MMRC_DBG_MIN, "Ch[%d] 1 rank\n", Channel));
      }
    }
  }

  MrcData->DigitalDll = DIGITALDLL;

  // Enable 2nd channel to configure PHY for 1x32 lpddr4 config
  if (ConfigToDdrType[TimingParamHash(MrcData->DramPolicyData.Profile - 1, TRUE)] == TypeLpDdr4) {
    if (MrcData->DramPolicyData.MemoryDown == MD_1X32_LPDDR4) {
      MrcData->Channel[1].Enabled = TRUE;
    }
  }

  MrcData->SystemMemorySize = 0;
  for (Channel = 0; Channel < MAX_CHANNELS; Channel++) {
#ifdef SLICE_0_MASTER
    if (MrcData->Channel[0].Enabled) {
      MrcData->DynamicVars[Channel][SLICE_0_MASTER] = 1;
      MrcData->DynamicVars[Channel][SLICE_1_MASTER] = 0;
    } else {
      MrcData->DynamicVars[Channel][SLICE_0_MASTER] = 0;
      MrcData->DynamicVars[Channel][SLICE_1_MASTER] = 1;
    }
#endif

    MrcData->Channel[Channel].TotalMem = 0;

    if (!MrcData->Channel[Channel].Enabled) {
      continue;
    }


    //
    // Look at both DIMMs and fill in DIMM-specific variables. Also add up total memory
    // per channel and system-wide.
    //
    if (MrcData->Channel[Channel].RankEnabled[0]) {

      MrcData->Channel[Channel].SlotMem[0] = CalculateRankSize(MrcData, Channel) * MrcData->Channel[Channel].D_Ranks[0];

      MrcData->Channel[Channel].TotalMem  += MrcData->Channel[Channel].SlotMem[0];
      MrcData->SystemMemorySize           += MrcData->Channel[Channel].SlotMem[0];

      MmrcDebugPrint((MMRC_DBG_MIN, "DIMM%d Memory Size: %5d, System Mem %5d in MB\n", Channel, MrcData->Channel[Channel].SlotMem[0], MrcData->SystemMemorySize));
      if (MrcData->Channel[Channel].DramType == TypeLpDdr3) {
        MrcData->NumberOfEnabledChannels[TypeLpDdr3] += 1;
      }
      if (MrcData->Channel[Channel].DramType == TypeDdr3L) {
        MrcData->NumberOfEnabledChannels[TypeDdr3L] += 1;
      }
      if (MrcData->Channel[Channel].DramType == TypeLpDdr4) {
        MrcData->NumberOfEnabledChannels[TypeLpDdr4] += 1;
      }
    } //rankEnabled

    switch (MrcData->Channel[Channel].DramType) {
    case TypeLpDdr3:
      MrcData->Channel[Channel].CurrentDdrType = T_LPDDR3;// >> 1;
      MrcData->CurrentBlueprint = 0;   //Lp3 : 0; LP4: 1
      MrcData->Channel[Channel].MaxDq[0] = MAX_STROBES_LPDDR3;
      MrcData->Channel[Channel].MaxDq[1] = MAX_STROBES_LPDDR3;
      MrcData->Channel[Channel].EccEnabled = 0;
      break;
    case TypeLpDdr4:
      MrcData->Channel[Channel].CurrentDdrType = T_LPDDR4;// >> 1;
      if (MrcData->CpuStepping == STEPPING_A0) {
        MrcData->CurrentBlueprint = 1;   //Lp3 : 0; LP4: 1
        MrcData->Channel[Channel].MaxDq[0] = MAX_STROBES_LPDDR4_COPOP;
        MrcData->Channel[Channel].MaxDq[1] = MAX_STROBES_LPDDR4_COPOP;
      }
      if (MrcData->BxtSeries == SERIES_BXT_P) {
        MrcData->CurrentBlueprint = 0;   //Lp3 : 0; LP4: 1
        MrcData->Channel[Channel].MaxDq[0] = MAX_STROBES_LPDDR4_BGA;
        MrcData->Channel[Channel].MaxDq[1] = MAX_STROBES_LPDDR4_BGA;
        MrcData->Channel[Channel].EccEnabled = 0;
      }
      break;
    case TypeDdr3L:
      MrcData->Channel[Channel].CurrentDdrType = T_DDR3L;// >> 1;
      MrcData->CurrentBlueprint = 1;    //Lp3 : 0; LP4: 1
      MrcData->Channel[Channel].MaxDq[0] = MAX_STROBES_DDR3L;
      MrcData->Channel[Channel].MaxDq[1] = MAX_STROBES_DDR3L;
      if (MrcData->DramPolicyData.MemoryDown == 5) {
        MrcData->Channel[Channel].EccEnabled = 1;
      }
      break;
    default:
      MrcData->Channel[Channel].CurrentDdrType = T_LPDDR3;// >> 1;
      break;
    }
    MrcData->MaxElements = MrcData->Channel[Channel].MaxDq[0];
    MrcData->MaxBits =  MAX_BITS;  //assign to maxDq initially
  }

#ifdef SLICE_0_MASTER
  MmrcDebugPrint((MMRC_DBG_MIN, "Ch0 SLICE_0_MASTER: %d SLICE_1_MASTER: %d\n", MrcData->DynamicVars[0][SLICE_0_MASTER], MrcData->DynamicVars[0][SLICE_1_MASTER]));
#else
  MmrcDebugPrint((MMRC_DBG_MIN, "Vars for master slice not used\n"));
#endif
  return MMRC_SUCCESS;
}

/**
  PrintMemoryConfig

  @param[in]  MrcData

  @retval  None
**/
void
  PrintMemoryConfig(
  IN  MMRC_DATA   *MrcData
  )
{
  UINT8  Channel;

  MmrcDebugPrint ((MMRC_DBG_MIN, "BootMode set to: %x (FB=0x10; S5=8; S0=1,2,4; S3=0x20)\n",MrcData->BootMode));
  MmrcDebugPrint ((MMRC_DBG_MIN, "CurrentFreq = %d     (1333 = %d, 1600 = %d, 1866  =%d, 2133 = %d, 2400 = %d, 2666 = %d, 3200 = %d)           \n", MrcData->CurrentFrequency, MMRC_1333, MMRC_1600, MMRC_1866, MMRC_2133, MMRC_2400, MMRC_2666, MMRC_3200));
  MmrcDebugPrint ((MMRC_DBG_MIN, "SystemMemSize = %x\n", MrcData->SystemMemorySize));

  for (Channel = 0; Channel < MAX_CHANNELS; Channel++) {
    if (MrcData->Channel[Channel].Enabled) {
        MmrcDebugPrint ((MMRC_DBG_MIN, "CH %d Enabled ", Channel));
      if (MrcData->Channel[Channel].DramType == TypeLpDdr3) {
        MmrcDebugPrint ((MMRC_DBG_MIN, "LPDDR3\n"));
      } else if (MrcData->Channel[Channel].DramType == TypeDdr3L) {
        MmrcDebugPrint ((MMRC_DBG_MIN, "DDR3L\n"));
      } else if (MrcData->Channel[Channel].DramType == TypeLpDdr4) {
        MmrcDebugPrint ((MMRC_DBG_MIN, "LPDDR4\n"));
      } else {
        MmrcDebugPrint ((MMRC_DBG_MIN, "Unknown mem type\n"));
      }
    }
  }

}

/**
  This routine is responsible to initialize ALL data required for
  MMRC to initialize memory. Typical flow for server would be:

  1) DIMM detection
  2) Timing calculations
  3) Variable initialization
  4) BIOS Setup overrides

  Typical phone/tablet flow would be:

  1) Timing variable population via lookup table
  2) BIOS Setup overrides

  This routine can be very complex in server segments or extremely
  lightweight in phone/tablet segments, depending on code size and
  MRC execution time requirements.

  @param[in,out]  MrcData  Host structure for all MRC global data.

  @retval  Success
  @retval  Failure
**/
MMRC_STATUS
GetInputParameters (
  IN  OUT   MMRC_DATA   *MrcData
)
{
#ifdef FSP_FLAG
  FSPM_UPD                        *FspmUpd;

  FspmUpd = (FSPM_UPD *)GetFspMemoryInitUpdDataPointer ();
#endif
  OemGetPlatformType (MrcData);
  //
  // Sets the OEM specific information
  //
  OemSetMrcData (MrcData);

  if (MrcData->BootMode != S3) {
    DetermineBootMode (MrcData);
  } else {
    if (MrcData->MrcParamsValidFlag == 0) {   //if S3 and restored data not valid.
      MmrcDebugPrint ((MMRC_DBG_MIN, "Force powergood reset. Vars not valid in S3 resume\n"));
#ifdef FSP_FLAG
      if (!(FspmUpd->FspmConfig.EnableResetSystem)) {
        FspApiReturnStatusReset (FSP_STATUS_RESET_REQUIRED_COLD);
      }
#endif
      IoOut8 (0xCF9, 0x0);
      IoOut8 (0xCF9, 0xE);
#if !defined SUSSW
#if defined __GNUC__  // GCC compiler
  __asm__ __volatile__ (
      "hlt"
  );
#else // MSFT compiler
      _asm hlt
#endif
#endif
      return MMRC_FAILURE;
    }
  }
  ///
  /// Select to use the restore path here we decide here after DetermineBootMode
  /// and DetectDimms, this may change for memory down we decide here in case a define
  /// prevents DetermineBootMode or DetectDimms functions to execute
  ///
//  if ((MrcData->BootMode != S5) && (MrcData->CpuStepping != STEPPING_A0)) {
  MmrcDebugPrint ((MMRC_DBG_MIN, "CurrentBootMode -> %d ValidFlag -> %d\n", MrcData->BootMode, MrcData->MrcParamsValidFlag));  //KW REMOVE
  if ((MrcData->BootMode != S5) && (MrcData->MrcParamsValidFlag == 1)) {
    MrcData->RestorePath = TRUE;
    MmrcDebugPrint ((MMRC_DBG_MIN, "Restore Path\n"));
  } else {
    MrcData->RestorePath = FALSE;
    MmrcDebugPrint ((MMRC_DBG_MIN, "NOT Restore Path\n"));
  }

  if (MrcData->PlatformID != VALUE_REAL_PLATFORM) {
    MmrcDebugPrint ((MMRC_DBG_MIN, "Skipping training. BootMode=CSE\n"));
    MrcData->BootMode = CSE;
    MrcData->RestorePath = FALSE;
  } /*else if (MrcData->PlatformID != 0) {
    MrcData->BootMode = EM;
    MrcData->RestorePath = FALSE;
    MmrcDebugPrint ((MMRC_DBG_MIN, "EM:Overriding Restore Path -> FALSE\n"));
  }*/

#if (MRC_PMIXOR_ENABLE == 1)
  MrcData->BootMode = CSE;
  MrcData->RestorePath = FALSE;
  MmrcDebugPrint ((MMRC_DBG_MIN, "CSE:Overriding via MRC_PMIXOR_ENABLE BuildFlag\n"));
#endif

#if DDR3_SUPPORT || DDR4_SUPPORT
  if ((MrcData->BootMode == S5) && (MrcData->DramPolicyData.MemoryDown != MD_MEMORYDOWN)) {
    DetectDimms (MrcData);
  }
#endif

  GatherSocConfigs (MrcData);

  DetermineCommonFreq (MrcData);

  if ((MrcData->DramPolicyData.MemoryDown == MD_CH0_MD_CH1_SOD) || (MrcData->DramPolicyData.MemoryDown == MD_CH0_SOD_CH1_MD)) {
    if (MrcData->DDR3L_PageSize < MrcData->DramPolicyData.DDR3LPageSize) {
      MrcData->DDR3L_PageSize = MrcData->DramPolicyData.DDR3LPageSize;
    }
  }

  //
  // Save DIMM information for fast boot path comparison
  //
#if DDR3_SUPPORT || DDR4_SUPPORT
  if ((ConfigToDdrType[TimingParamHash(MrcData->DramPolicyData.Profile - 1, TRUE)] == TypeDdr3L) && (MrcData->DramPolicyData.MemoryDown != MD_MEMORYDOWN)) {
    FillInputStructure (MrcData);
  }
#endif

  return MMRC_SUCCESS;
}

/**
  This routine assumes all input parameters to the MMRC have been
  populated by GetInputParameters(). Converting all timing variables
  to the correct values for DUNIT programming is the responsibility
  of this function.

  @param[in,out]  MrcData  Host structure for all MRC global data.

  @retval  Success
  @retval  Failure
**/
MMRC_STATUS
ConvertInputParameters (
  IN  OUT   MMRC_DATA   *MrcData
)
{



  return MMRC_SUCCESS;
}

/**
  ProgramMemoryMap

  @param[in]  MrcData  Host structure for all MRC global data.
  @param[in]  CapsuleStartIndex
  @param[in]  StringIndex
  @param[in]  Channel

  @retval  Success
  @retval  Failure
**/
MMRC_STATUS
ProgramMemoryMap (
  MMRC_DATA         *MrcData,
  UINT16            CapsuleStartIndex,
  UINT16            StringIndex,
  UINT8             Channel
)
{
  UINT32        MmioAllocation;
  UINT32        TOM;
  UINT32        LowMemory;
  BOOLEAN       ReclaimEn;
  UINT32        HighMemory;
  UINT16        TSegSize;
  UINT32        TSegBase;
  UINT16        GMSsize;
  UINT16        GTTsize;
  UINT32        GMSbase;
  UINT32        GTTbase;
  UINT32        SeCUmaBase;
  UINT32        TempTSEGbase;
//  UINT32    SeCfTPMUmaBase;
  REGISTER_ACCESS               Register;
  GGC_0_0_0_PCI_CUNIT_STRUCT    Cuint_GGC;
  UINT32                        Buffer32;

  SeCUmaBase = 0;
#ifdef FTPM_ENABLE
  UINT32  SeCfTPMUmaBase;
#endif

#ifdef ISH_ENABLE
  UINT32  IshBase;
#endif

  MrcData->OemMrcData.TsegSize = 0x8;
  MrcData->OemMrcData.MmioSize = 0x400;


  GMSsize = 0;
  GTTsize = 0;
  TSegSize = 0;
  Buffer32 = 0;

  Register.Mask = 0xFFFFFFFF;
  Register.ShiftBit = 0;

  Register.Offset = GGC_0_0_0_PCI_CUNIT_REG;
  Cuint_GGC.Data = MemRegRead (CUNIT, 0, 0, Register);


  //All unit in 1MB granularity
  if(Cuint_GGC.Bits.gms < 0x40){
    GMSsize = (UINT16) Cuint_GGC.Bits.gms * 32;
  } else {
    MmrcDebugPrint ((MMRC_DBG_MIN, "Warning GMSsize is zero\n"));
    //return MMRC_FAILURE;
  }

  if(Cuint_GGC.Bits.ggms > 0x0){
    GTTsize = 1 << Cuint_GGC.Bits.ggms;
  }


  if ((MrcData->DramPolicyData.SystemMemorySizeLimit != 0) &&
      (MrcData->DramPolicyData.SystemMemorySizeLimit < MrcData->SystemMemorySize)) {
        MrcData->SystemMemorySize = MrcData->DramPolicyData.SystemMemorySizeLimit;
  }

  TSegSize       = MrcData->OemMrcData.TsegSize;
  MmioAllocation = MrcData->OemMrcData.MmioSize;
  TOM            = MrcData->SystemMemorySize;

  // LowMemory not = 0x800 could reintroduce x64 bit addressing issues in some of the IAFW drivers
  //LowMemory       = MIN (0x1000 - MmioAllocation, TOM);//LowMemory       = 0x800;  //LowMemory       = MIN (0x1000 - MmioAllocation, TOM);
  LowMemory       = 0x800;

  if ((MrcData->DramPolicyData.LowMemMaxVal != 0) &&
      (MrcData->DramPolicyData.LowMemMaxVal < LowMemory)) {
    LowMemory = MrcData->DramPolicyData.LowMemMaxVal;
  }

  MmrcDebugPrint ((MMRC_DBG_MIN, "LowMemory:      0x%08x\n", LowMemory));

  ReclaimEn = FALSE;

  if (TOM > LowMemory) {
    ReclaimEn = TRUE;
  }

  if (ReclaimEn) {
    HighMemory = TOM - LowMemory + 0x1000;
  } else {
    HighMemory = TOM;
  }

  if ((MrcData->DramPolicyData.HighMemMaxVal != 0) &&
      (MrcData->DramPolicyData.HighMemMaxVal < HighMemory)) {
    HighMemory = MrcData->DramPolicyData.HighMemMaxVal;
  }
  MmrcDebugPrint ((MMRC_DBG_MIN, "HighMemory:     0x%08x\n", HighMemory));

#ifdef ISH_ENABLE
  //
  // 1MB alloc for ISH
  //
#ifdef FTPM_ENABLE
  IshBase = SeCfTPMUmaBase - CurrentMrcData->IshUmaSize;
#else
  IshBase = SeCUmaBase - CurrentMrcData->IshUmaSize;
#endif
  //
  //9. Determine GFX Memory base, which is calculated by the value of TOLUD minus GFX size.
  //
  GMSbase = IshBase - GMSsize;
#else
  //
  //9. Determine GFX Memory base, which is calculated by the value of TOLUD minus GFX size.
  //
  GMSbase = LowMemory - GMSsize;
#endif

  GTTbase = GMSbase - GTTsize;
  MmrcDebugPrint ((MMRC_DBG_MIN, "SeCUmaBase 0x%x GMSsize 0x%x GMSbase 0x%x GTTbase 0x%x\n", SeCUmaBase,GMSsize,GMSbase,GTTbase));

  ///
  ///11. Determine TSEG base, which is calculated by the value of TOLUD minus GFX size minus TSEG size.
  ///
  TempTSEGbase = GTTbase - (GTTbase & (TSegSize - 1));
  TSegBase = TempTSEGbase - TSegSize;

///////////////////////////////////////////////////////
///Program the registers
///////////////////////////////////////////////////////


  MmrcDebugPrint ((MMRC_DBG_MIN, "TSegSize:       0x%08x\n", TSegSize));
  MmrcDebugPrint ((MMRC_DBG_MIN, "TSegBase:       0x%08x\n", TSegBase));
  MmrcDebugPrint ((MMRC_DBG_MIN, "MmioAllocation: 0x%08x\n", MmioAllocation));
  MmrcDebugPrint ((MMRC_DBG_MIN, "TOM:            0x%08x\n", TOM));
  MmrcDebugPrint ((MMRC_DBG_MIN, "TOLUD:          0x%08x\n", LowMemory << 20));
  MmrcDebugPrint ((MMRC_DBG_MIN, "TOUUD_LO:       0x%08x\n", HighMemory << 20));
  MmrcDebugPrint ((MMRC_DBG_MIN, "TOUUD_HI:       0x%08x\n", HighMemory >> 12));

  Register.Mask = 0xFFFFFFFF;
  Register.ShiftBit = 0;

  Register.Offset = TOLUD_0_0_0_PCI_CUNIT_REG;
  MemRegWrite (CUNIT, 0, 0, Register, (LowMemory << 20) + BIT0, 0xF);
  Register.Offset = TOUUD_LO_0_0_0_PCI_CUNIT_REG;
  MemRegWrite (CUNIT, 0, 0, Register, (HighMemory << 20) & 0xFFFFFFFF, 0xF);
  Register.Offset = TOUUD_HI_0_0_0_PCI_CUNIT_REG;
  MemRegWrite (CUNIT, 0, 0, Register, (HighMemory >> 12), 0xF);
  Register.Offset = BDSM_0_0_0_PCI_CUNIT_REG;
  MemRegWrite (CUNIT, 0, 0, Register, (GMSbase << 20) + BIT0, 0xF);
  Register.Offset = BGSM_0_0_0_PCI_CUNIT_REG;
  MemRegWrite (CUNIT, 0, 0, Register, (GTTbase << 20) + BIT0, 0xF);
  Register.Offset = TSEGMB_0_0_0_PCI_CUNIT_REG;
  MemRegWrite (CUNIT, 0, 0, Register, (TSegBase << 20) + BIT0, 0xF);

  return MMRC_SUCCESS;
}

