/** @file
  Instance of Intel Processor Trace memory allocation library.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 2017 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:

**/

#include <Library/DebugLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/ConfigBlockLib.h>
#include <Library/CpuPolicyLib.h>
#include <Library/ProcessorTraceMemoryAllocationLib.h>
//#include <Ppi/SiPolicy.h>
#include <CpuRegs.h>

/**
  Allocate memory region for Processor Trace, based on requested size per thread.
  This function must be called after initializing the following policies in CPU policy, as this function consumes them.
    ActiveProcessorCores
    DisableCore1
    DisableCore2
    DisableCore3

  @param[in]  RequestedMemSize          Requested size per thread, specified using PROCESSOR_TRACE_MEM_SIZE encoding
  @param[out] *BaseAddress              Outputs a pointer to the base address of the allocated memory region. Base address is NULL on a failure
  @param[out] *LengthInBytes            Outputs a pointer to the size of the allocated memory region, in bytes

  @retval EFI_SUCCESS                   Successfully allocated the memory region
  @retval EFI_INVALID_PARAMETER         Invalid value for RequestedMemSize
  @retval EFI_OUT_OF_RESOURCES          AllocatePages failed to allocate the memory region
**/
EFI_STATUS
ProcessorTraceAllocateMemory (
  IN OUT SI_CPU_POLICY_PPI   *SiCpuPolicyPpi,
  IN PROCESSOR_TRACE_MEM_SIZE RequestedMemSize,
  OUT EFI_PHYSICAL_ADDRESS    *BaseAddress,
  OUT UINT32                  *LengthInBytes
  )
{
  EFI_STATUS                   Status;
  UINT32                       MemRegionSize;
  UINTN                        Pages;
  //SI_PREMEM_POLICY_PPI         *SiPreMemPolicyPpi;
//  CPU_CONFIG_LIB_PREMEM_CONFIG *CpuConfigLibPreMemConfig;
  UINT64                       MsrValue;
  UINT16                       MaxCores;
  UINT16                       MaxThreads;
  UINT16                       NumThreads;
  //SI_CPU_POLICY_PPI                   *SiCpuPolicyPpi;
  //CPU_SECURITY_PREMEM_CONFIG          *CpuSecurityPreMemConfig;
  CPU_CONFIG                       *CpuConfig;

  *BaseAddress = (EFI_PHYSICAL_ADDRESS) (UINTN) NULL;
  *LengthInBytes = 0;
   
  if (RequestedMemSize >= EnumProcessorTraceMemDisable) {
    DEBUG ((DEBUG_INFO, "ProcessorTraceAllocateMemory: No memory requested %x, %x.\n", RequestedMemSize,EnumProcessorTraceMemDisable));
    return EFI_INVALID_PARAMETER;
  }

 /* Status = PeiServicesLocatePpi(
	  &gSiCpuPolicyPpiGuid,
	  0,
	  NULL,
	  (VOID **)&SiCpuPolicyPpi
  );
  ASSERT_EFI_ERROR(Status);*/

  //if (!EFI_ERROR(Status)) {
//	  Status = GetConfigBlock((VOID *)SiCpuPolicyPpi, &gCpuSecurityPreMemConfigGuid, (VOID *)&CpuSecurityPreMemConfig);
//	  ASSERT_EFI_ERROR(Status);
//  }

  Status = GetConfigBlock((CONFIG_BLOCK_TABLE_HEADER*)SiCpuPolicyPpi, &gCpuConfigGuid, (VOID *)&CpuConfig);
  ASSERT_EFI_ERROR(Status);

  DEBUG((DEBUG_INFO, " Get config block for CpuConfig \n"));


  ///
  /// Get the total number of enabled processor threads.
  /// Read MSR 0x35 to get the max enabled number of cores/threads.
  ///
  MsrValue = AsmReadMsr64 (MSR_CORE_THREAD_COUNT);
  MaxCores = (UINT16) RShiftU64 (MsrValue, N_CORE_COUNT_OFFSET);
  MaxThreads = (UINT16) (MsrValue & B_THREAD_COUNT_MASK);

  DEBUG((DEBUG_INFO, "MaxCores: %x, MaxThread: %x \n", MaxCores, MaxThreads));
 
  ///
  /// Check ActiveCoreCount and HyperThreading policy to determine the number of active threads.
  ///
  if (CpuConfig->ActiveProcessorCores == 0) {
	  NumThreads = MaxCores;
    
  } else {        
	if (MaxCores == 4)
		NumThreads =  1 + (UINT16)(UINTN)CpuConfig->DisableCore1 + (UINT16)(UINTN)CpuConfig->DisableCore2 + (UINT16)(UINTN)CpuConfig->DisableCore3;
	else
		NumThreads =  1 + (UINT16)(UINTN)CpuConfig->DisableCore1;
  } 

  ///
  /// Refer to PROCESSOR_TRACE_MEM_SIZE Table for Size Encoding
  ///
  MemRegionSize = (UINT32) (1 << (RequestedMemSize + 12));
  DEBUG ((DEBUG_INFO, "ProcessorTraceAllocateMemory: MemSize requested per thread: 0x%X \n", MemRegionSize));

  ///
  /// Allocate enough pages for all threads.
  /// All Processor Trace regions must be aligned based on their size.
  ///
  Pages = EFI_SIZE_TO_PAGES (MemRegionSize) * NumThreads;
  *BaseAddress = (EFI_PHYSICAL_ADDRESS) (UINTN) AllocateAlignedReservedPages (Pages, MemRegionSize);
  if (*BaseAddress == 0) {
    DEBUG ((DEBUG_ERROR, "ProcessorTraceAllocateMemory: Out of memory, allocate pages failed\n"));
    return EFI_OUT_OF_RESOURCES;
  }

  *LengthInBytes = EFI_PAGES_TO_SIZE (Pages);

  DEBUG ((DEBUG_INFO, "ProcessorTraceAllocateMemory: MemBase: 0x%X \n", *BaseAddress));
  DEBUG ((DEBUG_INFO, "ProcessorTraceAllocateMemory: MemLength: 0x%X \n", *LengthInBytes));
  return EFI_SUCCESS;
}
